﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using EffectMaker.BusinessLogic.IO;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// ファイルを開くダイアログで選択されたファイルパスを管理します。
    /// </summary>
    public class SelectedFilePathList
    {
        /// <summary>
        /// ワークスペースファイルのパスリストです。
        /// </summary>
        private readonly List<string> workspaceList = new List<string>();

        /// <summary>
        ///  エミッタセットファイルのパスリストです。
        /// </summary>
        private readonly List<string> emitterSetList = new List<string>();

        /// <summary>
        /// プレビューファイルのパスリストです。
        /// </summary>
        private readonly List<string> previewList = new List<string>();

        /// <summary>
        /// 無効なファイルのパスリストです。
        /// </summary>
        private readonly List<string> invalidList = new List<string>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="filePaths">ファイルパスリスト</param>
        public SelectedFilePathList(string[] filePaths)
        {
            // ファイルパス/フォルダパスに対してそれぞれ振り分け処理を呼び出す
            foreach (string path in filePaths)
            {
                if (Directory.Exists(path))
                {
                    // フォルダパスに対して振り分け処理を行う
                    this.ClassfyDirectory(path);
                }
                else
                {
                    // ファイルパスに対して振り分け処理を行う
                    this.ClassfyFile(path);
                }
            }
        }

        /// <summary>
        /// ワークスペースファイルリストを設定または取得します。
        /// </summary>
        public List<string> WorkspaceList
        {
            get
            {
                return this.workspaceList;
            }
        }

        /// <summary>
        /// エミッタセットファイルリストを設定または取得します。
        /// </summary>
        public List<string> EmitterSetList
        {
            get
            {
                return this.emitterSetList;
            }
        }

        /// <summary>
        /// プレビューファイルリストを取得します。
        /// </summary>
        public List<string> PreviewFileList
        {
            get
            {
                return this.previewList;
            }
        }

        /// <summary>
        /// 無効なファイルリストを取得します。
        /// </summary>
        public List<string> InvalidFileList
        {
            get
            {
                return this.invalidList;
            }
        }

        /// <summary>
        /// フォルダパスに対して振り分け処理を行います。
        /// </summary>
        /// <param name="directoryPath">フォルダパス</param>
        private void ClassfyDirectory(string directoryPath)
        {
            // フォルダ内のフォルダに対して振り分け処理を行う
            foreach (string path in Directory.GetDirectories(directoryPath))
            {
                this.ClassfyDirectory(path);
            }

            // フォルダ内のファイルに対して振り分け処理を行う
            foreach (string path in Directory.GetFiles(directoryPath))
            {
                this.ClassfyFile(path);
            }
        }

        /// <summary>
        /// ファイルパスに対して振り分け処理を行います。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        private void ClassfyFile(string filePath)
        {
            // 拡張子を取得
            string extension = Path.GetExtension(filePath).ToLower();

            // 拡張子を判別してリストに振り分ける
            if (extension == IOConstants.EmitterSetFileExtension)
            {
                // エミッタセットファイルを登録
                this.emitterSetList.Add(filePath);
            }
            else if (extension == IOConstants.PreviewFileExtension)
            {
                // プレビューファイルを登録
                this.previewList.Add(filePath);
            }
            else if (extension == IOConstants.WorkspaceFileExtension)
            {
                // ワークスペースファイルを登録
                this.workspaceList.Add(filePath);
            }
            else
            {
                // 無効なファイルを登録
                this.invalidList.Add(filePath);
            }
        }
    }
}
