﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Drawing;
using System.Linq;
using System.Windows.Resources;
using EffectMaker.BusinessLogic;
using EffectMaker.BusinessLogic.Serializer;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the reserved shader node.
    /// </summary>
    [Tag("WorkspaceNodeViewModelKind")]
    public class ReservedShaderNodeViewModel : WorkspaceNodeViewModelBase<ReservedShaderNodeData>, IModificationFlagOwner
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public ReservedShaderNodeViewModel(
            HierarchyViewModel parent,
            ReservedShaderNodeData dataModel)
            : base(parent, dataModel)
        {
            if (dataModel.PageData == null ||
                dataModel.PageData.ContentsData == null)
            {
                throw new ArgumentException("The user page or the user data model is null.");
            }

            // Create the view model for the user page.
            var userPageVM = this.AddChildViewModel(
                new UserPageViewModel(this, dataModel.PageData, false));

            int index = ReservedShaderUserDataManager.FindReservedShaderIndexByUserData(
                dataModel.PageData.ContentsData);

            // Create the view model for the reserved shader user data and add it to the user page.
            userPageVM.Contents = userPageVM.AddChildViewModel(new ReservedShaderUserDataViewModel(
                userPageVM,
                dataModel.PageData.ContentsData,
                index));

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
        }

        /// <summary>
        /// The display name of this node on the workspace.
        /// </summary>
        public string Name
        {
            get
            {
                if (this.DataModel == null)
                {
                    return string.Empty;
                }

                ReservedShaderDefinition def =
                    ReservedShaderUserDataManager.FindReservedShaderDefinition(this.DataModel.PageData.ContentsData);
                if (def == null)
                {
                    return string.Empty;
                }

                var culture = System.Threading.Thread.CurrentThread.CurrentUICulture;
                var cultureEnUs = System.Globalization.CultureInfo.CreateSpecificCulture("en-US");
                if (culture.Equals(cultureEnUs) == true && string.IsNullOrEmpty(def.NameEn) == false)
                {
                    return def.NameEn;
                }
                else
                {
                    return def.Name;
                }
            }
        }

        /// <summary>
        /// The icon for this node to show on the workspace.
        /// </summary>
        public Image Icon
        {
            get
            {
                if (this.DataModel == null)
                {
                    return null;
                }

                ReservedShaderDefinition def =
                    ReservedShaderUserDataManager.FindReservedShaderDefinition(this.DataModel.PageData.ContentsData);
                if (def == null)
                {
                    return null;
                }

                return ReservedShaderUserDataManager.LoadReservedShaderIcon(def);
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel
        {
            get;
            private set;
        }

        /// <summary>
        /// The default property page view model to use
        /// on the first time the workspace node is selected.
        /// </summary>
        public override HierarchyViewModel DefaultPropertyPageViewModel
        {
            get
            {
                if (this.Children.Count > 0)
                {
                    return this.Children[0] as HierarchyViewModel;
                }
                else
                {
                    return null;
                }
            }
        }

        /// <summary>
        /// ノードデータをバイナリにコンバートするかどうか取得または設定します。
        /// </summary>
        [UseDataModelOriginalValue]
        public bool EnableConvert
        {
            get
            {
                return this.GetDataModelValue(() => this.EnableConvert);
            }

            set
            {
                using (new MessageBlockerWithSendBinaryOnce(this.DataModel))
                {
                    this.SetDataModelValue(value, () => this.EnableConvert);

                    // エミッタのGPUモードを更新
                    HierarchyViewModel.GetParent<EmitterViewModel>(this).EmitterBasicViewModel.EmitterBasicBasicViewModel.IsUsingCpu = value;
                    //HierarchyViewModel.GetParent<EmitterViewModel>(this).EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu();
                }
            }
        }

        /// <summary>
        /// Gets a value indicating whether enable gpu.
        /// </summary>
        public bool EnableGpu
        {
            get
            {
                if (this.DataModel == null)
                {
                    return false;
                }

                var def = ReservedShaderUserDataManager.FindReservedShaderDefinition(
                    this.DataModel.PageData.ContentsData);
                if (def == null)
                {
                    return false;
                }

                return def.EnableGpu;
            }
        }

        /// <summary>
        /// Gets a value indicating whether enable so.
        /// </summary>
        public bool EnableSo
        {
            get
            {
                if (this.DataModel == null)
                {
                    return false;
                }

                var def = ReservedShaderUserDataManager.FindReservedShaderDefinition(
                    this.DataModel.PageData.ContentsData);
                if (def == null)
                {
                    return false;
                }

                return def.EnableSo;
            }
        }

        /// <summary>
        /// ユーザーデータの中身までチェックし、ペーストが可能かどうかを返します。
        /// </summary>
        /// <param name="data">
        /// クリップボードデータです。
        /// </param>
        /// <returns>
        /// ペーストが可能ならtrue、不可能ならfalse.
        /// </returns>
        protected override bool CanPasteNode(NodeClipboardData data)
        {
            bool result = base.CanPasteNode(data);
            if (!result)
            {
                return false;
            }

            DataModelBase genericDataModel = null;

            {
                var res = DataModelSerializeUtility.DeserializeFromClipboardData(data.XmlData, data.ClassFullName);

                if (res.IsSuccess == false)
                {
                    Logger.Log(LogLevels.Warning, "ReservedShaderNodeViewModelBase.OnNodePaste : Failed to execute Deserialize.");

                    return false;
                }

                genericDataModel = res.DataModel;
            }

            var srcData = genericDataModel as ReservedShaderNodeData;
            var dstData = this.DataModel;
            if (srcData == null || dstData == null)
            {
                return false;
            }

            var srcTypeName = srcData.PageData.ContentsData.GetType().ToString();
            var dstTypeName = dstData.PageData.ContentsData.GetType().ToString();
            return string.Equals(srcTypeName, dstTypeName);
        }
    }
}
