﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.Properties;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the PreviewData.
    /// </summary>
    [Tag("WorkspaceNodeViewModelKind")]
    public class PreviewViewModel : WorkspaceNodeViewModelBase<PreviewData>, IFileOwner, IModificationFlagOwner
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public PreviewViewModel(HierarchyViewModel parent, PreviewData dataModel)
            : base(parent, dataModel)
        {
            var previewBasicViewModel = new PreviewBasicViewModel(this, dataModel.BasicSettingData);
            this.Children.Add(previewBasicViewModel);

            var previewMatrixViewModel = new PreviewMatrixViewModel(this, dataModel.PreviewMatrixData);
            this.Children.Add(previewMatrixViewModel);

            var previewAutoMoveViewModel = new PreviewAutoMoveViewModel(this, dataModel.AutoMoveData);
            this.Children.Add(previewAutoMoveViewModel);

            var previewEmitterViewModel = new PreviewEmitterViewModel(this, dataModel.EmitterData);
            this.Children.Add(previewEmitterViewModel);

            var previewEmissionViewModel = new PreviewEmissionViewModel(this, dataModel.ParticleEmissionData);
            this.Children.Add(previewEmissionViewModel);

            var previewControlViewModel = new PreviewParticleControlViewModel(this, dataModel.ParticleControlData);
            this.Children.Add(previewControlViewModel);

            var previewColorViewModel = new PreviewColorViewModel(this, dataModel.ColorData);
            this.Children.Add(previewColorViewModel);

            var previewScaleViewModel = new PreviewParticleScaleViewModel(this, dataModel.ParticleScaleData);
            this.Children.Add(previewScaleViewModel);

            this.NodeOpenExlorerExecutable = new AnonymousExecutable(this.OnOpenExplorer);
            this.EvaluateMenuItemsExecutable = new AnonymousExecutable(this.OnEvaluateMenuItems);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
        }

        /// <summary>
        /// メニューアイテム(エクスプローラで開く)のExecutable.
        /// </summary>
        public IExecutable NodeOpenExlorerExecutable { get; set; }

        /// <summary>
        /// コンテキストメニューアイテムの状態を評価するExecutable.
        /// </summary>
        public IExecutable EvaluateMenuItemsExecutable { get; set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel
        {
            get;
            private set;
        }

        /// <summary>
        /// The default property page view model to use
        /// on the first time the workspace node is selected.
        /// </summary>
        public override HierarchyViewModel DefaultPropertyPageViewModel
        {
            get
            {
                if (this.Children.Count > 0)
                {
                    return this.Children[0] as HierarchyViewModel;
                }
                else
                {
                    return null;
                }
            }
        }

        /// <summary>
        /// Get the selected item.
        /// </summary>
        public HierarchyViewModel SelectedItem { get; set; }

        /// <summary>
        /// 名前を設定できるかどうか取得します.
        /// </summary>
        public override bool CanSetName
        {
            get { return true; }
        }

        /// <summary>
        /// ファイル名(拡張子を含まない)を設定または取得します。
        /// </summary>
        public string FileName
        {
            get
            {
                return this.GetDataModelValue<string>(() => this.DataModel.Name);
            }

            set
            {
                var binder = new EffectMakerSetMemberBinder("Name", false, false);
                this.TrySetMember(binder, value);

                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// ファイルパス(ファイル名を含まない)を設定または取得します。
        /// </summary>
        public string FilePath
        {
            get { return this.GetDataModelValue(() => this.FilePath); }
            set { this.SetDataModelValue(value, () => this.FilePath); }
        }

        /// <summary>
        /// Get or set the flag indicating whether the data model should be displayed/rendered.
        /// </summary>
        public bool Displayed
        {
            get
            {
                return this.GetDataModelValue(() => this.Displayed);
            }

            set
            {
                bool originalValue = this.GetDataModelValue(() => this.Displayed);
                if (originalValue != value)
                {
                    var binder = new EffectMakerSetMemberBinder("Displayed", false, false);
                    this.TrySetMember(binder, value);

                    // Send visibility message to the viewer.
                    ViewerController.Instance.SendVisibility(this.DataModel);
                }
            }
        }

        /// <summary>
        /// ファイルが存在するかチェックします。
        /// </summary>
        public bool IsFileExist
        {
            get
            {
                if (string.IsNullOrEmpty(this.FileName))
                {
                    return false;
                }

                string path = this.FilePath + @"\" + this.FileName + IOConstants.PreviewFileExtension;
                if (File.Exists(path) == false)
                {
                    return false;
                }

                return true;
            }
        }

        /// <summary>
        /// 名前が正しいかチェックします.
        /// </summary>
        /// <param name="name">名前</param>
        /// <returns>名前が正しいときtrueを返します.</returns>
        public override bool IsNameValid(string name)
        {
            // 名前に不正な文字が入っていないかチェック
            Regex validPattern = new Regex(Resources.RegexpNodeNameValidation);

            if (validPattern.IsMatch(name) == false)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowWarningInvalidName();
                return false;
            }

            // 親のエミッタセットを取得
            EmitterSetViewModel emitterSetViewModel = ViewModelBase.GetParent<EmitterSetViewModel>(this);
            if (emitterSetViewModel == null)
            {
                return false;
            }

            // 同じエミッタセットに属するプレビューを取得
            IEnumerable<PreviewViewModel> previewList =
                emitterSetViewModel.GetChildren<PreviewViewModel>();
            if (previewList == null)
            {
                return true;
            }

            // プレビューの名前が被らないかチェック
            foreach (PreviewViewModel preview in previewList)
            {
                if (preview != this && preview.FileName == name)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowWarningPreviewNameAlreadyUsed(name);
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// 名前を設定します.
        /// </summary>
        /// <param name="name">名前</param>
        public override void SetName(string name)
        {
            // 名前を設定
            this.FileName = name;
            this.OnPropertyChanged(() => this.DataModel.Name);
        }

        /// <summary>
        /// 親エミッタセットの名前を設定します。
        /// </summary>
        /// <param name="esetName">エミッタセット名</param>
        public void SetEsetName(string esetName)
        {
            this.DataModel.EsetName = esetName;
            this.OnPropertyChanged(() => this.DataModel.EsetName);
        }

        /// <summary>
        /// Create a data model proxy.
        /// This method is called in the constructor.
        /// If you need a specific type of data model proxy,
        /// override this method and return the desired data model proxy.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The created data model proxy.</returns>
        protected override DataModelProxy CreateDataModelProxy(DataModelBase dataModel)
        {
            return new PreviewDataProxy(dataModel);
        }

        /// <summary>
        /// エクスプローラで開く.
        /// </summary>
        /// <param name="parameter">パラメータ.</param>
        private void OnOpenExplorer(object parameter)
        {
            if (!this.IsFileExist)
            {
                return;
            }

            string commandline;

            string path = this.FilePath + @"\" + this.FileName + IOConstants.PreviewFileExtension;
            bool selectFile = true;
            if (selectFile == true)
            {
                commandline = "/e, /select,\"" + path + "\"";
            }
            else
            {
                commandline = "/e,\"" + path + "\"";
            }

            System.Diagnostics.Process.Start("EXPLORER.EXE", commandline);
        }

        /// <summary>
        /// コンテキストメニューアイテムの状態を評価する.
        /// </summary>
        /// <param name="parameter">パラメータ.</param>
        private void OnEvaluateMenuItems(object parameter)
        {
            var openExlorerExec = this.NodeOpenExlorerExecutable as AnonymousExecutable;
            openExlorerExec.IsEnabled = this.IsFileExist;
        }
    }
}
