﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using EffectMaker.BusinessLogic.Manager;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.Properties;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the PreviewBasicFileViewModel.
    /// </summary>
    public class PreviewMatrixModelRelationViewModel : PropertyGroupViewModel<PreviewMatrixModelRelationData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "PreviewModelIndexItems",
            "PreviewBoneIndexItems",
            "ModelGuid",
            "BoneIndex",
        };

        /// <summary>
        /// コピーの対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignoreCopyPropertyNames = new string[]
        {
            "ModelGuid",
            "BoneIndex",
        };

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public PreviewMatrixModelRelationViewModel(
            HierarchyViewModel parent, PreviewMatrixModelRelationData dataModel)
            : base(parent, dataModel)
        {
            this.MatrixApplyModeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(Resources.PreviewMatrixApplyModeTRS, 0),
                new KeyValuePair<string, object>(Resources.PreviewMatrixApplyModeOnlyTranslation, 1),
            };

            this.PreviewModelIndexItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(Resources.PreviewMatrixModelRelationRelationLess, Guid.Empty),
            };

            this.PreviewBoneIndexItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(string.Empty, 0),
            };

            ModelInfoManager.ModelNameChanged += this.OnModelNamesChanged;
            ModelInfoManager.BoneNamesChanged += this.OnBoneNamesChanged;

            this.OnModelNamesChanged(null, EventArgs.Empty);
            this.OnBoneNamesChanged(null, EventArgs.Empty);

            this.AddIgnoreCopyProperties(this.ignoreCopyPropertyNames);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>マトリクス適用モードタイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> MatrixApplyModeItems { get; private set; }

        /// <summary>接続モデルタイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> PreviewModelIndexItems { get; private set; }

        /// <summary>接続ボーンタイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> PreviewBoneIndexItems { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// 接続先モデルのGUIDを取得または設定します.
        /// </summary>
        [UseDataModelOriginalValue]
        public Guid ModelGuid
        {
            get
            {
                return this.GetDataModelValue(() => this.ModelGuid);
            }

            set
            {
                if (value == this.GetDataModelValue(() => this.ModelGuid))
                {
                    return;
                }

                using (new CommandCombiner())
                {
                    // GUIDをデータモデルに設定
                    this.SetDataModelValue(value, () => this.ModelGuid);
                    this.UpdateModelName();

                    // ボーン名リストを更新
                    this.OnBoneNamesChanged(null, EventArgs.Empty);

                    // ボーン名を更新
                    this.OnPropertyChanged(() => this.BoneIndex);
                    this.UpdateBoneName();
                }
            }
        }

        /// <summary>
        /// 接続先ボーンのインデックスを取得または設定します.
        /// </summary>
        [UseDataModelOriginalValue]
        public int BoneIndex
        {
            get
            {
                return this.GetDataModelValue(() => this.BoneIndex);
            }

            set
            {
                int boneIndex = this.GetDataModelValue(() => this.BoneIndex);

                if (value != boneIndex)
                {
                    // インデックスをデータモデルに設定
                    this.SetDataModelValue(value, () => this.BoneIndex);
                    this.UpdateBoneName();
                }
            }
        }

        /// <summary>
        /// Dispose時にイベントハンドラへの登録を解除する
        /// </summary>
        public override void Dispose()
        {
            ModelInfoManager.BoneNamesChanged -= this.OnBoneNamesChanged;
            ModelInfoManager.ModelNameChanged -= this.OnModelNamesChanged;
        }

        /// <summary>
        /// The fire property changes.
        /// </summary>
        public override void FirePropertyChanges()
        {
            this.RestoreIndexFromNames();

            // モデル名を更新
            this.OnPropertyChanged(() => this.ModelGuid);
            this.UpdateModelName();

            // ボーン名リストを更新
            this.OnBoneNamesChanged(null, EventArgs.Empty);

            // ボーン名を更新
            this.OnPropertyChanged(() => this.BoneIndex);
            this.UpdateBoneName();

            this.OnPropertyChanged(() => this.DataModel.MatrixApplyMode);
        }

        /// <summary>
        /// モデルとボーンの選択状態を名前から復元します。
        /// </summary>
        private void RestoreIndexFromNames()
        {
            if (string.IsNullOrEmpty(this.DataModel.ModelName))
            {
                var binder = new EffectMakerSetMemberBinder("ModelGuid", false, false);
                this.TrySetMember(binder, Guid.Empty);
            }
            else
            {
                // 保存されているモデル名と一致するものが読み込まれていたらGUIDを復元する
                IEnumerable<KeyValuePair<string, Guid>> models = ModelInfoManager.GetSortedModelNameAndGuidList();
                foreach (KeyValuePair<string, Guid> model in models)
                {
                    if (string.IsNullOrEmpty(model.Key))
                    {
                        continue;
                    }

                    if (model.Key == this.GetDataModelValue<string>("ModelName"))
                    {
                        var binder = new EffectMakerSetMemberBinder("ModelGuid", false, false);
                        this.TrySetMember(binder, model.Value);
                        break;
                    }
                }

                // 保存されているボーン名と一致するものが読み込まれていたらインデックスを復元する
                ModelData modelInfo = ModelInfoManager.GetModel(this.ModelGuid);
                if (modelInfo != null)
                {
                    List<string> boneNames = modelInfo.BoneNames;

                    if (boneNames != null)
                    {
                        int boneIndex = 0;
                        foreach (string boneName in boneNames)
                        {
                            if (string.IsNullOrEmpty(boneName))
                            {
                                continue;
                            }

                            if (boneName == this.GetDataModelValue<string>("BoneName"))
                            {
                                var binder = new EffectMakerSetMemberBinder("BoneIndex", false, false);
                                this.TrySetMember(binder, boneIndex);
                                break;
                            }

                            ++boneIndex;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// モデル名を更新します.
        /// </summary>
        private void UpdateModelName()
        {
            ModelData model = ModelInfoManager.GetModel(this.ModelGuid);

            if (model != null)
            {
                this.SetDataModelValue(model.Name, () => this.DataModel.ModelName);
            }
            else if (this.ModelGuid == Guid.Empty)
            {
                this.SetDataModelValue(string.Empty, () => this.DataModel.ModelName);
            }
            else
            {
                this.OnPropertyChanged(() => this.DataModel.ModelName);
            }
        }

        /// <summary>
        /// ボーン名を更新します.
        /// </summary>
        private void UpdateBoneName()
        {
            ModelData model = ModelInfoManager.GetModel(this.ModelGuid);

            if (model != null && model.BoneNames.Count > this.BoneIndex)
            {
                this.SetDataModelValue(model.BoneNames[this.BoneIndex], () => this.DataModel.BoneName);
            }
            else if (this.ModelGuid == Guid.Empty)
            {
                this.SetDataModelValue(string.Empty, () => this.DataModel.BoneName);
            }
            else
            {
                this.OnPropertyChanged(() => this.DataModel.BoneName);
            }
        }

        /// <summary>
        /// モデル名リストが変わったときの処理を行います.
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnModelNamesChanged(object sender, EventArgs e)
        {
            // 新しいリストを作成
            var items = new List<KeyValuePair<string, object>>();

            // "接続無し"を追加
            items.Add(new KeyValuePair<string, object>(Resources.PreviewMatrixModelRelationRelationLess, Guid.Empty));

            // 登録されているモデル名をすべてアイテムとして追加する
            IEnumerable<KeyValuePair<string, Guid>> models = ModelInfoManager.GetSortedModelNameAndGuidList();
            foreach (KeyValuePair<string, Guid> model in models)
            {
                if (string.IsNullOrEmpty(model.Key) == false)
                {
                    items.Add(new KeyValuePair<string, object>(model.Key, model.Value));
                }
            }

            this.PreviewModelIndexItems = items.ToArray();
            this.OnPropertyChanged(() => this.PreviewModelIndexItems);

            this.RestoreIndexFromNames();

            // 選択中のGUIDに対応するモデルがなかったとき、選択をリセットする
            if (ModelInfoManager.GetModel(this.ModelGuid) == null)
            {
                this.ModelGuid = Guid.Empty;
            }
        }

        /// <summary>
        /// ボーン名リストが変わったときの処理を行います.
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnBoneNamesChanged(object sender, EventArgs e)
        {
            // 新しいリストを作成
            var items = new List<KeyValuePair<string, object>>();

            // インデックスに対応するボーンリストを取得
            ModelData model = ModelInfoManager.GetModel(this.ModelGuid);
            if (model != null)
            {
                List<string> boneNames = model.BoneNames;

                if (boneNames != null)
                {
                    // すべてのボーンアイテムを追加
                    int boneIndex = 0;
                    foreach (string boneName in boneNames)
                    {
                        if (string.IsNullOrEmpty(boneName) == false)
                        {
                            items.Add(new KeyValuePair<string, object>(boneName, boneIndex));
                        }

                        ++boneIndex;
                    }
                }
            }

            this.PreviewBoneIndexItems = items.ToArray();
            this.OnPropertyChanged(() => this.PreviewBoneIndexItems);

            this.RestoreIndexFromNames();

            if (this.BoneIndex > items.Count)
            {
                this.BoneIndex = 0;
            }
        }
    }
}
