﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.Properties;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the PreviewBasicFileViewModel.
    /// </summary>
    public class PreviewBasicFileViewModel : PropertyGroupViewModel<PreviewBasicFileData>, IModificationFlagOwner
    {
        /// <summary>The available label color items.</summary>
        private static KeyValuePair<string, object>[] labelColorItems = new KeyValuePair<string, object>[]
        {
            new KeyValuePair<string, object>("Color 0", 0),
            new KeyValuePair<string, object>("Color 1", 1),
            new KeyValuePair<string, object>("Color 2", 2),
            new KeyValuePair<string, object>("Color 3", 3),
            new KeyValuePair<string, object>("Color 4", 4),
            new KeyValuePair<string, object>("Color 5", 5),
            new KeyValuePair<string, object>("Color 6", 6),
            new KeyValuePair<string, object>("Color 7", 7),
        };

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public PreviewBasicFileViewModel(HierarchyViewModel parent, PreviewBasicFileData dataModel)
            : base(parent, dataModel)
        {
            this.OnPreviewLoadExecutable = new AnonymousExecutable(this.OnPreviewLoad);
            this.OnPreviewLoadNewExecutable = new AnonymousExecutable(this.OnPreviewLoadNew);

            this.OnPreviewSaveExecutable = new AnonymousExecutable(this.OnPreviewSave);
            this.OnPreviewSaveAsExecutable = new AnonymousExecutable(this.OnPreviewSaveAs);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
        }

        /// <summary>
        /// Get the available label color items.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> AvailableLabelColors
        {
            get { return labelColorItems; }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// ファイル名(拡張子を含まない)を設定または取得します。
        /// </summary>
        public string FileName
        {
            get
            {
                string fileName = string.Empty;
                IHierarchyObject parent = this.Parent;

                // プレビューのファイル名を取得
                while (parent != null)
                {
                    var previewViewModel = parent as PreviewViewModel;

                    if (previewViewModel != null)
                    {
                        fileName = previewViewModel.FileName;
                        break;
                    }

                    parent = parent.Parent;
                }

                return fileName;
            }
        }

        /// <summary>
        /// ファイルパス(ファイル名を含まない)を設定または取得します。
        /// </summary>
        public string FilePath
        {
            get
            {
                string filePath = string.Empty;
                IHierarchyObject parent = this.Parent;

                // プレビューのファイルパスを取得
                while (parent != null)
                {
                    var previewViewModel = parent as PreviewViewModel;

                    if (previewViewModel != null)
                    {
                        filePath = previewViewModel.FilePath;
                        break;
                    }

                    parent = parent.Parent;
                }

                // ファイルパスが決まっていないとき、メッセージを表示
                if (string.IsNullOrEmpty(filePath))
                {
                    filePath = Resources.PreviewBasicBasicFileEmptyFilePath;
                }

                return filePath;
            }
        }

        /// <summary>
        /// プレビュー追加の可否を取得します。
        /// </summary>
        public bool CanAddPreview
        {
            get { return WorkspaceRootViewModel.Instance.CanAddPreview; }
        }

        /// <summary>
        /// プレビューを読み込むためのIExecutableを取得します。
        /// </summary>
        public IExecutable OnPreviewLoadExecutable { get; private set; }

        /// <summary>
        /// プレビューを追加して読み込むためのIExecutableを取得します。
        /// </summary>
        public IExecutable OnPreviewLoadNewExecutable { get; private set; }

        /// <summary>
        /// プレビューを保存するためのIExecutableを取得します。
        /// </summary>
        public IExecutable OnPreviewSaveExecutable { get; private set; }

        /// <summary>
        /// プレビューを名前を付けて保存するためのIExecutableを取得します。
        /// </summary>
        public IExecutable OnPreviewSaveAsExecutable { get; private set; }

        /// <summary>
        /// プレビュー名更新用のプロパティ更新
        /// </summary>
        public void RaiseNameChanged()
        {
            this.OnPropertyChanged(() => this.FileName);
        }

        /// <summary>
        /// ファイルパス更新用のプロパティ更新
        /// </summary>
        public void RaiseFilePathChanged()
        {
            this.OnPropertyChanged(() => this.FilePath);
        }

        /// <summary>
        /// プレビューの読み込み処理を行います。
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnPreviewLoad(object parameter)
        {
            // WorkspaceRootViewModeを取得
            WorkspaceRootViewModel workspaceRootViewModel =
                ViewModelBase.GetParent<WorkspaceRootViewModel>(this);

            if (workspaceRootViewModel == null)
            {
                return;
            }

            workspaceRootViewModel.OpenPreviewFile();
        }

        /// <summary>
        /// プレビューを追加して読み込む処理を行います。
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnPreviewLoadNew(object parameter)
        {
            // WorkspaceRootViewModeを取得
            WorkspaceRootViewModel workspaceRootViewModel =
                ViewModelBase.GetParent<WorkspaceRootViewModel>(this);

            if (workspaceRootViewModel == null)
            {
                return;
            }

            workspaceRootViewModel.OpenNewPreviewFile();
        }

        /// <summary>
        /// プレビューの保存処理を行います。
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnPreviewSave(object parameter)
        {
            // WorkspaceRootViewModeを取得
            WorkspaceRootViewModel workspaceRootViewModel =
                ViewModelBase.GetParent<WorkspaceRootViewModel>(this);

            if (workspaceRootViewModel == null)
            {
                return;
            }

            // この関数が呼ばれるときは必ずプレビューノードを選択しているので
            // WorkspaceRootViewModel.OnFilSaveを実行してプレビューの保存画面を開く
            workspaceRootViewModel.OnFileSave(false);
        }

        /// <summary>
        /// プレビューの名前を付けて保存する処理を行います。
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnPreviewSaveAs(object parameter)
        {
            // WorkspaceRootViewModeを取得
            WorkspaceRootViewModel workspaceRootViewModel =
                ViewModelBase.GetParent<WorkspaceRootViewModel>(this);

            if (workspaceRootViewModel == null)
            {
                return;
            }

            // この関数が呼ばれるときは必ずプレビューノードを選択しているので
            // WorkspaceRootViewModel.OnFilSaveを実行してプレビューの保存画面を開く
            workspaceRootViewModel.OnFileSave(true);
        }
    }
}
