﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic;
using EffectMaker.BusinessLogic.Manager;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Dynamic;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the ViewerData.
    /// </summary>
    [Tag("WorkspaceNodeViewModelKind")]
    public class ModelViewModel : WorkspaceNodeViewModelBase<ModelData>, IModificationFlagOwner
    {
        /// <summary>The property page view model for this node.</summary>
        private PropertyPageViewModel propertyPageViewModel;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public ModelViewModel(HierarchyViewModel parent, ModelData dataModel)
            : base(parent, dataModel)
        {
            this.propertyPageViewModel = new ModelBasicViewModel(this, dataModel.ModelBasicSettingData);
            this.Children.Add(this.propertyPageViewModel);

            // ユーザーページを作成
            this.UpdateUserData(true);

            // ユーザーデータのアップデート通知イベントを登録.
            UserDataManager.UserDataUpdated += this.OnUserDataUpdated;
            ModelInfoManager.ModelNameChanged += this.OnModelNameChanged;

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(new[] { "Name" });   // ノード名はスターマークの対象外
        }

        /// <summary>
        /// モデル表示名を取得します.
        /// </summary>
        public string Name
        {
            get
            {
                // モデル名が設定されているとき、モデル名を返す
                string name = this.GetDataModelValue<string>(() => this.Name);
                if (string.IsNullOrEmpty(name) == false)
                {
                    return this.Proxy.Name;
                }

                // 何も設定されていないとき、"モデル未設定"を返す
                return Properties.Resources.ModelTreeNodeDefaultName;
            }

            set
            {
                this.UpdateName(value, true);
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether the data model should be displayed/rendered.
        /// </summary>
        public bool Displayed
        {
            get
            {
                return this.GetDataModelValue(() => this.Displayed);
            }

            set
            {
                bool originalValue = this.GetDataModelValue(() => this.Displayed);
                if (originalValue != value)
                {
                    var binder = new EffectMakerSetMemberBinder("Displayed", false, false);
                    this.TrySetMember(binder, value);

                    // Send visibility message to the viewer.
                    ViewerController.Instance.SendVisibility(this.DataModel);
                }
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// The default property page view model to use
        /// on the first time the workspace node is selected.
        /// </summary>
        public override HierarchyViewModel DefaultPropertyPageViewModel
        {
            get { return this.propertyPageViewModel; }
        }

        /// <summary>
        /// Get the selected item.
        /// </summary>
        public HierarchyViewModel SelectedItem { get; set; }

        /// <summary>
        /// Disposes the instance.
        /// </summary>
        public override void Dispose()
        {
            base.Dispose();

            // ユーザーデータアップデートの通知イベントを解除.
            ModelInfoManager.ModelNameChanged -= this.OnModelNameChanged;
            UserDataManager.UserDataUpdated -= this.OnUserDataUpdated;
        }

        /// <summary>
        /// ビューアにモデルバイナリを送信します。
        /// </summary>
        /// <param name="forceClear">強制的にクリアする際にtrue,転送する場合はfalseか省略.</param>
        public void SendModelInfo(bool forceClear = false)
        {
            // モデルのファイルパスを取得
            var modelBasicViewModel = this.GetChild<ModelBasicViewModel>();
            var modelBasicModelViewModel = modelBasicViewModel.GetChild<ModelBasicModelViewModel>();
            string modelFilePath = modelBasicModelViewModel.ModelFilePath;

            if (Path.IsPathRooted(modelFilePath) && !forceClear)
            {
                //// モデルファイルが指定されていたとき、モデル作成・更新メッセージを送信
                // モデルノードを削除
                ViewerMessageHelper.RequestDeleteModels(new [] { this.DataModel });

                // モデルバイナリを更新
                ModelInfoManager.UpdateModel(this.DataModel);

                // モデルノードを作成
                ViewerMessageHelper.SendModels(new [] { this.DataModel });
            }
            else
            {
                //// モデルファイルが指定されていなかったとき、モデル削除メッセージを送信
                // モデルノードを削除
                ViewerMessageHelper.RequestDeleteModels(new [] { this.DataModel });

                ModelInfoManager.RaiseModelEvents();
            }
        }

        /// <summary>
        /// Update child view models with the current data model.
        /// This method is usually called when data model is modified, thus some child
        /// view models might need to be created or removed according to the data model.
        /// </summary>
        public override void UpdateChildViewModels()
        {
            this.UpdateUserData(true);
            base.UpdateChildViewModels();
        }

        /// <summary>
        /// モデルノード名を更新します。
        /// </summary>
        /// <param name="value">ノード名</param>
        /// <param name="issueCommand">履歴に詰むか否か</param>
        public void UpdateName(string value, bool issueCommand)
        {
            var binder = new EffectMakerSetMemberBinder("Name", false, issueCommand);
            this.TrySetMember(binder, value);
            this.OnModelNameChanged(null, EventArgs.Empty);
        }

        /// <summary>
        /// データモデルプロキシを作成する.
        /// Create a data model proxy.
        /// This method is called in the constructor.
        /// If you need a specific type of data model proxy,
        /// override this method and return the desired data model proxy.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The created data model proxy.</returns>
        protected override DataModelProxy CreateDataModelProxy(DataModelBase dataModel)
        {
            return new ModelDataProxy(dataModel);
        }

        /// <summary>
        /// モデル名が変わったときの処理を行います.
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnModelNameChanged(object sender, EventArgs e)
        {
            this.OnPropertyChanged(() => this.Name);
        }

        /// <summary>
        /// Handle UserDataUpdated event from the user data manager.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnUserDataUpdated(
            object sender,
            UserDataUpdatedEventArgs e)
        {
            if (e.IsUserDataTypeUpdated(UpdatedUserDataTypes.GeneralUserPage) == false)
            {
                // The updated user data is not for model data, bail out.
                return;
            }

            this.UpdateUserData(false);
        }

        /// <summary>
        /// Helper method for updating user data.
        /// </summary>
        /// <param name="onlyUpdateViewModels">
        /// True to update only the view models for to the existing user data models.
        /// Otherwise, remove all the user data models and update them.
        /// </param>
        private void UpdateUserData(bool onlyUpdateViewModels)
        {
            // First remove all the user page view models.
            var userPages = this.Children.Where(ch => ch is UserPageViewModel).ToArray();
            foreach (var userPage in userPages)
            {
                this.Children.Remove(userPage);
            }

            if (onlyUpdateViewModels == false)
            {
                // Update user data models.
                (this.Proxy as ModelDataProxy).UpdateUserData();
            }

            // Create user page view models with the new user data models.
            foreach (UserPageData page in this.DataModel.UserPages)
            {
                var userPageViewModel = new UserPageViewModel(this, page);
                this.Children.Add(userPageViewModel);
            }
        }
    }
}
