﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Manager;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.FileMonitor;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.Properties;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the ModelBasicModelViewModel.
    /// </summary>
    public class ModelBasicModelViewModel : PropertyGroupViewModel<ModelBasicModelData>, IModificationFlagOwner
    {
        /// <summary>
        /// ファイルモニタイベントをロックするフラグ
        /// </summary>
        private bool monitorEventLock = false;

        /// <summary>
        /// ファイルモニタをViewModelごとに持つことにした
        /// </summary>
        private FileMonitorManager monitorManager = null;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public ModelBasicModelViewModel(HierarchyViewModel parent, ModelBasicModelData dataModel)
            : base(parent, dataModel)
        {
            this.OnReloadModelFileExecutable = new AnonymousExecutable(this.OnReloadModelFile);
            this.OnClearModelFileExecutable = new AnonymousExecutable(this.OnClearModelFile);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// モデルのファイルパスを取得または設定します。
        /// </summary>
        [UseDataModelOriginalValue]
        public string ModelFilePath
        {
            get
            {
                return this.GetDataModelValue(() => this.ModelFilePath);
            }

            set
            {
                if (!string.IsNullOrEmpty(this.ModelFilePath) && Path.IsPathRooted(this.ModelFilePath))
                {
                    this.MonitorManager.UnregisterMonitor(this.ModelFilePath);
                }

                if (!string.IsNullOrEmpty(value) && Path.IsPathRooted(value))
                {
                    this.MonitorManager.RegisterMonitorPreCopyPath(value, null, this.OnMonitorEvent);
                }

                using (new CommandCombiner())
                {
                    // モデルのファイルパスを設定
                    if (this.SetDataModelValue(value, () => this.ModelFilePath))
                    {
                        // モデル情報を送信
                        var modelViewModel = this.Parent.Parent as ModelViewModel;
                        Debug.Assert(modelViewModel != null, "Hierarchy of ViewModel wrong");
                        modelViewModel.SendModelInfo();

                        // ファイル名をモデルノード名に設定
                        string fileName = Path.GetFileNameWithoutExtension(value);
                        modelViewModel.Name = string.IsNullOrEmpty(fileName) == false ?
                            fileName : string.Empty;

                        ModelData modelData = ((ModelViewModel)this.Parent.Parent).DataModel;
                        int binaryId = ModelInfoManager.GetModelId(modelData);
                        modelViewModel.SetDataModelValue(binaryId, () => modelViewModel.DataModel.ModelBinaryId);
                    }
                }

                WorkspaceRootViewModel.Instance.UpdateCloseAllState();
            }
        }

        /// <summary>
        /// モデルファイルをリロードするためのIExecutableを取得します。
        /// </summary>
        public IExecutable OnReloadModelFileExecutable { get; private set; }

        /// <summary>
        /// モデルファイルをクリアするためのIExecutableを取得します。
        /// </summary>
        public IExecutable OnClearModelFileExecutable { get; private set; }

        /// <summary>
        /// モニターマネージャを取得します。
        /// </summary>
        private FileMonitorManager MonitorManager
        {
            get
            {
                return this.monitorManager ??
                    (this.monitorManager = new FileMonitorManager(Application.OpenForms["MainForm"], true));
            }
        }

        /// <summary>
        /// Disposer
        /// </summary>
        public override void Dispose()
        {
            if (!string.IsNullOrEmpty(this.ModelFilePath))
            {
                this.MonitorManager.UnregisterMonitor(this.ModelFilePath);
            }

            base.Dispose();
        }

        /// <summary>
        /// ペースト時にモデルノード名を更新する処理を追加
        /// </summary>
        public override void FirePropertyChanges()
        {
            base.FirePropertyChanges();

            var modelViewModel = this.Parent.Parent as ModelViewModel;
            Debug.Assert(modelViewModel != null, "Hierarchy of ViewModel wrong");
            string fileName = Path.GetFileNameWithoutExtension(this.ModelFilePath);
            modelViewModel.UpdateName(fileName, false);

            if (string.IsNullOrEmpty(this.ModelFilePath))
            {
                this.OnClearModelFile(null);
            }
        }

        /// <summary>
        /// モデルファイルのリロード処理を行います。
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnReloadModelFile(object parameter)
        {
            // モデル情報を送信
            var modelViewModel = this.Parent.Parent as ModelViewModel;
            Debug.Assert(modelViewModel != null, "Hierarchy of ViewModel wrong");
            modelViewModel.SendModelInfo();
        }

        /// <summary>
        /// モデルファイルのクリア処理を行います。
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnClearModelFile(object parameter)
        {
            var modelViewModel = this.Parent.Parent as ModelViewModel;
            Debug.Assert(modelViewModel != null, "Hierarchy of ViewModel wrong");

            using (new CommandCombiner())
            {
                // モデルのファイルパスをクリア
                this.SetDataModelValue(string.Empty, () => this.ModelFilePath);

                // モデルのGUIDをクリア
                ModelData modelData = ((ModelViewModel)this.Parent.Parent).DataModel;
                int binaryId = ModelInfoManager.GetModelId(modelData);
                modelViewModel.SetDataModelValue(binaryId, () => modelViewModel.DataModel.ModelBinaryId);

                // モデルノード名をクリア
                modelViewModel.Name = string.Empty;

                // モデルのボーンリストをクリア
                modelViewModel.SetDataModelValue(new List<string>(), () => modelViewModel.DataModel.BoneNames);

                // モデル情報を送信
                modelViewModel.SendModelInfo();
            }
        }

        /// <summary>
        /// ファイル監視システムのイベントがあった場合の処理
        /// </summary>
        /// <param name="path">監視パス</param>
        /// <param name="userData">ユーザーデータ(未使用)</param>
        /// <param name="watcherChangeTypes">変更タイプ</param>
        private void OnMonitorEvent(string path, object userData, WatcherChangeTypes watcherChangeTypes)
        {
            // プリミティブの自動リロードがOffになっていたら、何もしない
            if (OptionStore.RootOptions.Basic.Primitives.AutoReload == false)
            {
                return;
            }

            if (this.monitorEventLock)
            {
                return;
            }

            this.MonitorManager.UnregisterMonitor(this.ModelFilePath);

            this.monitorEventLock = true;

            switch (watcherChangeTypes)
            {
                case WatcherChangeTypes.Changed:
                case WatcherChangeTypes.Created:
                    this.OnReloadModelFile("ModelFilePath");
                    break;
                case WatcherChangeTypes.Renamed:
                    if (File.Exists(path))
                    {
                        this.OnReloadModelFile("ModelFilePath");
                    }
                    else
                    {
                        Logger.Log("LogView", LogLevels.Warning, Resources.WarningLostModelFile, this.ModelFilePath);
                    }

                    break;
                case WatcherChangeTypes.Deleted:
                    Logger.Log("LogView", LogLevels.Warning, Resources.WarningLostModelFile, this.ModelFilePath);
                    break;
            }

            this.monitorEventLock = false;

            this.MonitorManager.RegisterMonitorPreCopyPath(
                this.ModelFilePath, null, this.OnMonitorEvent);
        }
    }
}
