﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.Properties;
using EffectMaker.UILogic.ViewModels.IO;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// メインで使用するダイアログです。
    /// TODO: WorkspaceRootViewModelから取り出してきただけなのでInterface化する。
    /// </summary>
    public class MainDialogs
    {
        /// <summary>
        /// 描画パスの不整合を全ての項目に適用するかどうか
        /// </summary>
        private bool applyAllInvalidDrawPath;

        /// <summary>
        /// 保存された不正描画パスダイアログの選択内容
        /// </summary>
        private DialogResult savedResultInvalidDrawPathDialog;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MainDialogs()
        {
        }

        #region IExecutable - ファイルの保存

        /// <summary>
        /// 変更保存ダイアログを表示するExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowSaveModificationDialogExecutable { get; set; }

        /// <summary>
        /// エミッタセット保存ダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowSaveEmitterSetDialogExecutable { get; set; }

        /// <summary>
        /// プレビュー保存ダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowSavePreviewDialogExecutable { get; set; }

        /// <summary>
        /// ワークスペース保存ダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowSaveWorkspaceDialogExecutable { get; set; }

        /// <summary>
        /// エクスポートダイアログを表示するExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowExportDialogExecutable { get; set; }

        #endregion

        #region IExecutable - カーブエディタ呼び出し

        /// <summary>
        /// カーブエディタウィンドウを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowCurveEditorExecutable { get; set; }

        /// <summary>
        /// カーブエディタにViewModelを接続するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnConnectCurveEditorExecutable { get; set; }

        /// <summary>
        /// カーブエディタからViewModelを切断するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnDisconnectCurveEditorExecutable { get; set; }

        /// <summary>
        /// カーブエディタに選択されているエミッタの変更を通知するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnNotifyEmitterToCurveEditorExecutable { get; set; }

        /// <summary>
        /// カーブエディタに開いているタブページの変更を通知するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnNotifyTabPageToCurveEditorExecutable { get; set; }

        #endregion

        #region IExecutable - ファイルの読み込み

        /// <summary>
        /// ファイルを開くダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowFileOpenDialogExecutable { get; set; }

        /// <summary>
        /// ファイル読み込み時のエラーを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowFileOpenErrorExecutable { get; set; }

        /// <summary>
        /// カスタムアクションのコールバックIDが更新されたことを通知するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowCustomActionIndexUpdatedDialogExecutable { get; set; }

        /// <summary>
        /// プレビューを開くダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowPreviewFileOpenDialogExecutable { get; set; }

        /// <summary>
        /// ワークスペースを開くダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowWorkspaceFileOpenDialogExecutable { get; set; }

        /// <summary>
        /// プレビューに関連したエミッタセットファイルが見つからないことを通知するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowEsetFileNotFoundDialogExecutable { get; set; }

        /// <summary>
        /// 読み込んだファイルと同じ名前のプレビューがエミッタセットにあることを通知するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowPreviewFileAlreadyExistsDialogExecutable { get; set; }

        /// <summary>
        /// FE1世代のesetアップデート中にプログレスバダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowAncientEsetUpdaterDialogExecutable { get; set; }

        #endregion

        #region IExecutable - 名前入力

        /// <summary>
        /// エミッタセット名入力ダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowEmitterSetNameInputDialogExecutable { get; set; }

        /// <summary>
        /// エミッタ名入力ダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowEmitterNameInputDialogExecutable { get; set; }

        /// <summary>
        /// プレビュー名入力ダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowPreviewNameInputDialogExecutable { get; set; }

        /// <summary>
        /// 入力された名前が不正なことを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowWarningInvalidNameExecutable { get; set; }

        /// <summary>
        /// 入力された名前が使用済みであることを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowWarningNameAlreadyUsedExecutable { get; set; }

        #endregion

        #region IExecutable - アセット処理

        /// <summary>
        /// テクスチャのリニアモードがオプションと違うことを通知するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowTextureTypeMismatchExecutable { get; set; }

        /// <summary>
        /// テクスチャのパスが、エミッタセットの探索範囲外であることを通知します。
        /// </summary>
        public IExecutable OnShowWarningOnLoadTextureExecutable { get; set; }

        /// <summary>
        /// プリミティブのパスが、エミッタセットの探索範囲外であることを通知します。
        /// </summary>
        public IExecutable OnShowWarningOnLoadPrimitiveExecutable { get; set; }

        /// <summary>
        /// テクスチャが設定されておらず保存をキャンセルするかどうか選択するダイアログを表示します。
        /// </summary>
        public IExecutable OnShowWarningTextureIsNotAssignedExecutable { get; set; }

        /// <summary>
        /// アセットファイルの上書き確認をするIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnPromptOverWriteAssetExecutable { get; set; }

        /// <summary>
        /// アセットファイルのコピーダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowCopyAssetDialogExecutable { get; set; }

        /// <summary>
        /// アセットファイルの参照エラーを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowAssetPathProblemExecutable { get; set; }

        /// <summary>
        /// 無効な終了処理タイプの警告の表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowInvalidEndingTypeExecutable { get; set; }

        /// <summary>
        /// 描画パスに不整合があることを通知するダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowInvalidDrawPathDialogExecutable { get; set; }

        #endregion

        #region IExecutable - その他

        /// <summary>
        /// エラーダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowErrorDialogExecutable { get; set; }

        /// <summary>
        /// 警告ダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowWarningDialogExecutable { get; set; }

        /// <summary>
        /// OK・キャンセルの確認ダイアログを表示するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnShowOkCancelDialogExecutable { get; set; }

        #endregion

        #region Wrapper Function - ファイルの保存

        /// <summary>
        /// 変更保存ダイアログを表示します。
        /// </summary>
        /// <param name="fileName">ファイル名</param>
        /// <returns>DialogResult</returns>
        public DialogResult ShowSaveModificationDialog(string fileName)
        {
            if (this.OnShowSaveModificationDialogExecutable == null)
            {
                return DialogResult.No;
            }

            string message = string.Format(
                Resources.ConfirmSaveModification,
                fileName);

            object[] parameters = new object[] { null, message, Resources.ConfirmCaption };

            this.OnShowSaveModificationDialogExecutable.Execute(parameters);

            return (DialogResult)parameters[0];
        }

        /// <summary>
        /// エミッタセット保存ダイアログを表示します。
        /// </summary>
        /// <param name="fileName">初期ファイル名</param>
        /// <param name="filePath">初期ファイルパス(省略可)</param>
        /// <returns>ファイル名を返します。</returns>
        public string ShowSaveEmitterSetDialog(string fileName, string filePath = "")
        {
            if (this.OnShowSaveEmitterSetDialogExecutable == null)
            {
                return null;
            }

            object[] parameters = new object[] { null, fileName, filePath };

            this.OnShowSaveEmitterSetDialogExecutable.Execute(parameters);

            return (string)parameters[0];
        }

        /// <summary>
        /// プレビュー保存ダイアログを表示します。
        /// </summary>
        /// <param name="fileName">初期ファイル名</param>
        /// <returns>ファイル名を返します。</returns>
        public string ShowSavePreviewDialog(string fileName)
        {
            if (this.OnShowSavePreviewDialogExecutable == null)
            {
                return null;
            }

            object[] parameters = new object[] { null, fileName };

            this.OnShowSavePreviewDialogExecutable.Execute(parameters);

            return (string)parameters[0];
        }

        /// <summary>
        /// ワークスペース保存ダイアログを表示します。
        /// </summary>
        /// <param name="fileName">初期ファイル名</param>
        /// <returns>ファイル名を返します。</returns>
        public string ShowSaveWorkspaceDialog(string fileName)
        {
            if (this.OnShowSaveWorkspaceDialogExecutable == null)
            {
                return null;
            }

            object[] parameters = new object[] { null, fileName };

            this.OnShowSaveWorkspaceDialogExecutable.Execute(parameters);

            return (string)parameters[0];
        }

        /// <summary>
        /// エクスポートダイアログを表示します。
        /// </summary>
        /// <param name="fileList">ファイルリスト</param>
        /// <returns>ダイアログでの選択結果をリストのリストで返します。</returns>
        public object ShowExportDilaog(List<string> fileList)
        {
            if (this.OnShowExportDialogExecutable == null)
            {
                return null;
            }

            object[] parameters = new object[] { null, fileList };

            this.OnShowExportDialogExecutable.Execute(parameters);

            return parameters[0];
        }

        #endregion

        #region Wrapper Function - ファイルの読み込み

        /// <summary>
        /// ファイルを開くダイアログを表示します。
        /// </summary>
        /// <returns>選択されたファイルリストを返します。</returns>
        public string[] ShowFileOpenDialog()
        {
            if (this.OnShowFileOpenDialogExecutable == null)
            {
                return null;
            }

            object[] parameters = new object[1];

            this.OnShowFileOpenDialogExecutable.Execute(parameters);

            return (string[])parameters[0];
        }

        /// <summary>
        /// 開いたエミッタセットの名前が制限を超えていたときのエラーを表示します。
        /// </summary>
        /// <param name="filePath">エミッタセットのファイルパス</param>
        public void ShowOpenFileNameLengthError(string filePath)
        {
            string message = string.Format(
                Resources.WarningOpenFileNameLength,
                Path.GetFileName(filePath));

            object[] parameters = new object[] { message, Resources.WarningCaption };

            this.OnShowFileOpenErrorExecutable.Execute(parameters);
        }

        /// <summary>
        /// 開いたエミッタセットの名前が重複していたときのエラーを表示します。
        /// </summary>
        /// <param name="filePath">エミッタセットのファイルパス</param>
        public void ShowOpenFileNameCollisionError(string filePath)
        {
            string message = string.Format(
                Resources.WarningOpenFileNameCollision,
                Path.GetFileName(filePath));

            object[] parameters = new object[] { message, Resources.WarningCaption };

            this.OnShowFileOpenErrorExecutable.Execute(parameters);
        }

        /// <summary>
        /// カスタムアクションのコールバックIDが書き換えられたことを通知するダイアログを表示します。
        /// </summary>
        /// <param name="esetName">エミッタセット名</param>
        public void ShowCustomActionIndexUpdated(string esetName)
        {
            if (this.OnShowCustomActionIndexUpdatedDialogExecutable == null)
            {
                return;
            }

            string message = string.Format(
                Resources.WarningCustomActionIndexUpdated,
                esetName);

            object[] parameters = new string[] { message, Resources.WarningCaption };

            this.OnShowCustomActionIndexUpdatedDialogExecutable.Execute(parameters);
        }

        /// <summary>
        /// ユーザーデータ定義が定義不一致により書き換えられたことを通知するダイアログを表示します。
        /// </summary>
        /// <param name="esetName">エミッタセット名</param>
        public void ShowEmitterUserDataDefinitionReplaced(string esetName)
        {
            if (this.OnShowWarningDialogExecutable == null)
            {
                return;
            }

            string message = string.Format(
                Resources.WarningUserDataDefinitionReplaced,
                esetName);

            object[] parameters = new string[] { message, Resources.WarningCaption };

            this.OnShowWarningDialogExecutable.Execute(parameters);
        }

        /// <summary>
        /// プレビューファイルを開くダイアログを表示します。
        /// </summary>
        /// <returns>選択されたファイルを返します。</returns>
        public string ShowPreviewFileOpenDialog()
        {
            if (this.OnShowPreviewFileOpenDialogExecutable == null)
            {
                return null;
            }

            object[] parameters = new object[1];

            this.OnShowPreviewFileOpenDialogExecutable.Execute(parameters);

            return (string)parameters[0];
        }

        /// <summary>
        /// プレビューに関連したエミッタセットファイルが見つからないことを通知します。
        /// </summary>
        /// <param name="filePath">エミッタセットのファイルパス</param>
        public void ShowEsetFileNotFoundError(string filePath)
        {
            string message = string.Format(
                Resources.ErrorEsetFileNotFound,
                filePath);

            object[] parameters = new object[] { message, Resources.WarningCaption };

            this.OnShowEsetFileNotFoundDialogExecutable.Execute(parameters);
        }

        /// <summary>
        /// 読み込んだファイルと同じ名前のプレビューがエミッタセットにあることを通知します。
        /// </summary>
        public void ShowPreviewFileAlreadyExistsDialog()
        {
            object[] parameters = new object[]
            {
                Resources.ErrorPrevFileAlreadyExists,
                Resources.ErrorLoadPrevFileCaption
            };

            this.OnShowPreviewFileAlreadyExistsDialogExecutable.Execute(parameters);
        }

        /// <summary>
        /// プレビューの個数が上限に達していることを通知するダイアログを表示します。
        /// </summary>
        /// <param name="esetName">エミッタセット名</param>
        public void ShowPreviewCountMaximumDialog(string esetName)
        {
            if (this.OnShowWarningDialogExecutable == null)
            {
                return;
            }

            string message = string.Format(
                Resources.ErrorPreviewCountMaximum,
                esetName);

            object[] parameters = new string[] { message, Resources.WarningCaption };

            this.OnShowWarningDialogExecutable.Execute(parameters);
        }

        /// <summary>
        /// 読み込んだファイルのバージョンに不整合があることを通知するダイアログを表示します。
        /// </summary>
        public void ShowVersionMismatchDialog()
        {
            if (this.OnShowErrorDialogExecutable == null)
            {
                return;
            }

            object[] parameters = new string[] {
                Resources.ErrorVersionMismatch,
                Resources.ErrorCaption
            };

            this.OnShowErrorDialogExecutable.Execute(parameters);
        }

        #endregion

        #region Wrapper Function - 名前入力

        /// <summary>
        /// エミッタセット名入力ダイアログを表示します。
        /// </summary>
        /// <param name="inputedName">入力した名前</param>
        /// <param name="defaultName">デフォルトの名前</param>
        /// <param name="emitterSetNameList">エミッタセット名のリスト</param>
        /// <param name="presetList">プリセットリスト</param>
        /// <param name="selectedPreset">選択されたプリセットのインデックス</param>
        /// <returns>DialogResult</returns>
        public DialogResult ShowEmitterSetNameInputDialog(
            out string inputedName,
            string defaultName,
            IEnumerable<string> emitterSetNameList,
            List<PresetItem> presetList,
            ref int selectedPreset)
        {
            inputedName = null;
            Regex validPattern = new Regex(Resources.RegexpNodeNameValidation);

            DialogResult result;
            string name = defaultName;

            while (true)
            {
                // 名前入力ダイアログを表示
                object[] parameters =
                {
                    null,
                    null,
                    Resources.NameInputDialogCaptionEmitterSet,
                    name,
                    presetList,
                    selectedPreset,
                    emitterSetNameList,
                };

                this.OnShowEmitterSetNameInputDialogExecutable.Execute(parameters);

                result = (DialogResult)parameters[0];
                name = (string)parameters[1];
                selectedPreset = (int)parameters[2];

                // OKを押されなかったとき入力内容のチェックを行わずに処理をキャンセルする
                if (result != DialogResult.OK)
                {
                    return result;
                }

                // 入力内容が不正なとき、エラーを表示して再度入力ダイアログを表示する
                if (!validPattern.IsMatch(name))
                {
                    this.ShowWarningInvalidName();
                    continue;
                }

                // 同名のエミッタセットが既にあるとき、エラーを表示して再度入力ダイアログを表示する
                if (emitterSetNameList.FirstOrDefault(item => string.Equals(item, name)) != null)
                {
                    this.ShowWarningEmitterSetNameAlreadyUsed(name);
                    continue;
                }

                // 入力した名前に問題がなければ処理を終了する
                inputedName = name;

                break;
            }

            return result;
        }

        /// <summary>
        /// エミッタ名入力ダイアログを表示します。
        /// </summary>
        /// <param name="inputedName">入力した名前</param>
        /// <param name="defaultName">デフォルトの名前</param>
        /// <param name="emitterNameList">エミッタ名のリスト</param>
        /// <param name="presetList">プリセットリスト</param>
        /// <param name="selectedPreset">選択されたプリセットのインデックス</param>
        /// <returns>DialogResult</returns>
        public DialogResult ShowEmitterNameInputDialog(
            out string inputedName,
            string defaultName,
            IEnumerable<string> emitterNameList,
            List<PresetItem> presetList,
            ref int selectedPreset)
        {
            inputedName = null;
            Regex validPattern = new Regex(Resources.RegexpNodeNameValidation);

            DialogResult result;
            string name = defaultName;

            while (true)
            {
                // 名前入力ダイアログを表示
                object[] parameters =
                {
                    null,
                    null,
                    Resources.NameInputDialogCaptionEmitter,
                    name,
                    presetList,
                    selectedPreset,
                    emitterNameList,
                };

                this.OnShowEmitterNameInputDialogExecutable.Execute(parameters);

                result = (DialogResult)parameters[0];
                name = (string)parameters[1];
                selectedPreset = (int)parameters[2];

                // OKを押されなかったとき入力内容のチェックを行わずに処理をキャンセルする
                if (result != DialogResult.OK)
                {
                    return result;
                }

                // 入力内容が不正なとき、エラーを表示して再度入力ダイアログを表示する
                if (!validPattern.IsMatch(name))
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowWarningInvalidName();
                    continue;
                }

                // 同名のエミッタが既にあるとき、エラーを表示して再度入力ダイアログを表示する
                if (emitterNameList.FirstOrDefault(item => string.Equals(item, name)) != null)
                {
                    this.ShowWarningEmitterNameAlreadyUsed(name);
                    continue;
                }

                // 入力した名前に問題がなければ処理を終了する
                inputedName = name;
                break;
            }

            return result;
        }

        /// <summary>
        /// プレビュー名入力ダイアログを表示します。
        /// </summary>
        /// <param name="inputedName">入力した名前</param>
        /// <param name="defaultName">デフォルトの名前</param>
        /// <param name="previewNameList">プレビュー名のリスト.</param>
        /// <returns>DialogResult</returns>
        public DialogResult ShowPreviewNameInputDialog(
            out string inputedName,
            string defaultName,
            IEnumerable<string> previewNameList)
        {
            inputedName = null;
            Regex validPattern = new Regex(Resources.RegexpNodeNameValidation);

            DialogResult result;
            string name = defaultName;

            while (true)
            {
                // 名前入力ダイアログを表示
                object[] parameters = new object[] { null, null, Resources.NameInputDialogCaptionPreview, defaultName };

                this.OnShowPreviewNameInputDialogExecutable.Execute(parameters);

                result = (DialogResult)parameters[0];
                name = (string)parameters[1];

                // OKを押されなかったとき入力内容のチェックを行わずに処理をキャンセルする
                if (result != DialogResult.OK)
                {
                    return result;
                }

                // 入力内容が不正なとき、エラーを表示して再度入力ダイアログを表示する
                if (!validPattern.IsMatch(name))
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowWarningInvalidName();
                    continue;
                }

                // 同名のエミッタが既にあるとき、エラーを表示して再度入力ダイアログを表示する
                if (previewNameList.FirstOrDefault(item => string.Equals(item, name)) != null)
                {
                    this.ShowWarningPreviewNameAlreadyUsed(name);
                    continue;
                }

                // 入力した名前に問題がなければ処理を終了する
                inputedName = name;
                break;
            }

            return result;
        }

        /// <summary>
        /// 入力された名前が不正なことを表示します。
        /// </summary>
        public void ShowWarningInvalidName()
        {
            object[] parameters = new object[] { Resources.WarningInvalidName, Resources.WarningInvalidNameCaption };

            this.OnShowWarningInvalidNameExecutable.Execute(parameters);
        }

        /// <summary>
        /// 入力されたエミッタ名が長すぎることを表示します。
        /// </summary>
        public void ShowWarningTooLongEmitterName()
        {
            object[] parameters = new object[] { Resources.WarningTooLongEmitterName, Resources.WarningInvalidNameCaption };

            this.OnShowWarningInvalidNameExecutable.Execute(parameters);
        }

        /// <summary>
        /// 入力されたエミッタセット名が使用済みであることを表示します。
        /// </summary>
        /// <param name="name">エミッタセット名</param>
        public void ShowWarningEmitterSetNameAlreadyUsed(string name)
        {
            string msg = string.Format(Resources.WarningEmitterSetNameAlreadyUsed, name);

            object[] parameters = new object[] { msg, Resources.WarningInvalidNameCaption };

            this.OnShowWarningNameAlreadyUsedExecutable.Execute(parameters);
        }

        /// <summary>
        /// 入力されたエミッタ名が使用済みであることを表示します。
        /// </summary>
        /// <param name="name">エミッタ名</param>
        public void ShowWarningEmitterNameAlreadyUsed(string name)
        {
            string msg = string.Format(Resources.WarningEmitterNameAlreadyUsed, name);

            object[] parameters = new object[] { msg, Resources.WarningInvalidNameCaption };

            this.OnShowWarningNameAlreadyUsedExecutable.Execute(parameters);
        }

        /// <summary>
        /// 入力されたプレビュー名が使用済みであることを表示します。
        /// </summary>
        /// <param name="name">プレビュー名</param>
        public void ShowWarningPreviewNameAlreadyUsed(string name)
        {
            string msg = string.Format(Resources.WarningPreviewNameAlreadyUsed, name);

            object[] parameters = new object[] { msg, Resources.WarningInvalidNameCaption };

            this.OnShowWarningNameAlreadyUsedExecutable.Execute(parameters);
        }

        #endregion

        #region Wrapper Function - アセット処理

        /// <summary>
        /// テクスチャのリニアモードがオプションと違うことを通知します。
        /// </summary>
        /// <param name="text">エラーの説明</param>
        public void ShowTextureTypeMismatch(string text)
        {
            if (this.OnShowTextureTypeMismatchExecutable == null)
            {
                return;
            }

            object[] parameters = new object[] { text };

            this.OnShowTextureTypeMismatchExecutable.Execute(parameters);
        }

        /// <summary>
        /// テクスチャのパスが、エミッタセットの探索範囲外であることを通知します。
        /// </summary>
        /// <param name="text">エラーの説明</param>
        /// <returns>DialogResult</returns>
        public DialogResult ShowWarningOnLoadTexture(string text)
        {
            if (this.OnShowWarningOnLoadTextureExecutable == null)
            {
                return DialogResult.OK;
            }

            object[] parameters = new object[] { null, text };
            this.OnShowWarningOnLoadTextureExecutable.Execute(parameters);

            return (DialogResult)parameters[0];
        }

        /// <summary>
        /// プリミティグのパスが、エミッタセットの探索範囲外であることを通知します.
        /// </summary>
        /// <param name="text">エラーの説明</param>
        /// <returns>DialogResult</returns>
        public DialogResult ShowWarningOnLoadPrimitive(string text)
        {
            if (this.OnShowWarningOnLoadPrimitiveExecutable == null)
            {
                return DialogResult.OK;
            }

            object[] parameters = new object[] { null, text };
            this.OnShowWarningOnLoadPrimitiveExecutable.Execute(parameters);

            return (DialogResult)parameters[0];
        }

        /// <summary>
        /// テクスチャが設定されておらず保存をキャンセルするかどうか選択するダイアログを表示します。
        /// </summary>
        /// <param name="text">エラーの説明</param>
        /// <returns>ユーザーの選択を返します</returns>
        /// <param name="showCancel">キャンセルボタンの表示On/Off</param>
        public DialogResult ShowWarningTextureIsNotAssigned(string text, bool showCancel)
        {
            if (this.OnShowWarningTextureIsNotAssignedExecutable == null)
            {
                return DialogResult.Cancel;
            }

            object[] parameters = new object[] { null, text, showCancel };

            this.OnShowWarningTextureIsNotAssignedExecutable.Execute(parameters);

            return (DialogResult)parameters[0];
        }

        /// <summary>
        /// アセットファイルの上書き確認をするダイアログを表示します。
        /// </summary>
        /// <param name="result">ダイアログに表示する情報</param>
        /// <returns>ユーザーの選択を返します。</returns>
        public DialogResult PromptOverWriteAsset(FileReferenceCheckResult result)
        {
            if (this.OnPromptOverWriteAssetExecutable == null)
            {
                return DialogResult.Cancel;
            }

            object[] parameters = new object[]
            {
                null,
                result.SrcFilePath,
                result.SrcFileLastWriteTime,
                result.DstFilePath,
                result.DstFileLastWriteTime
            };

            // ダイアログを表示
            this.OnPromptOverWriteAssetExecutable.Execute(parameters);

            return (DialogResult)parameters[0];
        }

        /// <summary>
        /// アセットファイルのコピーダイアログを表示します。
        /// </summary>
        /// <param name="textureCopyDestinationPath">テクスチャのコピー先</param>
        /// <param name="primitiveCopyDestinationPath">プリミティブのコピー先</param>
        /// <param name="combinerCopyDestinationPath">コンバイナのコピー先</param>
        /// <param name="filePath">エミッタセットの保存先</param>
        /// <param name="srcPaths">コピー元のファイルリスト</param>
        /// <param name="texturePaths">テクスチャファイルのコピー先リスト</param>
        /// <param name="primitivePaths">プリミティブファイルのコピー先リスト</param>
        /// <param name="combinerPaths">コンバイナファイルのコピー先リスト</param>
        /// <returns>ユーザーの選択を返します</returns>
        public DialogResult ShowCopyAssetDialog(
            out string textureCopyDestinationPath,
            out string primitiveCopyDestinationPath,
            out string combinerCopyDestinationPath,
            string filePath,
            List<string> srcPaths,
            List<string> texturePaths,
            List<string> primitivePaths,
            List<string> combinerPaths)
        {
            if (this.OnShowCopyAssetDialogExecutable == null)
            {
                textureCopyDestinationPath = null;
                primitiveCopyDestinationPath = null;
                combinerCopyDestinationPath = null;

                return DialogResult.Cancel;
            }

            object[] parameters = new object[]
            {
                null,
                null,
                null,
                null,
                filePath,
                srcPaths,
                texturePaths,
                primitivePaths,
                combinerPaths,
            };

            // ダイアログを表示
            this.OnShowCopyAssetDialogExecutable.Execute(parameters);

            textureCopyDestinationPath = (string)parameters[1];
            primitiveCopyDestinationPath = (string)parameters[2];
            combinerCopyDestinationPath = (string)parameters[3];

            return (DialogResult)parameters[0];
        }

        /// <summary>
        /// アセットファイルの参照エラーを表示します。
        /// </summary>
        /// <param name="text">エラーの説明</param>
        /// <param name="caption">キャプション</param>
        /// <returns>ユーザーの選択を返します</returns>
        public DialogResult ShowAssetPathProblem(string text, string caption)
        {
            if (this.OnShowAssetPathProblemExecutable == null)
            {
                return DialogResult.Cancel;
            }

            object[] parameters = new object[] { null, text, caption };

            this.OnShowAssetPathProblemExecutable.Execute(parameters);

            return (DialogResult)parameters[0];
        }

        /// <summary>
        /// 無効な終了処理タイプの警告を表示します。
        /// </summary>
        /// <param name="text">エラーの説明</param>
        /// <param name="caption">キャプション</param>
        /// <param name="showCancel">キャンセルボタンの表示On/Off</param>
        /// <returns>ユーザーの選択を返します</returns>
        public DialogResult ShowInvalidEndingType(string text, string caption, bool showCancel)
        {
            if (this.OnShowInvalidEndingTypeExecutable == null)
            {
                return DialogResult.Cancel;
            }

            object[] parameters = new object[] { null, text, caption, showCancel };

            this.OnShowInvalidEndingTypeExecutable.Execute(parameters);

            return (DialogResult)parameters[0];
        }

        #endregion

        #region Wrapper Function - 不正描画パス

        /// <summary>
        /// 不正描画パスダイアログの表示設定をリセットします。
        /// </summary>
        public void ResetInvalidDrawPathDialogInfo()
        {
            Logger.Log(LogLevels.Information, "WorkspaceRootViewModel::ResetInvalidDrawPathDialogInfo(): Reset.");

            // 次回は必ずダイアログを表示するようにする
            this.applyAllInvalidDrawPath = false;

            // 保存結果をリセット
            this.savedResultInvalidDrawPathDialog = DialogResult.No;
        }

        /// <summary>
        /// 不正描画パスダイアログを表示します。
        /// </summary>
        /// <param name="emitterSetName">エミッタセット名</param>
        /// <param name="emitterName">エミッタ名</param>
        /// <param name="showCancel">キャンセルボタンの表示On/Off</param>
        /// <returns>ユーザーの選択を返します</returns>
        public DialogResult ShowInvalidDrawPathDialog(string emitterSetName, string emitterName, bool showCancel)
        {
            if (this.OnShowInvalidDrawPathDialogExecutable == null)
            {
                this.applyAllInvalidDrawPath = false;
                return DialogResult.Cancel;
            }

            // 前回の結果を適用するように設定されたときは保存された結果を返す
            if (this.applyAllInvalidDrawPath)
            {
                return this.savedResultInvalidDrawPathDialog;
            }

            string targetName = emitterSetName + " : " + emitterName;

            // ダイアログを表示
            object[] parameters = new object[] { null, null, targetName, showCancel };
            this.OnShowInvalidDrawPathDialogExecutable.Execute(parameters);

            DialogResult result = (DialogResult)parameters[0];

            // 押されたボタンがYes/Noであれば結果を記録する
            if (result != DialogResult.Cancel)
            {
                this.applyAllInvalidDrawPath = (bool)parameters[1];
                if (this.applyAllInvalidDrawPath)
                {
                    this.savedResultInvalidDrawPathDialog = (DialogResult)parameters[0];
                }
            }

            return result;
        }

        #endregion

        #region Wrapper Function - ビューワー処理

        /// <summary>
        /// 接続ターゲットが見つからなかったことを通知するダイアログを表示します。
        /// </summary>
        public void ShowTargetManagerNotFoundDialog()
        {
            if (this.OnShowErrorDialogExecutable == null)
            {
                return;
            }

            object[] parameters = new string[] {
                Resources.ErrorTargetManagerNotFound,
                Resources.ErrorCaption
            };

            this.OnShowErrorDialogExecutable.Execute(parameters);
        }

        /// <summary>
        /// ビューワー接続が失敗したことを通知するダイアログを表示します。
        /// </summary>
        public void ShowFailedViewerConnectionDialog()
        {
            if (this.OnShowErrorDialogExecutable == null)
            {
                return;
            }

            object[] parameters = new string[] {
                Resources.ErrorFailedViewerConnection,
                Resources.ErrorCaption
            };

            this.OnShowErrorDialogExecutable.Execute(parameters);
        }

        #endregion
    }
}
