﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Linq;
using System.Text;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// 階層構造を持つビューモデル.
    /// </summary>
    public abstract class HierarchyViewModel : ExportableViewModel, IHierarchyObject
    {
        /// <summary>
        /// The constructor
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">dataModel</param>
        protected HierarchyViewModel(HierarchyViewModel parent, DataModelBase dataModel)
            : base(dataModel)
        {
            this.parent = parent;
            this.Children = new TypeOrderedObservableCollection<IHierarchyObject>();

            this.Children.CollectionChanged += this.OnChildrenCollectionChanged;
        }

        /// <summary>
        /// 子ビューモデルが追加されたときに発生します。
        /// </summary>
        public event EventHandler<IHierarchyObject> ChildrenAdded = null;

        /// <summary>
        /// 子ビューモデルが削除されたときに発生します。
        /// </summary>
        public event EventHandler<IHierarchyObject> ChildrenRemoved = null;

        /// <summary>
        /// 親プロパティの実装.
        /// </summary>
        public IHierarchyObject Parent
        {
            get {
                return this.parent;
            }

            protected set {
                if (value != this.parent)
                {
                    this.parent = value;
                    this.OnPropertyChanged(() => this.Parent);
                }
            }
        }

        /// <summary>
        /// 子プロパティの実装.
        /// </summary>
        public ObservableCollection<IHierarchyObject> Children { get; protected set; }

        /// <summary>
        /// 名前を設定できるかどうか取得します.
        /// </summary>
        public virtual bool CanSetName
        {
            get { return false; }
        }

        /// <summary>
        /// 子ビューモデルが追加されたときのイベントを発行します.
        /// </summary>
        /// <param name="child">子ビューモデル</param>
        public void TriggerChildrenAddedEvent(IHierarchyObject child)
        {
            if (this.ChildrenAdded != null)
            {
                this.ChildrenAdded(null, child);
            }
        }

        /// <summary>
        /// 子ビューモデルが削除されたときのイベントを発行します.
        /// </summary>
        /// <param name="child">子ビューモデル</param>
        public void TriggerChildrenRemovedEvent(IHierarchyObject child)
        {
            if (this.ChildrenRemoved != null)
            {
                this.ChildrenRemoved(null, child);
            }
        }

        /// <summary>
        /// Disposes the instance.
        /// </summary>
        public override void Dispose()
        {
            foreach (IHierarchyObject child in this.Children)
            {
                var vm = child as ViewModelBase;
                if (vm != null)
                {
                    vm.Dispose();
                }
            }

            base.Dispose();
        }

        /// <summary>
        /// 名前が正しいかチェックします.
        /// </summary>
        /// <param name="name">名前</param>
        /// <returns>名前が正しいときtrueを返します.</returns>
        public virtual bool IsNameValid(string name)
        {
            return false;
        }

        /// <summary>
        /// 名前を設定します.
        /// </summary>
        /// <param name="name">名前</param>
        public virtual void SetName(string name)
        {
            return;
        }

        /// <summary>
        /// Resend PropertyChanged notification for all properties.
        /// This is required when the data model changes independently from the view model.
        /// </summary>
        public override void FirePropertyChanges()
        {
            base.FirePropertyChanges();
            this.FirePropertyChangeEventForChildren();
        }

        /// <summary>
        /// Set child view model to the specified member and the child list.
        /// </summary>
        /// <typeparam name="T">The type of the child view model.</typeparam>
        /// <param name="member">The member variable.</param>
        /// <param name="child">The child view model to set.</param>
        public void SetChildViewModel<T>(ref T member, T child) where T : HierarchyViewModel
        {
            // First remove the original child view model from the child list.
            if (member != null)
            {
                this.Children.Remove(member);
            }

            // Set the child to the member.
            member = child;

            // Add the view model to the child list.
            if (member != null)
            {
                this.Children.Add(member);
            }
        }

        /// <summary>
        /// Add child view model to the child list.
        /// </summary>
        /// <typeparam name="T">The type of the child view model.</typeparam>
        /// <param name="child">The child view model to set.</param>
        /// <returns>The child view model.</returns>
        public T AddChildViewModel<T>(T child) where T : HierarchyViewModel
        {
            // Add the view model to the child list.
            if (child != null && this.Children.IndexOf(child) < 0)
            {
                this.Children.Add(child);
            }

            return child;
        }

        /// <summary>
        /// 指定したViewModelTypeの子プロパティを取得します。
        /// </summary>
        /// <typeparam name="ViewModelType">ViewModelType.</typeparam>
        /// <returns>指定したViewModelTypeの子プロパティを返します。</returns>
        public ViewModelType GetChild<ViewModelType>()
        {
            return this.Children.OfType<ViewModelType>().FirstOrDefault();
        }

        /// <summary>
        /// 指定したViewModelTypeの子プロパティのリストを取得します。
        /// </summary>
        /// <typeparam name="ViewModelType">ViewModelType.</typeparam>
        /// <returns>指定したViewModelTypeの子プロパティのリストを返します。</returns>
        public IEnumerable<ViewModelType> GetChildren<ViewModelType>()
        {
            return this.Children.OfType<ViewModelType>();
        }

        /// <summary>
        /// Update child view models with the current data model.
        /// This method is usually called when data model is modified, thus some child
        /// view models might need to be created or removed according to the data model.
        /// </summary>
        public virtual void UpdateChildViewModels()
        {
            foreach (IHierarchyObject child in this.Children)
            {
                var childViewModel = child as HierarchyViewModel;
                if (childViewModel != null)
                {
                    childViewModel.UpdateChildViewModels();
                }
            }
        }

        /// <summary>
        /// Call FirePropertyChanges method for children.
        /// </summary>
        protected void FirePropertyChangeEventForChildren()
        {
            foreach (var child in this.Children.OfType<HierarchyViewModel>())
            {
                child.FirePropertyChanges();
            }
        }

        /// <summary>
        /// Handle CollectionChanged event for the children collection.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The event arguments.</param>
        private void OnChildrenCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            this.OnPropertyChanged("Children");

            // The original items.
            var oldItems = e.OldItems == null ?
                Enumerable.Empty<IHierarchyObject>() :
                e.OldItems.Cast<IHierarchyObject>();

            // The new items.
            var newItems = e.NewItems == null ?
                Enumerable.Empty<IHierarchyObject>() :
                e.NewItems.Cast<IHierarchyObject>();

            if (e.Action == NotifyCollectionChangedAction.Add)
            {
                IEnumerable<IHierarchyObject> addedItems = newItems.Except(oldItems);
                this.ProcessAddedChildren(addedItems);
            }
            else if (e.Action == NotifyCollectionChangedAction.Remove)
            {
                IEnumerable<IHierarchyObject> removedItems = oldItems.Except(newItems);
                this.ProcessRemovedChildren(removedItems);
            }
            else if (e.Action == NotifyCollectionChangedAction.Replace)
            {
                IEnumerable<IHierarchyObject> addedItems = newItems.Except(oldItems);
                IEnumerable<IHierarchyObject> removedItems = oldItems.Except(newItems);

                this.ProcessRemovedChildren(removedItems);
                this.ProcessAddedChildren(addedItems);
            }
        }

        /// <summary>
        /// Process the added children.
        /// </summary>
        /// <param name="addedItems">The added children.</param>
        private void ProcessAddedChildren(IEnumerable<IHierarchyObject> addedItems)
        {
            foreach (var item in addedItems)
            {
                var vm = item as ViewModelBase;
                if (vm != null && vm.Proxy != null)
                {
                    DataModelBase dm = vm.Proxy.DataModel;
                    if (dm != null)
                    {
                        BinaryConversionInfoManager.ReportDataModelCreated(dm);
                    }
                }

                // 子ビューモデル追加イベントを発行
                var child = item as IHierarchyObject;
                if (child != null)
                {
                    this.TriggerChildrenAddedEvent(child);
                }
            }
        }

        /// <summary>
        /// Process the removed children.
        /// </summary>
        /// <param name="removedItems">The removed children.</param>
        private void ProcessRemovedChildren(IEnumerable<IHierarchyObject> removedItems)
        {
            foreach (var item in removedItems)
            {
                var vm = item as ViewModelBase;
                if (vm != null && vm.Proxy != null)
                {
                    DataModelBase dm = vm.Proxy.DataModel;
                    if (dm != null)
                    {
                        BinaryConversionInfoManager.ReportDataModelDestroyed(dm);
                    }
                }

                // 子ビューモデル削除イベントを発行
                var child = item as IHierarchyObject;
                if (child != null)
                {
                    this.TriggerChildrenRemovedEvent(child);
                }
            }
        }

        /// <summary>
        /// 親プロパティ
        /// </summary>
        private IHierarchyObject parent;
    }

    /// <summary>
    /// The suppress node type ordering.
    /// </summary>
    public class SuppressNodeTypeOrdering : IDisposable
    {
        /// <summary>
        /// The type ordered.
        /// </summary>
        private readonly TypeOrderedObservableCollection<IHierarchyObject> typeOrdered = null;

        /// <summary>
        /// Initializes a new instance of the <see cref="SuppressNodeTypeOrdering"/> class.
        /// </summary>
        /// <param name="viewModel">
        /// The view model.
        /// </param>
        public SuppressNodeTypeOrdering(HierarchyViewModel viewModel)
        {
            this.typeOrdered = viewModel.Children
                as TypeOrderedObservableCollection<IHierarchyObject>;

            if (this.typeOrdered != null)
            {
                this.typeOrdered.SuspendTypeOrdering();
            }
        }

        /// <summary>
        /// The dispose.
        /// </summary>
        public void Dispose()
        {
            if (this.typeOrdered != null)
            {
                this.typeOrdered.ResumeTypeOrdering();
            }
        }
    }
}
