﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.Serializer;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Extensions;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.DataModelLogic.BinaryData;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.FileMonitor;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.Properties;
using EffectMaker.UILogic.ViewModels.IO;
using Microsoft.CSharp.RuntimeBinder;
using Clipboard = System.Windows.Clipboard;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterSetData.
    /// </summary>
    [CommandStackOwner]
    public class EmitterSetViewModel : WorkspaceNodeViewModelBase<EmitterSetData>, IFileOwner, IEmitterOwnerViewModel, IModificationFlagOwner
    {
        /// <summary>
        /// アセットファイルに変更があったときのデリゲートです。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="path">アセットファイルパス</param>
        public delegate void AssetFileDelegate(object sender, string path);

        /// <summary>
        /// ファイルモニターでテクスチャファイルの変更を補足したとき、
        /// 代表のビューモデルでリロード処理を行った後に発生します。
        /// </summary>
        public AssetFileDelegate TextureFileChanged { get; set; }

        /// <summary>
        /// ファイルモニターでプリミティブファイルの変更を補足したとき、
        /// 代表のビューモデルでリロード処理を行った後に発生します。
        /// </summary>
        public AssetFileDelegate PrimitiveFileChanged { get; set; }

        /// <summary>
        /// A regular expression that detects the number of an Emitter in its name.
        /// </summary>
        private static Regex emitterNameRegex =
            new Regex(@"^Emitter(?<num>\d+)$", RegexOptions.Compiled);

        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "Linked",
        };

        /// <summary>
        /// リンク状態です。
        /// </summary>
        private bool isLinked;

        /// <summary>
        /// ファイルモニタをViewModelごとに持つことにした
        /// </summary>
        private FileMonitorManager monitorManager =
            new FileMonitorManager(Application.OpenForms["MainForm"], true);

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterSetViewModel(HierarchyViewModel parent, EmitterSetData dataModel)
            : base(parent, dataModel)
        {
            var typeOrdered = Children as TypeOrderedObservableCollection<IHierarchyObject>;

            // チャイルドノードの並び順を設定
            if (typeOrdered != null)
            {
                typeOrdered.TypeOrder = new Type[]
                {
                    typeof(EmitterViewModel),
                    typeof(PreviewViewModel),
                };
                typeOrdered.Sort();
            }

            this.EmitterSetBasicViewModel =
                new EmitterSetBasicViewModel(this, dataModel.EmitterSetBasicData);
            this.Children.Add(this.EmitterSetBasicViewModel);

            this.EmitterSetUserViewModel =
                new EmitterSetUserViewModel(this, dataModel.EmitterSetUserData);
            this.Children.Add(this.EmitterSetUserViewModel);

            // エミッタのビューモデルを作成
            foreach (EmitterData emitterData in dataModel.EmitterList)
            {
                var emitterViewModel = new EmitterViewModel(this, emitterData);
                this.Children.Add(emitterViewModel);
            }

            // プレビューのビューモデルを作成
            foreach (PreviewData previewData in dataModel.PreviewList)
            {
                var previewViewModel = new PreviewViewModel(this, previewData);
                this.Children.Add(previewViewModel);
            }

            // Add a listener to monitor child list modification.
            this.Children.CollectionChanged += this.OnChildrenCollectionChanged;

            // アセットリロード時のイベントを登録
            BusinessLogic.Manager.TextureManager.Instance.FileReloaded += this.OnTextureFileReloaded;
            BusinessLogic.Manager.PrimitiveManager.Instance.FileReloaded += this.OnPrimitiveFileReloaded;

            this.NodeOpenExlorerExecutable = new AnonymousExecutable(this.OnOpenExplorer);
            this.NodePasteEmitterExecutable = new AnonymousExecutable(this.OnNodePasteEmitter);
            this.EvaluateMenuItemsExecutable = new AnonymousExecutable(this.OnEvaluateMenuItems);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// Get or set the flag indicating whether the data model should be displayed/rendered.
        /// </summary>
        public bool Displayed
        {
            get
            {
                return this.GetDataModelValue(() => this.Displayed);
            }

            set
            {
                bool originalValue = this.GetDataModelValue(() => this.Displayed);
                if (originalValue != value)
                {
                    var binder = new EffectMakerSetMemberBinder("Displayed", false, false);
                    this.TrySetMember(binder, value);

                    // Send visibility message to the viewer.
                    ViewerController.Instance.SendVisibility(this.DataModel);
                }
            }
        }

        /// <summary>
        /// リンク状態を取得または設定します。
        /// </summary>
        public bool Linked
        {
            get { return this.isLinked; }
            set { this.SetValue(ref this.isLinked, value); }
        }

        /// <summary>
        /// メニューアイテム(エクスプローラで開く)のExecutable.
        /// </summary>
        public IExecutable NodeOpenExlorerExecutable { get; set; }

        /// <summary>
        /// Gets the Executable instance to run when the Paste Node menu is clicked.
        /// </summary>
        public IExecutable NodePasteEmitterExecutable { get; private set; }

        /// <summary>
        /// コンテキストメニューアイテムの状態を評価するExecutable.
        /// </summary>
        public IExecutable EvaluateMenuItemsExecutable { get; set; }

        /// <summary>
        /// The default property page view model to use
        /// on the first time the workspace node is selected.
        /// </summary>
        public override HierarchyViewModel DefaultPropertyPageViewModel
        {
            get { return this.EmitterSetBasicViewModel; }
        }

        /// <summary>
        /// Get the selected item.
        /// </summary>
        public HierarchyViewModel SelectedItem { get; set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// Get the view model for emitter set basic information page.
        /// </summary>
        public EmitterSetBasicViewModel EmitterSetBasicViewModel { get; private set; }

        /// <summary>
        /// Get the view model for emitter set user information page.
        /// </summary>
        public EmitterSetUserViewModel EmitterSetUserViewModel { get; private set; }

        /// <summary>
        /// Perform a deep enumeration of all the emitters, including
        /// all the child emitters.
        /// </summary>
        public IEnumerable<EmitterViewModel> Emitters
        {
            get
            {
                IEnumerable<EmitterViewModel> emitters =
                    from item in this.Children
                    where item is EmitterViewModel
                    select (EmitterViewModel)item;

                foreach (EmitterViewModel emitter in emitters)
                {
                    yield return emitter;

                    foreach (EmitterViewModel childEmitter in emitter.Emitters)
                    {
                        yield return childEmitter;
                    }
                }
            }
        }

        /// <summary>
        /// １番上のエミッタだけのenumerableを返す
        /// </summary>
        public IEnumerable<EmitterViewModel> TopEmitters
        {
            get
            {
                IEnumerable<EmitterViewModel> emitters =
                    from item in this.Children
                    where item is EmitterViewModel
                    select (EmitterViewModel)item;

                foreach (EmitterViewModel emitter in emitters)
                {
                    yield return emitter;
                }
            }
        }

        /// <summary>
        /// 名前を設定できるかどうか取得します.
        /// </summary>
        public override bool CanSetName
        {
            get { return true; }
        }

        /// <summary>
        /// ファイル名(拡張子を含まない)を設定または取得します。
        /// </summary>
        public string FileName
        {
            get
            {
                return this.DataModel.Name;
            }

            set
            {
                this.DataModel.Name = value;
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// ファイルパス(ファイル名を含まない)を設定または取得します。
        /// </summary>
        public string FilePath
        {
            get
            {
                return this.GetDataModelValue(() => this.FilePath);
            }

            set
            {
                // ファイルパスの変更は履歴に積まない
                var binder = new EffectMakerSetMemberBinder("FilePath", false, false);
                this.TrySetMember(binder, value);

                // アセットパスをロケート
                AssetsManager.LocateAssetPaths(this);
            }
        }

        /// <summary>
        /// ファイルが存在するかチェックします。
        /// </summary>
        public bool IsFileExist
        {
            get
            {
                if (string.IsNullOrEmpty(this.FileName))
                {
                    return false;
                }

                string path = this.FilePath + @"\" + this.FileName + IOConstants.EmitterSetFileExtension;
                if (File.Exists(path) == false)
                {
                    return false;
                }

                return true;
            }
        }

        /// <summary>
        /// エミッタが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewEmitter
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveEmitter(this); }
        }

        /// <summary>
        /// エミッタが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewEmitterSet
        {
            // このプロパティは選択ノード非依存なので使用可
            get { return WorkspaceRootViewModel.Instance.CanCreateNewEmitterSet; }
        }

        /// <summary>
        /// エミッタが貼り付け可能かチェックする。
        /// </summary>
        public bool CanPasteEmitter
        {
            get { return this.CanPasteEmitterNode && (this.Emitters.Count() + 1 <= IOConstants.MaxEmitterCount); }
        }

        /// <summary>
        /// エミッタノードのペーストが可能かチェックする.
        /// </summary>
        public bool CanPasteEmitterNode
        {
            get
            {
                // EffectConverter.exeから実行したときはThreadStateExceptionになるためスキップする
                if (OptionStore.RuntimeOptions.IsCommandLineMode)
                {
                    return false;
                }

                NodeClipboardData value = Clipboard.GetData(NodeClipboardData.CustomerFormatName) as NodeClipboardData;;

                if (value == null)
                {
                    return false;
                }

                return typeof(EmitterData).FullName == value.ClassFullName;
            }
        }

        /// <summary>
        /// ファイル監視に伴うリロードをエミッタセット単位で管理する
        /// </summary>
        public FileMonitorManager FileMonitor
        {
            get { return this.monitorManager; }
        }

        /// <summary>
        /// Update child view models with the current data model.
        /// This method is usually called when data model is modified, thus some child
        /// view models might need to be created or removed according to the data model.
        /// </summary>
        public override void UpdateChildViewModels()
        {
            // Remove old emitter view models.
            var viewModelsToRemove = this.Children.Where(ch => ch is EmitterViewModel).ToArray();
            viewModelsToRemove.ForEach(vm => this.Children.Remove(vm));

            // Add emitter view models for each emitter data.
            foreach (EmitterData emitterData in this.DataModel.EmitterList)
            {
                this.AddChildViewModel(new EmitterViewModel(this, emitterData));
            }

            // Remove old preview view models.
            viewModelsToRemove = this.Children.Where(ch => ch is PreviewViewModel).ToArray();
            viewModelsToRemove.ForEach(vm => this.Children.Remove(vm));

            // Add preview view models for each preview data.
            foreach (PreviewData previewData in this.DataModel.PreviewList)
            {
                this.AddChildViewModel(new PreviewViewModel(this, previewData));
            }

            // Although all the emitter and preview view models are up to date
            // but there are also other child view models that need to be updated
            // so we still have to call base.UpdateChildViewModels().
            base.UpdateChildViewModels();
        }

        /// <summary>
        /// 名前が正しいかチェックします.
        /// </summary>
        /// <param name="name">名前</param>
        /// <returns>名前が正しいときtrueを返します.</returns>
        public override bool IsNameValid(string name)
        {
            // 名前に不正な文字が入ってないかチェック
            Regex validPattern = new Regex(Resources.RegexpNodeNameValidation);

            if (validPattern.IsMatch(name) == false)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowWarningInvalidName();
                return false;
            }

            // 親のワークスペースを取得
            WorkspaceViewModel workspaceViewModel = ViewModelBase.GetParent<WorkspaceViewModel>(this);
            if (workspaceViewModel == null)
            {
                return false;
            }

            // 同じワークスペースに属するエミッタセットを取得
            IEnumerable<EmitterSetViewModel> emitterSetList =
                workspaceViewModel.GetChildren<EmitterSetViewModel>();
            if (emitterSetList == null)
            {
                return true;
            }

            // エミッタセットの名前が被らないかチェック
            foreach (EmitterSetViewModel emitterSet in emitterSetList)
            {
                if (emitterSet != this && emitterSet.FileName == name)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowWarningEmitterSetNameAlreadyUsed(name);
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// 名前を設定します.
        /// </summary>
        /// <param name="name">名前</param>
        public override void SetName(string name)
        {
            // リンク状態をリセット
            if (name != this.FileName && this.Linked)
            {
                this.Linked = false;
            }

            // 名前を設定
            this.FileName = name;
            this.OnPropertyChanged(() => this.DataModel.Name);
        }

        /// <summary>
        /// Add the view model of field.
        /// </summary>
        /// <param name="fieldViewModel">フィールドビューモデル.</param>
        public void AddFieldViewModel(FieldViewModel fieldViewModel)
        {
            // プレビューノードの一つ上にエミッタノードが表示されるように
            // インデックスを計算する.
            int index = 0;

            foreach (HierarchyViewModel vm in this.Children)
            {
                if (vm is PreviewViewModel)
                {
                    break;
                }
                else
                {
                    ++index;
                }
            }

            this.Children.Insert(index, fieldViewModel);
        }

        #region Function - IEmitterOwnerViewModel

        /// <summary>
        /// エミッタを作成します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="emitterName">エミッタ名</param>
        /// <param name="presetEmitter">プリセットとなるエミッタデータ(nullなら完全新規作成)</param>
        /// <returns>エミッタビューモデルを返します。</returns>
        public EmitterViewModel CreateNewEmitter(string emitterName, EmitterData presetEmitter)
        {
            using (new DataModelPropertyModifiedEventBlock())
            {
                var proxy = this.Proxy as EmitterSetDataProxy;
                if (proxy == null)
                {
                    return null;
                }

                EmitterData emitterData = null;

                // Create the emitter data.
                if (presetEmitter == null)
                {
                    emitterData = proxy.CreateEmitter(emitterName);
                    if (emitterData == null)
                    {
                        return null;
                    }
                }
                else
                {
                    emitterData = presetEmitter;
                    emitterData.Name = emitterName;
                    proxy.AddEmitter(presetEmitter);
                }

                // Create the emitter view model.
                EmitterViewModel emitterViewModel = new EmitterViewModel(this, emitterData);

                return emitterViewModel;
            }
        }

        /// <summary>
        /// エミッタを追加します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="emitterViewModel">エミッタビューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool AddEmitter(EmitterViewModel emitterViewModel)
        {
            var proxy = this.Proxy as EmitterSetDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // Add the emitter data model.
            proxy.AddEmitter(emitterViewModel.DataModel);

            // Add the emitter view model.
            emitterViewModel.EntryIntoEmitterSet(this);

            return true;
        }

        /// <summary>
        /// エミッタを削除します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="emitterViewModel">エミッタビューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool RemoveEmitter(EmitterViewModel emitterViewModel)
        {
            var proxy = this.Proxy as EmitterSetDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // Remove the emitter data model.
            proxy.RemoveEmitter(emitterViewModel.DataModel);

            // Remove the emitter view model.
            emitterViewModel.RemoveFromParent();

            return true;
        }

        /// <summary>
        /// エミッタのインデックスを取得します。
        /// </summary>
        /// <param name="emitterViewModel">エミッタビューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        /// <param name="dataModelIndex">データモデルのインデックス</param>
        public void GetEmitterIndex(
            EmitterViewModel emitterViewModel,
            out int viewModelIndex,
            out int dataModelIndex)
        {
            dataModelIndex = -1;
            viewModelIndex = -1;

            var proxy = this.Proxy as EmitterSetDataProxy;
            if (proxy == null)
            {
                return;
            }

            // データモデルのインデックスを取得
            dataModelIndex = this.Proxy.EmitterList.IndexOf(emitterViewModel.DataModel);

            // ビューモデルのインデックスを取得
            viewModelIndex = this.Children.IndexOf(emitterViewModel);
        }

        /// <summary>
        /// エミッタを指定インデックスに追加します。
        /// </summary>
        /// <param name="emitterViewModel">エミッタビューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        /// <param name="dataModelIndex">データモデルのインデックス</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool InsertEmitter(
            EmitterViewModel emitterViewModel,
            int viewModelIndex,
            int dataModelIndex)
        {
            var proxy = this.Proxy as EmitterSetDataProxy;
            if (proxy == null)
            {
                return false;
            }

            using (new SuppressNodeTypeOrdering(this))
            {
                // データモデルを追加
                proxy.InsertChildDataModel(dataModelIndex, emitterViewModel.DataModel);

                // ビューモデルを追加
                emitterViewModel.EntryIntoEmitterSet(this, viewModelIndex);
            }

            return true;
        }

        #endregion

        #region Function - プレビュー管理

       /// <summary>
        /// プレビューを作成します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="previewName">プレビュー名</param>
        /// <returns>プレビュービューモデルを返します。</returns>
        public PreviewViewModel CreateNewPreview(string previewName)
        {
            using (new DataModelPropertyModifiedEventBlock())
            {
                var proxy = this.Proxy as EmitterSetDataProxy;
                if (proxy == null)
                {
                    return null;
                }

                // プレビューデータモデルを作成.
                PreviewData previewData = proxy.CreatePreview(previewName);
                if (previewData == null)
                {
                    return null;
                }

                // プレビュービューモデルを作成.
                PreviewViewModel previewViewModel = new PreviewViewModel(this, previewData);

                return previewViewModel;
            }
        }

        /// <summary>
        /// プレビューを追加します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="previewViewModel">プレビュービューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool AddPreview(PreviewViewModel previewViewModel)
        {
            var proxy = this.Proxy as EmitterSetDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // プレビューデータモデルを追加.
            proxy.AddPreview(previewViewModel.DataModel);

            // プレビュービューモデルを追加.
            ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;
            this.Children.Add(previewViewModel);
            ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;

            return true;
        }

        /// <summary>
        /// プレビューを削除します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="previewViewModel">プレビュービューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool RemovePreview(PreviewViewModel previewViewModel)
        {
            var proxy = this.Proxy as EmitterSetDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // プレビュービューモデルを追加.
            proxy.RemovePreview(previewViewModel.DataModel);

            // プレビュービューモデルを削除.
            ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;
            this.Children.Remove(previewViewModel);
            ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;

            return true;
        }

        /// <summary>
        /// プレビューのインデックスを取得します。
        /// </summary>
        /// <param name="previewViewModel">プレビュービューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        /// <param name="dataModelIndex">データモデルのインデックス</param>
        public void GetPreviewIndex(
            PreviewViewModel previewViewModel,
            out int viewModelIndex,
            out int dataModelIndex)
        {
            dataModelIndex = -1;
            viewModelIndex = -1;

            var proxy = this.Proxy as EmitterSetDataProxy;
            if (proxy == null)
            {
                return;
            }

            // データモデルのインデックスを取得
            dataModelIndex = this.Proxy.PreviewList.IndexOf(previewViewModel.DataModel);

            // ビューモデルのインデックスを取得
            viewModelIndex = this.Children.IndexOf(previewViewModel);
        }

        /// <summary>
        /// プレビューを指定インデックスに追加します。
        /// </summary>
        /// <param name="previewViewModel">プレビュービューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        /// <param name="dataModelIndex">データモデルのインデックス</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool InsertPreview(
            PreviewViewModel previewViewModel,
            int viewModelIndex,
            int dataModelIndex)
        {
            var proxy = this.Proxy as EmitterSetDataProxy;
            if (proxy == null)
            {
                return false;
            }

            using (new SuppressNodeTypeOrdering(this))
            {
                // データモデルを追加
                proxy.InsertChildDataModel(dataModelIndex, previewViewModel.DataModel);

                // ビューモデルを追加
                ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;
                this.Children.Insert(viewModelIndex, previewViewModel);
                ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;
            }

            return true;
        }

        #endregion

        /// <summary>
        /// プレビュービューモデルを追加します。
        /// </summary>
        /// <param name="previewViewModel">プレビュービューモデル</param>
        public void AddPreviewViewModel(PreviewViewModel previewViewModel)
        {
            ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;
            this.Children.Add(previewViewModel);
            ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;
        }

        /// <summary>
        /// プレビュービューモデルを削除します。
        /// </summary>
        /// <param name="previewViewModel">プレビュービューモデル</param>
        public void RemovePreviewViewModel(PreviewViewModel previewViewModel)
        {
            ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;
            this.Children.Remove(previewViewModel);
            ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;
        }

        /// <summary>
        /// 名前が一致するプレビューのビューモデルを取得します。
        /// </summary>
        /// <param name="name">プレビューの名前</param>
        /// <returns>プレビュービューモデルを返します。</returns>
        public PreviewViewModel GetPreviewViewModel(string name)
        {
            foreach (IHierarchyObject child in this.Children)
            {
                var previewViewModel = child as PreviewViewModel;

                if (previewViewModel == null)
                {
                    continue;
                }

                if (previewViewModel.DataModel.Name == name)
                {
                    return previewViewModel;
                }
            }

            return null;
        }

        /// <summary>
        /// Find unique Emitter names, with a unique numbers.
        /// </summary>
        /// <param name="count">The amount of unique name desired.</param>
        /// <returns>Returns an array of unique Emitter name.</returns>
        public string[] FindUniqueEmitterNames(int count)
        {
            if (count <= 0)
            {
                return new string[0];
            }

            var names = new HashSet<string>();
            this.GatherAllNodeNames(this, names);

            var numbers = new List<int>();
            foreach (string name in names)
            {
                Match m = emitterNameRegex.Match(name);
                if (m.Success)
                {
                    int value;
                    if (int.TryParse(m.Groups["num"].Value, out value))
                    {
                        numbers.Add(value);
                    }
                }
            }

            var newNames = new List<string>();
            for (int i = 0; i < count; i++)
            {
                numbers.Sort();
                int newNumber = GetUniqueAvailableNumber(numbers);
                newNames.Add(string.Format("Emitter{0}", newNumber));
                numbers.Add(newNumber);
            }

            return newNames.ToArray();
        }

        /// <summary>
        /// エミッタセット内のエミッタに対してカーブエディタとの接続を更新.
        /// </summary>
        public void UpdateEmitterAnimetion()
        {
            foreach (var emitter in this.Emitters)
            {
                emitter.UpdateEmitterAnimation();
            }
        }

        /// <summary>
        /// エミッタセット内のエミッタをカーブエディタから切断.
        /// </summary>
        public void DisconnectEmitterAnimetion()
        {
            foreach (var emitter in this.Emitters)
            {
                emitter.DisconnectEmitterAnimation();
            }
        }

        /// <summary>
        /// 描画パスをチェックします.
        /// </summary>
        /// <param name="showCancel">キャンセルボタンの表示On/Off</param>
        /// <returns>チェック途中にユーザーが処理をキャンセルしたときfalseを返します.</returns>
        public bool VerifyDrawPathId(bool showCancel)
        {
            // オプションに設定されている描画パスリストを取得
            var drawPaths = OptionStore.ProjectConfig.DrawPaths;

            if (drawPaths.Count == 0)
            {
                return true;
            }

            // デフォルトの描画パスを取得
            var defaultDrawPath = OptionStore.ProjectConfig.DefaultDrawPath;

            // 全てのエミッタの描画パスをチェック
            foreach (var emitter in this.Emitters)
            {
                // 描画パス名を取得
                string drawPathName = emitter.EmitterBasicViewModel.EmitterBasicRenderViewModel.Proxy.DrawPathName;

                // 描画パスが未設定(文字列が空)のときはチェックしない
                if (string.IsNullOrEmpty(drawPathName))
                {
                    continue;
                }

                // オプションに対応する描画パスがあるか検索
                bool find = drawPaths.Any(item => item.Text.Equals(drawPathName));

                // 描画パスが見つからないとき
                if (find == false)
                {
                    // ダイアログを表示
                    DialogResult result = WorkspaceRootViewModel.Instance.Dialogs.ShowInvalidDrawPathDialog(
                        this.FileName,
                        emitter.Name,
                        showCancel);

                    if (result == DialogResult.Cancel)
                    {
                        return false;
                    }
                    else if (result == DialogResult.Yes)
                    {
                        emitter.EmitterBasicViewModel.EmitterBasicRenderViewModel.DrawPath = defaultDrawPath.Id;
                    }
                }
            }

            return true;
        }

        /// <summary>
        /// 終了処理タイプをチェックします.
        /// </summary>
        /// <param name="showCancel">キャンセルボタンの表示On/Off</param>
        /// <returns>チェック結果を返します.</returns>
        public DialogResult CheckEndingType(bool showCancel)
        {
            string names = string.Empty;

            // エミッタの終了処理タイプをチェック
            foreach (EmitterViewModel emitter in this.Emitters)
            {
                // 関係する情報を取得
                bool enableInfinityLife = emitter.EmitterParticleViewModel.EmitterParticleLifeViewModel.Proxy.EnableInfinityLife;
                bool enableEndProcess = emitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.Proxy.EnableEndProcess;
                bool enableAlphaFade = emitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.Proxy.EnableAlphaFade;
                bool enableScaleFade = emitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.Proxy.EnableScaleFade;

                // 寿命:無限, 終了処理:放出停止/フェードなし のとき、ダイアログに表示する名前を追加
                if (enableInfinityLife == true && enableEndProcess == true && enableAlphaFade == false && enableScaleFade == false)
                {
                    names += (names.Length > 0 ? "\n" : string.Empty) +
                             this.Proxy.Name + ":" + emitter.Name;
                }
            }

            // エミッタ名が一つでも登録されていたらダイアログを表示
            if (string.IsNullOrEmpty(names) == false)
            {
                DialogResult result = WorkspaceRootViewModel.Instance.Dialogs.ShowInvalidEndingType(
                    names + "\n" + Properties.Resources.WarningInvalidEndingType,
                    Properties.Resources.WarningCaption,
                    showCancel);

                return result;
            }

            return DialogResult.Yes;
        }

        /// <summary>
        /// TextureFileChanged イベントを発生させます。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="path">アセットファイルパス</param>
        public void TriggerTextureFileChanged(object sender, string path)
        {
            if (this.TextureFileChanged != null)
            {
                this.TextureFileChanged(sender, path);
            }
        }

        /// <summary>
        /// PrimitiveFileChanged イベントを発生させます。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="path">アセットファイルパス</param>
        public void TriggerPrimitiveFileChanged(object sender, string path)
        {
            if (this.PrimitiveFileChanged != null)
            {
                this.PrimitiveFileChanged(sender, path);
            }
        }

        /// <summary>
        /// Dispose時にアセットリロードイベントを解除
        /// </summary>
        public override void Dispose()
        {
            BusinessLogic.Manager.TextureManager.Instance.FileReloaded -= this.OnTextureFileReloaded;
            BusinessLogic.Manager.PrimitiveManager.Instance.FileReloaded -= this.OnPrimitiveFileReloaded;
            BusinessLogic.GfxToolsUtility.GfxTextureBinaryConverter.Instance.RemoveCache(this.DataModel.Guid);
            BusinessLogic.GfxToolsUtility.GfxPrimitiveBinaryConverter.Instance.RemoveCache(this.DataModel.Guid);

            base.Dispose();
        }

        /// <summary>
        /// Create a data model proxy.
        /// This method is called in the constructor.
        /// If you need a specific type of data model proxy,
        /// override this method and return the desired data model proxy.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The created data model proxy.</returns>
        protected override DataModelProxy CreateDataModelProxy(DataModelBase dataModel)
        {
            return new EmitterSetDataProxy(dataModel);
        }

        /// <summary>
        /// Check if the property value has been modified.
        /// This method is and should only be used for comparing property values
        /// with their original value and default value.
        /// </summary>
        /// <param name="propertyName">The property name.</param>
        /// <param name="origValue">The original/default value.</param>
        /// <param name="currValue">The current value.</param>
        /// <returns>True if the value has been modified.</returns>
        protected override bool CheckPropertyValueModification(
            string propertyName,
            object origValue,
            object currValue)
        {
            if (propertyName != "EmitterList")
            {
                return base.CheckPropertyValueModification(propertyName, origValue, currValue);
            }

            // Check if the value is a sequence of Guid.
            // If it is, the property is probably a data model collection, in which
            // the elements are the Guid of the data models.
            if (origValue is IEnumerable<Guid>)
            {
                var origGuidList = (IEnumerable<Guid>)origValue;
                var currGuidList = DataModelProxy.ExtractDataModelGuids(currValue) as IEnumerable<Guid>;
                if (currGuidList == null)
                {
                    return false;
                }

                // Unlike the original implementation in ViewModelBase, we want to perform
                // ordered comparison for EmitterList and PreviewList because we want to
                // mark the property as modified when user drags and reorders the emitter
                // or preview nodes from the project tree view.
                return !currGuidList.SequenceEqual(origGuidList);
            }

            return false;
        }

        /// <summary>
        /// Handle CollectionChanged event for the children collection.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The event arguments.</param>
        private void OnChildrenCollectionChanged(
            object sender,
            NotifyCollectionChangedEventArgs e)
        {
            this.OnPropertyChanged(() => this.DataModel.EmitterList);
        }

        /// <summary>
        /// エミッタを貼り付け.
        /// </summary>
        /// <param name="parameter">パラメータ.</param>
        private void OnNodePasteEmitter(object parameter)
        {
            var value = Clipboard.GetData(NodeClipboardData.CustomerFormatName) as NodeClipboardData;
            if (value == null || typeof(EmitterData).FullName != value.ClassFullName)
            {
                return;
            }

            DataModelBase genericDataModel;

            {
                var result = DataModelSerializeUtility.DeserializeFromClipboardData(value.XmlData, value.ClassFullName);

                if (result.IsSuccess == false)
                {
                    Logger.Log(LogLevels.Warning, "EmitterSetViewModel.OnNodePasteEmitter : Failed to execute Deserialize.");

                    return;
                }

                genericDataModel = result.DataModel;
            }

            using (new EnableDataModelCloneSetter())
            {
                genericDataModel = (DataModelBase)genericDataModel.CloneWithoutGuid();
            }

            var addedVm = this.CreateAndAddViewModel(this, genericDataModel, true) as IModificationFlagOwner;
            if (addedVm != null)
            {
                addedVm.ModificationFlagViewModel.UpdateDefaultValues();
            }
        }

        /// <summary>
        /// エクスプローラで開く.
        /// </summary>
        /// <param name="parameter">パラメータ.</param>
        private void OnOpenExplorer(object parameter)
        {
            if (!this.IsFileExist)
            {
                return;
            }

            string commandline;

            string path = this.FilePath + @"\" + this.FileName + IOConstants.EmitterSetFileExtension;
            bool selectFile = true;
            if (selectFile == true)
            {
                commandline = "/e, /select,\"" + path + "\"";
            }
            else
            {
                commandline = "/e,\"" + path + "\"";
            }

            System.Diagnostics.Process.Start("EXPLORER.EXE", commandline);
        }

        /// <summary>
        /// コンテキストメニューアイテムの状態を評価する.
        /// </summary>
        /// <param name="parameter">パラメータ.</param>
        private void OnEvaluateMenuItems(object parameter)
        {
            var openExlorerExec = this.NodeOpenExlorerExecutable as AnonymousExecutable;
            if (openExlorerExec != null)
            {
                // ストライプのインスタンスが作成されている場合、無効にする.
                openExlorerExec.IsEnabled = this.IsFileExist;
            }

            var pasteEmitterExec = this.NodePasteEmitterExecutable as AnonymousExecutable;
            if (pasteEmitterExec != null)
            {
                pasteEmitterExec.IsEnabled = this.CanPasteEmitter;
            }
        }

        /// <summary>
        /// Retrives all the Emitter names in the EmitterSet.
        /// </summary>
        /// <param name="node">The current node to traverse.</param>
        /// <param name="names">A writable collection of name.</param>
        private void GatherAllNodeNames(IHierarchyObject node, HashSet<string> names)
        {
            foreach (var child in node.Children)
            {
                if ((child is WorkspaceNodeViewModelBase) == false)
                {
                    continue;
                }

                string name = GetName(child);
                if (string.IsNullOrWhiteSpace(name) == false)
                {
                    names.Add(name);
                }

                this.GatherAllNodeNames(child, names);
            }
        }

        /// <summary>
        /// テクスチャリロード後イベント
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void OnTextureFileReloaded(object sender, FileReloadedEventArgs e)
        {
            // 全エミッタ取得.
            foreach (var emitter in this.Emitters)
            {
                if (emitter.DataModel.TexturePaths.Any(textureFile => textureFile == e.FilePath))
                {
                    using (new ForceRefreshBinary())
                    {
                        ViewerMessageHelper.SendEmitterSet(this.DataModel);
                    }

                    return;
                }
            }
        }

        /// <summary>
        /// プリミティブリロード後イベント
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void OnPrimitiveFileReloaded(object sender, FileReloadedEventArgs e)
        {
            // どれか1つでもヒットしたら1回だけリロードする
            // 全エミッタ取得.
            foreach (var emitter in this.Emitters)
            {
                if (emitter.DataModel.PrimitivePaths.Any(primitiveFile => primitiveFile == e.FilePath))
                {
                    using (new ForceRefreshBinary())
                    {
                        ViewerMessageHelper.SendEmitterSet(this.DataModel);
                    }

                    return;
                }
            }
        }
    }
}
