﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.Properties;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterSetData.
    /// </summary>
    public class EmitterSetBasicBasicViewModel : PropertyGroupViewModel<EmitterSetBasicData>, IModificationFlagOwner
    {
        /// <summary>The available label color items.</summary>
        private static KeyValuePair<string, object>[] labelColorItems = new KeyValuePair<string, object>[]
        {
            new KeyValuePair<string, object>("Color 0", 0),
            new KeyValuePair<string, object>("Color 1", 1),
            new KeyValuePair<string, object>("Color 2", 2),
            new KeyValuePair<string, object>("Color 3", 3),
            new KeyValuePair<string, object>("Color 4", 4),
            new KeyValuePair<string, object>("Color 5", 5),
            new KeyValuePair<string, object>("Color 6", 6),
            new KeyValuePair<string, object>("Color 7", 7),
        };

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterSetBasicBasicViewModel(HierarchyViewModel parent, EmitterSetBasicData dataModel)
            : base(parent, dataModel)
        {
            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
        }

        /// <summary>
        /// Get the available label color items.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> AvailableLabelColors
        {
            get { return labelColorItems; }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// ファイル名(拡張子を含まない)を設定または取得します。
        /// このプロパティはEmitterSetBasicFileInfoViewModelに移動予定です。
        /// </summary>
        public string Name
        {
            get
            {
                string fileName = string.Empty;
                IHierarchyObject parent = this.Parent;

                // エミッタセットのファイル名を取得
                EmitterSetViewModel emitterSetViewModel = HierarchyViewModel.GetParent<EmitterSetViewModel>(this);
                if (emitterSetViewModel != null)
                {
                    fileName = emitterSetViewModel.FileName;
                }

                return fileName;
            }
        }

        /// <summary>
        /// ファイルパス(ファイル名を含まない)を設定または取得します。
        /// このプロパティはEmitterSetBasicFileInfoViewModelに移動予定です。
        /// </summary>
        public string FilePath
        {
            get
            {
                string filePath = string.Empty;
                IHierarchyObject parent = this.Parent;

                // エミッタセットのファイルパスを取得
                EmitterSetViewModel emitterSetViewModel = HierarchyViewModel.GetParent<EmitterSetViewModel>(this);
                if (emitterSetViewModel != null)
                {
                    filePath = emitterSetViewModel.FilePath;
                }

                // ファイルパスが決まっていないとき、メッセージを表示
                if (string.IsNullOrEmpty(filePath))
                {
                    filePath = Resources.EmitterSetBasicFileInfoEmptyFilePath;
                }

                return filePath;
            }
        }

        /// <summary>
        /// エミッタセット名更新用のプロパティ更新
        /// </summary>
        public void RaiseNameChanged()
        {
            this.OnPropertyChanged(() => this.Name);
        }

        /// <summary>
        /// ファイルパス更新用のプロパティ更新
        /// </summary>
        public void RaiseFilePathChanged()
        {
            this.OnPropertyChanged(() => this.FilePath);
        }
    }
}
