﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using EffectMaker.BusinessLogic;
using EffectMaker.BusinessLogic.Commands;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Texture;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.Properties;
using EffectMaker.UILogic.ViewModels.Factories;
using EffectMaker.UILogic.ViewModels.IO;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterData.
    /// </summary>
    public class EmitterViewModel : WorkspaceNodeViewModelBase<EmitterData>, IEmitterOwnerViewModel, IModificationFlagOwner
    {
        /// <summary>
        /// ファイルの存在しないアセットがあるかどうかのフラグです.
        /// 子エミッタの状態も含めます.
        /// </summary>
        private bool hasUnreachableAsset;

        /// <summary>
        /// 不正なアセットがあるかどうかのフラグです.
        /// 子エミッタの状態も含めます.
        /// </summary>
        private bool hasInvalidAsset;

        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignoreModifyPropertyNames = new string[]
        {
            "CanPasteEmitterAsChild",
            "AvailableReservedShaders",
            "HasInvalidAsset"
        };

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterViewModel(HierarchyViewModel parent, EmitterData dataModel)
            : base(parent, dataModel)
        {
            var typeOrdered = Children as TypeOrderedObservableCollection<IHierarchyObject>;

            // チャイルドノードの並び順を設定
            if (typeOrdered != null)
            {
                typeOrdered.TypeOrder = new Type[]
                {
                    typeof(FieldViewModel),
                    typeof(ReservedShaderNodeViewModel),
                    typeof(CustomActionViewModel),
                    typeof(EmitterExtParamsViewModel),
                    typeof(EmitterViewModel),
                };
                typeOrdered.Sort();
            }

            EmitterAnimationViewModel.IsInitializing = true;

            // 基本設定.
            this.EmitterBasicViewModel = this.AddChildViewModel(
                new EmitterBasicViewModel(this, dataModel.EmitterBasicSettingData));

            // エミッタ.
            this.EmitterEmitterViewModel = this.AddChildViewModel(
                new EmitterEmitterViewModel(this, dataModel.EmitterEmitterData));

            // 放出.
            this.EmitterEmissionViewModel = this.AddChildViewModel(
                new EmitterEmissionViewModel(this, dataModel.EmitterEmissionData));

            // パーティクル.
            this.EmitterParticleViewModel = this.AddChildViewModel(
                new EmitterParticleViewModel(this, dataModel.EmitterParticleData));

            // コンバイナ.
            this.EmitterCombinerViewModel = this.AddChildViewModel(
                new EmitterCombinerViewModel(this, dataModel.EmitterCombinerData));

            // テクスチャ.
            this.EmitterTextureGroupViewModel = this.AddChildViewModel(
                new EmitterTextureGroupViewModel(this, dataModel.Textures));

            // カラー.
            this.EmitterColorGroupViewModel = this.AddChildViewModel(
                new EmitterColorPageViewModel(this, dataModel.EmitterColorData));

            // スケール.
            this.EmitterScaleViewModel = this.AddChildViewModel(
                new EmitterScaleViewModel(this, dataModel.EmitterScaleData));

            // 回転.
            this.EmitterRotationViewModel = this.AddChildViewModel(
                new EmitterRotationViewModel(this, dataModel.EmitterRotationData));

            // カスタムシェーダー.
            this.EmitterCustomShaderViewModel = this.AddChildViewModel(
                new EmitterCustomShaderViewModel(this, dataModel.EmitterCustomShaderData));

            EmitterAnimationViewModel.IsInitializing = false;

            // フィールドノードを[エミッタより先に]追加する.
            foreach (FieldDataBase fieldData in dataModel.FieldList)
            {
                this.AddChildViewModel(this.CreateFieldViewModelByType(this, fieldData));
            }

            // エミッターノードを追加する.
            foreach (EmitterData emitterData in dataModel.EmitterList)
            {
                this.AddChildViewModel(new EmitterViewModel(this, emitterData));
            }

            // エミッタプラグインノードを追加する.
            if (dataModel.ReservedShader != null)
            {
                if (dataModel.ReservedShader.PageData == null ||
                    dataModel.ReservedShader.PageData.ContentsData == null)
                {
                    // If the user data was not loaded correctly, remove the node.
                    this.Proxy.RemoveReservedShader(dataModel.ReservedShader);
                }
                else
                {
                    this.AddChildViewModel(new ReservedShaderNodeViewModel(this, dataModel.ReservedShader));
                }
            }

            // カスタムアクションノードを追加する.
            if (dataModel.CustomActionData != null)
            {
                this.AddChildViewModel(new CustomActionViewModel(this, dataModel.CustomActionData));
            }

            // エミッタ拡張パラメータノードを追加する
            if (dataModel.EmitterExtParams != null)
            {
                this.AddChildViewModel(new EmitterExtParamsViewModel(this, dataModel.EmitterExtParams));
            }

            // Create view models for the existing user data models.
            this.UpdateUserData(true);

            // ユーザーデータのアップデート通知イベントを登録.
            UserDataManager.UserDataUpdated += this.OnUserDataUpdated;

            // プロセスタイプ変更イベントを登録
            this.EmitterBasicViewModel.EmitterBasicBasicViewModel.PropertyChanged += (s, e) =>
            {
                if (e.PropertyName == "ProcessType")
                {
                    this.OnPropertyChanged(() => this.ProcessType);
                }
            };

            // Add a listener to monitor child list modification.
            this.Children.CollectionChanged += this.OnChildrenCollectionChanged;

            this.ReservedShaderMenuExecutable = new AnonymousExecutable(this.OnCreateReservedShader);

            this.CustomActionMenuExecutable = new AnonymousExecutable(this.OnCreateCustomAction);

            this.EmitterExtParamsMenuExecutable = new AnonymousExecutable(this.OnCreateEmitterExtParams);

            this.EvaluateMenuItemsExecutable = new AnonymousExecutable(this.OnEvaluateMenuItems);

            // アセットのステータスを初期化
            this.UpdateAssetStatus(false, false);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            var modificationFlagsViewModel = new ModificationFlagViewModel(this);
            modificationFlagsViewModel.AddIgnoreProperties(this.ignoreModifyPropertyNames);
            this.ModificationFlagViewModel = modificationFlagsViewModel;
        }

        /// <summary>
        /// Get or set the flag indicating whether the data model should be displayed/rendered.
        /// </summary>
        public bool Displayed
        {
            get
            {
                return this.GetDataModelValue(() => this.Displayed);
            }

            set
            {
                bool originalValue = this.GetDataModelValue(() => this.Displayed);
                if (originalValue != value)
                {
                    var binder = new EffectMakerSetMemberBinder("Displayed", false, false);
                    this.TrySetMember(binder, value);

                    // Send visibility message to the viewer.
                    ViewerController.Instance.SendVisibility(this.DataModel);
                }
            }
        }

        /// <summary>
        /// ノードデータをバイナリにコンバートするかどうか取得または設定します。
        /// </summary>
        [UseDataModelOriginalValue]
        public bool EnableConvert
        {
            get
            {
                return this.GetDataModelValue(() => this.EnableConvert);
            }

            set
            {
                using (new MessageBlockerWithSendBinaryOnce(this.DataModel))
                {
                    this.SetDataModelValue(value, () => this.EnableConvert);

                    // 親エミッタのGPUモードを更新
                    var parentEmitter = this.Parent as EmitterViewModel;
                    if (parentEmitter != null)
                    {
                        parentEmitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.IsUsingCpu = value;
                        //parentEmitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu();
                    }
                }
            }
        }

        /// <summary>
        /// エミッタが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewEmitter
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveEmitter(this.Parent); }
        }

        /// <summary>
        /// 子エミッタが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewChildEmitter
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveEmitter(this); }
        }

        /// <summary>
        /// 子エミッタとして貼り付け可能か調べる.
        /// </summary>
        public bool CanPasteEmitterAsChild
        {
            get
            {
                IExecutable update = this.EvaluateCopyPasteAvailabilityExecutable;
                IExecutable exec = this.NodePasteExecutable;
                if (update == null || exec == null)
                {
                    return false;
                }

                update.Execute(null);
                return this.CanCreateNewChildEmitter && exec.CanExecute(null);
            }
        }

        /// <summary>
        /// 新ランダムフィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewRandomField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveRandomField(this);  }
        }

        /// <summary>
        /// ランダムフィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewRandomFe1Field
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveRandomFe1Field(this);  }
        }

        /// <summary>
        /// 磁力フィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewMagnetField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveMagnetField(this); }
        }

        /// <summary>
        /// スピンフィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewSpinField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveSpinField(this); }
        }

        /// <summary>
        /// 収束フィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewConvergeField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveConvergeField(this); }
        }

        /// <summary>
        /// "位置に加算"フィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewAddLocationField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveAddLocationField(this); }
        }

        /// <summary>
        /// シンプルコリジョンフィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewCollisionField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveCollisionField(this); }
        }

        /// <summary>
        /// カールノイズフィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewCurlNoiseField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveCurlNoiseField(this);  }
        }

        /// <summary>
        /// カスタムフィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewCustomField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveCustomField(this); }
        }

        /// <summary>
        /// エミッタプラグインが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewReservedShader
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveReservedShader(this); }
        }

        /// <summary>
        /// 階層化を解除できるか（＝子エミッタか）取得します。
        /// </summary>
        public bool CanUnparentEmitter
        {
            get
            {
                var parent = this.Parent as EmitterViewModel;
                return parent != null;
            }
        }

        /// <summary>
        /// コンテキストメニューアイテムの状態を評価するExecutable.
        /// </summary>
        public IExecutable EvaluateMenuItemsExecutable { get; set; }

        /// <summary>
        /// Executable for creating reserved shader node.
        /// </summary>
        public IExecutable ReservedShaderMenuExecutable { get; set; }

        /// <summary>
        /// カスタムアクションを作成するExecutable.
        /// </summary>
        public IExecutable CustomActionMenuExecutable { get; set; }

        /// <summary>
        /// エミッタ拡張パラメータを作成するExecutable.
        /// </summary>
        public IExecutable EmitterExtParamsMenuExecutable { get; set; }

        /// <summary>
        /// Get the selected item.
        /// </summary>
        public HierarchyViewModel SelectedItem { get; set; }

        /// <summary>
        /// EmitterBasicViewModel.
        /// </summary>
        public EmitterBasicViewModel EmitterBasicViewModel { get; private set; }

        /// <summary>
        /// EmitterEmitterViewModel.
        /// </summary>
        public EmitterEmitterViewModel EmitterEmitterViewModel { get; private set; }

        /// <summary>
        /// EmitterEmissionViewModel.
        /// </summary>
        public EmitterEmissionViewModel EmitterEmissionViewModel { get; private set; }

        /// <summary>
        /// EmitterParticleViewModel.
        /// </summary>
        public EmitterParticleViewModel EmitterParticleViewModel { get; private set; }

        /// <summary>
        /// EmitterCombinerViewModel.
        /// </summary>
        public EmitterCombinerViewModel EmitterCombinerViewModel { get; private set; }

        /// <summary>
        /// EmitterTextureGroupViewModel.
        /// </summary>
        public EmitterTextureGroupViewModel EmitterTextureGroupViewModel { get; private set; }

        /// <summary>
        /// EmitterColorGroupViewModel.
        /// </summary>
        public EmitterColorPageViewModel EmitterColorGroupViewModel { get; private set; }

        /// <summary>
        /// EmitterScaleViewModel.
        /// </summary>
        public EmitterScaleViewModel EmitterScaleViewModel { get; private set; }

        /// <summary>
        /// EmitterRotationViewModel.
        /// </summary>
        public EmitterRotationViewModel EmitterRotationViewModel { get; private set; }

        /// <summary>
        /// EmitterCustomShaderViewModel.
        /// </summary>
        public EmitterCustomShaderViewModel EmitterCustomShaderViewModel { get; private set; }

        /// <summary>
        /// The default property page view model to use
        /// on the first time the workspace node is selected.
        /// </summary>
        public override HierarchyViewModel DefaultPropertyPageViewModel
        {
            get { return this.EmitterBasicViewModel; }
        }

        /// <summary>
        /// SelectedPropertyPage.
        /// </summary>
        public override HierarchyViewModel SelectedPropertyPage
        {
            get
            {
                return base.SelectedPropertyPage;
            }

            set
            {
                base.SelectedPropertyPage = value;

                // カーブエディタウィンドウにタブページの切り替えを通知
                var exe = WorkspaceRootViewModel.Instance.Dialogs.OnNotifyTabPageToCurveEditorExecutable;
                if (exe != null)
                {
                    exe.Execute(this.CurrentEditingCurveOwener);
                    this.CurrentEditingCurveOwener = null;
                }
            }
        }

        /// <summary>
        /// カーブエディタウィンドウのアンドゥ・リドゥ対応のためのプロパティ
        /// </summary>
        public object CurrentEditingCurveOwener { get; set; }

        /// <summary>
        /// Enumerates the available reserved shader items.
        /// (each item contains the name, the icon image, and the user data info)
        /// </summary>
        public IEnumerable<Tuple<string, Image, object>> AvailableReservedShaders
        {
            get { return WorkspaceRootViewModel.Instance.AvailableReservedShaders; }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// 名前を設定できるかどうか取得します.
        /// </summary>
        public override bool CanSetName
        {
            get { return true; }
        }

        /// <summary>
        /// Get the name of the emitter.
        /// </summary>
        public string Name
        {
            get { return this.GetDataModelValue(() => this.Name); }
        }

        /// <summary>
        /// プロセスタイプを取得します。
        /// </summary>
        public int ProcessType
        {
            get
            {
                return this.EmitterBasicViewModel.EmitterBasicBasicViewModel.GetDataModelValue<int>("ProcessType");
            }
        }

        /// <summary>
        /// 実ファイルの存在しないアセットがあるかどうかのフラグを取得または設定します.
        /// </summary>
        public bool HasUnreachableAsset
        {
            get { return this.hasUnreachableAsset; }
            set { this.SetValue(ref this.hasUnreachableAsset, value); }
        }

        /// <summary>
        /// 不正なアセットがあるかどうかのフラグを取得または設定します.
        /// </summary>
        public bool HasInvalidAsset
        {
            get { return this.hasInvalidAsset; }
            set { this.SetValue(ref this.hasInvalidAsset, value); }
        }

        /// <summary>
        /// Perform a deep enumeration of all the emitters, including
        /// all the child emitters.
        /// </summary>
        public IEnumerable<EmitterViewModel> Emitters
        {
            get
            {
                IEnumerable<EmitterViewModel> emitters =
                    from item in this.Children
                    where item is EmitterViewModel
                    select (EmitterViewModel)item;

                foreach (EmitterViewModel emitter in emitters)
                {
                    yield return emitter;

                    foreach (EmitterViewModel childEmitter in emitter.Emitters)
                    {
                        yield return childEmitter;
                    }
                }
            }
        }

        /// <summary>
        /// 自分の直下にある子エミッタだけを返す
        /// </summary>
        public IEnumerable<EmitterViewModel> ChildEmitters
        {
            get
            {
                IEnumerable<EmitterViewModel> emitters =
                    from item in this.Children
                    where item is EmitterViewModel
                    select (EmitterViewModel)item;

                foreach (EmitterViewModel emitter in emitters)
                {
                    yield return emitter;
                }
            }
        }

        /// <summary>
        /// Disposes the instance.
        /// </summary>
        public override void Dispose()
        {
            this.DisconnectEmitterAnimation();
            base.Dispose();

            // ユーザーデータアップデートの通知イベントの解除.
            UserDataManager.UserDataUpdated -= this.OnUserDataUpdated;
        }

        /// <summary>
        /// 名前が正しいかチェックします.
        /// </summary>
        /// <param name="name">名前</param>
        /// <returns>名前が正しいときtrueを返します.</returns>
        public override bool IsNameValid(string name)
        {
            // 名前に不正な文字が入っていないかチェック
            Regex validPattern = new Regex(Resources.RegexpNodeNameValidation);

            if (validPattern.IsMatch(name) == false)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowWarningInvalidName();
                return false;
            }

            // エミッタ名を64バイトに制限。NULL終端があるので63文字制限。
            if (name.Length > 63)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowWarningTooLongEmitterName();
                return false;
            }

            // 親のエミッタセットを取得
            EmitterSetViewModel emitterSetViewModel = ViewModelBase.GetParent<EmitterSetViewModel>(this);
            if (emitterSetViewModel == null)
            {
                return false;
            }

            // エミッタの名前が被らないかチェック
            foreach (EmitterViewModel emitter in emitterSetViewModel.Emitters)
            {
                if (emitter != this && emitter.Name == name)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowWarningEmitterNameAlreadyUsed(name);
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// 名前を設定します.
        /// </summary>
        /// <param name="name">名前</param>
        public override void SetName(string name)
        {
            // 名前を設定
            this.DataModel.Name = name;
            this.OnPropertyChanged(() => this.DataModel.Name);
        }

        /// <summary>
        /// アセットのステータスを更新します。
        /// </summary>
        /// <param name="unreachable">アセットが存在しない</param>
        /// <param name="invalid">アセットが不正</param>
        public void UpdateAssetStatus(bool unreachable, bool invalid)
        {
            // 子エミッタのステータスを取得
            foreach (EmitterViewModel child in this.Emitters)
            {
                if (child.HasUnreachableAsset)
                {
                    unreachable = true;
                }

                if (child.hasInvalidAsset)
                {
                    invalid = true;
                }
            }

            var textureManager = EffectMaker.BusinessLogic.Manager.TextureManager.Instance;
            bool isLinearMode = OptionStore.ProjectConfig.LinearMode;

            // 自身のステータスを取得
            List<AssetInfo> assetInfo = AssetsManager.GetAssetInfo(this);

            foreach (AssetInfo info in assetInfo)
            {
                if (string.IsNullOrEmpty(info.FilePath))
                {
                    continue;
                }
                else if (unreachable == false && Path.IsPathRooted(info.FilePath) == false)
                {
                    unreachable = true;
                }
                else if (unreachable == false && File.Exists(info.FilePath) == false)
                {
                    unreachable = true;
                }

                // リニアモードのチェック
                if (File.Exists(info.FilePath))
                {
                    LoadTextureResult result = textureManager.LoadTexture(info.FilePath, true);

                    if (result.ResultCode == LoadTextureResultCode.Success)
                    {
                        bool isLinearTexture = result.TextureData.Linear.Any(x => x);

                        // コンバイナの設定によってエミッタアイコンの差し替えを抑制する
                        var flag = this.EmitterTextureGroupViewModel.Texture0.EmitterTextureFileViewModel
                                .CheckExceptionLinearWarning(isLinearTexture);

                        // オプションで非表示になるよう設定しているときも抑制する
                        flag |= !OptionStore.RootOptions.Basic.Emitters.ShowWarningIconLinearTexture;

                        // 抑制対象時は、テクスチャと編集モードが一致していなくても無効にはしない
                        if (isLinearMode != isLinearTexture && !flag)
                        {
                            invalid = true;
                        }
                    }
                }
            }

            bool statusChanged = unreachable != this.HasUnreachableAsset || invalid != this.HasInvalidAsset;

            this.HasUnreachableAsset = unreachable;
            this.HasInvalidAsset = invalid;

            // 親エミッタを再帰的に更新
            if (statusChanged)
            {
                EmitterViewModel parent = ViewModelBase.GetParent<EmitterViewModel>((ViewModelBase)this.Parent);
                if (parent != null)
                {
                    parent.UpdateAssetStatus(this.HasUnreachableAsset, this.HasInvalidAsset);
                }
            }
        }

        /// <summary>
        /// Update child view models with the current data model.
        /// This method is usually called when data model is modified, thus some child
        /// view models might need to be created or removed according to the data model.
        /// </summary>
        public override void UpdateChildViewModels()
        {
            // Remove old field view models.
            var viewModelsToRemove = this.Children.Where(ch => ch is FieldViewModel).ToArray();
            viewModelsToRemove.ForEach(vm => this.Children.Remove(vm));

            // Add field view models for each field data.
            foreach (FieldDataBase fieldData in this.DataModel.FieldList)
            {
                this.AddChildViewModel(this.CreateFieldViewModelByType(this, fieldData));
            }

            // Remove old emitter view models.
            viewModelsToRemove = this.Children.Where(ch => ch is EmitterViewModel).ToArray();
            viewModelsToRemove.ForEach(vm => this.Children.Remove(vm));

            // Add emitter view models for each emitter data.
            foreach (EmitterData emitterData in this.DataModel.EmitterList)
            {
                this.AddChildViewModel(new EmitterViewModel(this, emitterData));
            }

            // Assign reserved shader data to its view model.
            var reservedShaderViewModel =
                (ReservedShaderNodeViewModel)this.Children.FirstOrDefault(ch => ch is ReservedShaderNodeViewModel);

            if (this.DataModel.ReservedShader != null && reservedShaderViewModel == null)
            {
                this.AddChildViewModel(new ReservedShaderNodeViewModel(this, this.DataModel.ReservedShader));
            }
            else if (this.DataModel.ReservedShader == null && reservedShaderViewModel != null)
            {
                this.Children.Remove(reservedShaderViewModel);
            }

            // Assign custom action data to its view model.
            var customActionViewModel =
                (CustomActionViewModel)this.Children.FirstOrDefault(ch => ch is CustomActionViewModel);

            if (this.DataModel.CustomActionData != null && customActionViewModel == null)
            {
                this.AddChildViewModel(new CustomActionViewModel(this, this.DataModel.CustomActionData));
            }
            else if (this.DataModel.CustomActionData == null && customActionViewModel != null)
            {
                this.Children.Remove(customActionViewModel);
            }

            // Assign emitter ext params to its view model.
            var emitterExtParamsViewModel =
                (EmitterExtParamsViewModel)this.Children.FirstOrDefault(ch => ch is EmitterExtParamsViewModel);

            if (this.DataModel.EmitterExtParams != null && emitterExtParamsViewModel == null)
            {
                this.AddChildViewModel(new EmitterExtParamsViewModel(this, this.DataModel.EmitterExtParams));
            }
            else if (this.DataModel.EmitterExtParams == null && emitterExtParamsViewModel != null)
            {
                this.Children.Remove(emitterExtParamsViewModel);
            }

            // Create view models for the existing user data models.
            this.UpdateUserData(false);

            this.UpdateAssetStatus(false, false);

            // Although all the emitter, field, stripe and custom action view models
            // are up to date but there are also other child view models that need to
            // be updated so we still have to call base.UpdateChildViewModels().
            base.UpdateChildViewModels();
        }

        #region Function - IEmitterOwnerViewModel

        /// <summary>
        /// エミッタを作成します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="emitterName">エミッタ名</param>
        /// <param name="presetEmitter">プリセットとなるエミッタデータ(nullなら完全新規作成)</param>
        /// <returns>エミッタビューモデルを返します。</returns>
        public EmitterViewModel CreateNewEmitter(string emitterName, EmitterData presetEmitter)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return null;
            }

            EmitterData emitterData = null;

            // Create the emitter data.
            if (presetEmitter == null)
            {
                emitterData = proxy.CreateEmitter(emitterName);
                if (emitterData == null)
                {
                    return null;
                }
            }
            else
            {
                emitterData = presetEmitter;
                emitterData.Name = emitterName;
                proxy.AddEmitter(emitterData);
            }

            // Create the emitter view model.
            EmitterViewModel emitterViewModel = new EmitterViewModel(this, emitterData);

            return emitterViewModel;
        }

        /// <summary>
        /// エミッタを追加します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="emitterViewModel">エミッタビューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool AddEmitter(EmitterViewModel emitterViewModel)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // Add the emitter data model.
            proxy.AddEmitter(emitterViewModel.DataModel);

            // Add the emitter view model.
            this.Children.Add(emitterViewModel);
            emitterViewModel.Parent = this;

            return true;
        }

        /// <summary>
        /// エミッタを削除します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="emitterViewModel">エミッタビューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool RemoveEmitter(EmitterViewModel emitterViewModel)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // Remove the emitter data model.
            proxy.RemoveEmitter(emitterViewModel.DataModel);

            // Remove the emitter view model.
            this.Children.Remove(emitterViewModel);

            emitterViewModel.Parent = null;

            return true;
        }

        /// <summary>
        /// このエミッタをエミッタセット直下に追加します。
        /// </summary>
        /// <param name="emitterSetViewModel">新たに親とするエミッタセット</param>
        public void EntryIntoEmitterSet(EmitterSetViewModel emitterSetViewModel)
        {
            this.Parent = emitterSetViewModel;
            if (!emitterSetViewModel.Children.Contains(this))
            {
                emitterSetViewModel.Children.Add(this);
            }
        }

        /// <summary>
        /// このエミッタをエミッタセット直下にインデックスを指定して追加します。
        /// </summary>
        /// <param name="emitterSetViewModel">新たに親とするエミッタセット</param>
        /// <param name="index"></param>
        public void EntryIntoEmitterSet(EmitterSetViewModel emitterSetViewModel, int index)
        {
            this.Parent = emitterSetViewModel;
            if (!emitterSetViewModel.Children.Contains(this))
            {
                emitterSetViewModel.Children.Insert(index, this);
            }
        }

        /// <summary>
        /// このエミッタを親から切り離します。
        /// </summary>
        public void RemoveFromParent()
        {
            if (this.Parent.Children.Contains(this))
            {
                this.Parent.Children.Remove(this);
            }

            this.Parent = null;
        }

        /// <summary>
        /// エミッタのインデックスを取得します。
        /// </summary>
        /// <param name="emitterViewModel">エミッタビューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        /// <param name="dataModelIndex">データモデルのインデックス</param>
        public void GetEmitterIndex(
            EmitterViewModel emitterViewModel,
            out int viewModelIndex,
            out int dataModelIndex)
        {
            dataModelIndex = -1;
            viewModelIndex = -1;

            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return;
            }

            // データモデルのインデックスを取得
            dataModelIndex = this.Proxy.EmitterList.IndexOf(emitterViewModel.DataModel);

            // ビューモデルのインデックスを取得
            viewModelIndex = this.Children.IndexOf(emitterViewModel);
        }

        /// <summary>
        /// エミッタを指定インデックスに追加します。
        /// </summary>
        /// <param name="emitterViewModel">エミッタビューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        /// <param name="dataModelIndex">データモデルのインデックス</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool InsertEmitter(
            EmitterViewModel emitterViewModel,
            int viewModelIndex,
            int dataModelIndex)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            using (new SuppressNodeTypeOrdering(this))
            {
                // データモデルを追加
                proxy.InsertChildDataModel(dataModelIndex, emitterViewModel.DataModel);

                // ビューモデルを追加
                this.Children.Insert(viewModelIndex, emitterViewModel);
                emitterViewModel.Parent = this;
            }

            return true;
        }

        #endregion

        #region Function - フィールド管理

        /// <summary>
        /// フィールドを作成します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="fieldType">フィールドタイプ</param>
        /// <returns>フィールドビューモデルを返します。</returns>
        public FieldViewModel CreateNewField(FieldTypes fieldType)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return null;
            }

            // フィールドデータモデルを作成.
            FieldDataBase fieldData = proxy.CreateField(fieldType);
            if (fieldData == null)
            {
                return null;
            }

            // フィールドビューモデルを作成.
            FieldViewModel fieldViewModel = FieldViewModelFactory.CreateFieldViewModel(this, fieldData);

            return fieldViewModel;
        }

        /// <summary>
        /// フィールドを追加します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="fieldViewModel">フィールドビューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool AddField(FieldViewModel fieldViewModel)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // データモデルを追加.
            proxy.AddField(fieldViewModel.Proxy.DataModel);

            // ビューモデルを追加.
            this.Children.Add(fieldViewModel);

            return true;
        }

        /// <summary>
        /// フィールドを削除します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="fieldViewModel">フィールドビューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool RemoveField(FieldViewModel fieldViewModel)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // Remove the field data model.
            proxy.RemoveField(fieldViewModel.Proxy.DataModel);

            // Remove the field view model.
            this.Children.Remove(fieldViewModel);

            return true;
        }

        /// <summary>
        /// フィールドのインデックスを取得します。
        /// </summary>
        /// <param name="fieldViewModel">フィールドビューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        /// <param name="dataModelIndex">データモデルのインデックス</param>
        public void GetFieldIndex(
            FieldViewModel fieldViewModel,
            out int viewModelIndex,
            out int dataModelIndex)
        {
            dataModelIndex = -1;
            viewModelIndex = -1;

            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return;
            }

            // データモデルのインデックスを取得
            dataModelIndex = this.Proxy.FieldList.IndexOf(fieldViewModel.Proxy.DataModel);

            // ビューモデルのインデックスを取得
            viewModelIndex = this.Children.IndexOf(fieldViewModel);
        }

        /// <summary>
        /// フィールドを指定インデックスに追加します。
        /// </summary>
        /// <param name="fieldViewModel">フィールドビューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        /// <param name="dataModelIndex">データモデルのインデックス</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool InsertField(
            FieldViewModel fieldViewModel,
            int viewModelIndex,
            int dataModelIndex)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            using (new SuppressNodeTypeOrdering(this))
            {
                // データモデルを追加
                proxy.InsertChildDataModel(dataModelIndex, fieldViewModel.Proxy.DataModel);

                // ビューモデルを追加
                this.Children.Insert(viewModelIndex, fieldViewModel);
            }

            return true;
        }

        #endregion

        #region Function - エミッタプラグイン管理

        /// <summary>
        /// Create a reserved shader node view model.
        /// (without adding it to the emitter.)
        /// </summary>
        /// <param name="reservedShaderUserDataInfo">The user data info for the reserved shader to create.</param>
        /// <returns>The created reserved shader node view model.</returns>
        public ReservedShaderNodeViewModel CreateNewReservedShader(
            UserDataInfo reservedShaderUserDataInfo)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return null;
            }

            // Create the data model.
            ReservedShaderNodeData dm = proxy.CreateReservedShader(reservedShaderUserDataInfo);
            if (dm == null)
            {
                return null;
            }

            // Create the view model.
            var vm = new ReservedShaderNodeViewModel(this, dm);

            return vm;
        }

        /// <summary>
        /// Add a created reserved shader node to the emitter.
        /// </summary>
        /// <param name="reservedShaderViewModel">The reserved shader node view model to add.</param>
        /// <returns>True on success.</returns>
        public bool AddReservedShader(ReservedShaderNodeViewModel reservedShaderViewModel)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // Add the data model.
            proxy.AddReservedShader(reservedShaderViewModel.DataModel);

            // Add the view model.
            this.Children.Add(reservedShaderViewModel);

            return true;
        }

        /// <summary>
        /// Remove the reserved shader view model from this emitter.
        /// </summary>
        /// <param name="reservedShaderViewModel">The reserved shader node view model to remove.</param>
        /// <returns>True on success.</returns>
        public bool RemoveReservedShader(ReservedShaderNodeViewModel reservedShaderViewModel)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // Remove the data model.
            proxy.RemoveReservedShader(reservedShaderViewModel.DataModel);

            // Remove the view model.
            this.Children.Remove(reservedShaderViewModel);

            return true;
        }

        /// <summary>
        /// Get the index of the reserved shader in the child list.
        /// </summary>
        /// <param name="childViewModel">The child view model.</param>
        /// <param name="childIndex">The index in the child list.</param>
        public void GetIndexInChildList(
            HierarchyViewModel childViewModel,
            out int childIndex)
        {
            childIndex = -1;

            // Get the index of the view model in the child list.
            childIndex = this.Children.IndexOf(childViewModel);
        }

        /// <summary>
        /// Add the reserved shader view model to the specified child list index.
        /// </summary>
        /// <param name="viewModel">The view model to insert.</param>
        /// <param name="childIndex">Index in the child list.</param>
        /// <returns>True on success.</returns>
        public bool InsertReservedShader(
            ReservedShaderNodeViewModel viewModel,
            int childIndex)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            using (new SuppressNodeTypeOrdering(this))
            {
                // Add the data model.
                proxy.AddReservedShader(viewModel.DataModel);

                // Insert the view model to the child list.
                this.Children.Insert(childIndex, viewModel);
            }

            return true;
        }

        #endregion // Function - エミッタプラグイン管理

        #region Function - カスタムアクション管理

        /// <summary>
        /// カスタムアクションを作成します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <returns>カスタムアクションビューモデルを返します。</returns>
        public CustomActionViewModel CreateNewCustomAction()
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return null;
            }

            // Create the custom action data.
            CustomActionData customActionData = proxy.CreateCustomAction();
            if (customActionData == null)
            {
                return null;
            }

            // Create the custom action view model.
            var customActionViewModel = new CustomActionViewModel(this, customActionData);

            return customActionViewModel;
        }

        /// <summary>
        /// カスタムアクションを追加します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="customActionViewModel">カスタムアクションビューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool AddCustomAction(CustomActionViewModel customActionViewModel)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // Add the custom action data model.
            proxy.AddCustomAction(customActionViewModel.DataModel);

            // Add the custom action view model.
            this.Children.Add(customActionViewModel);

            return true;
        }

        /// <summary>
        /// カスタムアクションを削除します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="customActionViewModel">カスタムアクションビューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool RemoveCustomAction(CustomActionViewModel customActionViewModel)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // Remove the custom action data model.
            proxy.RemoveCustomAction(customActionViewModel.DataModel);

            // Remove the custom action view model.
            this.Children.Remove(customActionViewModel);

            return true;
        }

        /// <summary>
        /// カスタムアクションのインデックスを取得します。
        /// </summary>
        /// <param name="customActionViewModel">カスタムアクションビューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        public void GetCustomActionIndex(
            CustomActionViewModel customActionViewModel,
            out int viewModelIndex)
        {
            viewModelIndex = -1;

            // ビューモデルのインデックスを取得
            viewModelIndex = this.Children.IndexOf(customActionViewModel);
        }

        /// <summary>
        /// カスタムアクションを指定インデックスに追加します。
        /// </summary>
        /// <param name="customActionViewModel">カスタムアクションビューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool InsertCustomAction(
            CustomActionViewModel customActionViewModel,
            int viewModelIndex)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            using (new SuppressNodeTypeOrdering(this))
            {
                // データモデルを追加
                proxy.AddCustomAction(customActionViewModel.DataModel);

                // ビューモデルを追加
                this.Children.Insert(viewModelIndex, customActionViewModel);
            }

            return true;
        }

        #endregion // Fuction - カスタムアクション管理

        #region Function - エミッタ拡張パラメータ管理

        /// <summary>
        /// エミッタ拡張パラメータを作成します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <returns>エミッタ拡張パラメータビューモデルを返します。</returns>
        public EmitterExtParamsViewModel CreateNewEmitterExtParams()
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return null;
            }

            // Create the Emitter Extension Parameters.
            EmitterExtParams emitterExtParams = proxy.CreateEmitterExtParams();
            if (emitterExtParams == null)
            {
                return null;
            }

            // Create the Emitter Extension Parameters view model.
            var emitterExtParamsViewModel = new EmitterExtParamsViewModel(this, emitterExtParams);

            return emitterExtParamsViewModel;
        }

        /// <summary>
        /// エミッタ拡張パラメータを追加します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="emitterExtParamsViewModel">エミッタ拡張パラメータビューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool AddEmitterExtParams(EmitterExtParamsViewModel emitterExtParamsViewModel)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // Add the Emitter Extension Parameters model.
            proxy.AddEmitterExtParams(emitterExtParamsViewModel.DataModel);

            // Add the Emitter Extension Parameters view model.
            this.Children.Add(emitterExtParamsViewModel);

            return true;
        }

        /// <summary>
        /// エミッタ拡張パラメータを削除します。
        /// ビューモデル, データモデル両方の処理を行います。
        /// </summary>
        /// <param name="emitterExtParamsViewModel">エミッタ拡張パラメータビューモデル</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool RemoveEmitterExtParams(EmitterExtParamsViewModel emitterExtParamsViewModel)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            // Remove the Emitter Extension Parameters model.
            proxy.RemoveEmitterExtParams(emitterExtParamsViewModel.DataModel);

            // Remove the Emitter Extension Parameters view model.
            this.Children.Remove(emitterExtParamsViewModel);

            return true;
        }

        /// <summary>
        /// エミッタ拡張パラメータのインデックスを取得します。
        /// </summary>
        /// <param name="emitterExtParamsViewModel">エミッタ拡張パラメータビューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        public void GetEmitterExtParamsIndex(
            EmitterExtParamsViewModel emitterExtParamsViewModel,
            out int viewModelIndex)
        {
            // ビューモデルのインデックスを取得
            viewModelIndex = this.Children.IndexOf(emitterExtParamsViewModel);
        }

        /// <summary>
        /// エミッタ拡張パラメータを指定インデックスに追加します。
        /// </summary>
        /// <param name="emitterExtParamsViewModel">エミッタ拡張パラメータビューモデル</param>
        /// <param name="viewModelIndex">ビューモデルのインデックス</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool InsertEmitterExtParams(
            EmitterExtParamsViewModel emitterExtParamsViewModel,
            int viewModelIndex)
        {
            var proxy = this.Proxy as EmitterDataProxy;
            if (proxy == null)
            {
                return false;
            }

            using (new SuppressNodeTypeOrdering(this))
            {
                // データモデルを追加
                proxy.AddEmitterExtParams(emitterExtParamsViewModel.DataModel);

                // ビューモデルを追加
                this.Children.Insert(viewModelIndex, emitterExtParamsViewModel);
            }

            return true;
        }

        #endregion // Fuction - エミッタ拡張パラメータ管理

        #region Function - エミッタ時間アニメ管理

        /// <summary>
        /// エミッタ時間アニメとカーブエディタウィンドウの接続を更新します。
        /// </summary>
        public void UpdateEmitterAnimation()
        {
            EmitterAnimationViewModel.IsInitializing = true;
            this.EmitterScaleViewModel.EmitterScaleBasicViewModel.UpdateCurveEditor();
            this.EmitterEmitterViewModel.EmitterEmitterTransformViewModel.UpdatePositionAnimation();
            this.EmitterEmitterViewModel.EmitterEmitterTransformViewModel.UpdateScaleAnimation();
            this.EmitterEmitterViewModel.EmitterEmitterTransformViewModel.UpdateRotationAnimation();
            this.EmitterEmissionViewModel.EmitterEmissionTimingViewModel.UpdateEmitTimeRateAnimation();
            this.EmitterEmissionViewModel.EmitterEmissionVelocityViewModel.UpdateOmniVelocityAnimation();
            this.EmitterEmissionViewModel.EmitterEmissionVelocityViewModel.UpdateOrientedDirectionalVelocityAnimation();
            this.EmitterEmissionViewModel.EmitterEmissionGravityViewModel.UpdateGravityAnimation();
            this.EmitterParticleViewModel.EmitterParticleLifeViewModel.UpdateLifeAnimation();
            this.EmitterEmitterViewModel.EmitterEmitterShapeViewModel.UpdateShapeScaleAnimation();
            EmitterAnimationViewModel.IsInitializing = false;
        }

        /// <summary>
        /// エミッタ時間アニメとカーブエディタウィンドウの接続を解除します。
        /// </summary>
        public void DisconnectEmitterAnimation()
        {
            EmitterAnimationViewModel.IsInitializing = true;
            this.EmitterScaleViewModel.EmitterScaleBasicViewModel.UpdateCurveEditor(true);
            this.EmitterEmitterViewModel.EmitterEmitterTransformViewModel.UpdatePositionAnimation(true);
            this.EmitterEmitterViewModel.EmitterEmitterTransformViewModel.UpdateScaleAnimation(true);
            this.EmitterEmitterViewModel.EmitterEmitterTransformViewModel.UpdateRotationAnimation(true);
            this.EmitterEmissionViewModel.EmitterEmissionTimingViewModel.UpdateEmitTimeRateAnimation(true);
            this.EmitterEmissionViewModel.EmitterEmissionVelocityViewModel.UpdateOmniVelocityAnimation(true);
            this.EmitterEmissionViewModel.EmitterEmissionVelocityViewModel.UpdateOrientedDirectionalVelocityAnimation(true);
            this.EmitterEmissionViewModel.EmitterEmissionGravityViewModel.UpdateGravityAnimation(true);
            this.EmitterParticleViewModel.EmitterParticleLifeViewModel.UpdateLifeAnimation(true);
            this.EmitterEmitterViewModel.EmitterEmitterShapeViewModel.UpdateShapeScaleAnimation(true);
            EmitterAnimationViewModel.IsInitializing = false;
        }

        #endregion

        /// <summary>
        /// データモデルプロキシを作成する.
        /// Create a data model proxy.
        /// This method is called in the constructor.
        /// If you need a specific type of data model proxy,
        /// override this method and return the desired data model proxy.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The created data model proxy.</returns>
        protected override DataModelProxy CreateDataModelProxy(DataModelBase dataModel)
        {
            return new EmitterDataProxy(dataModel);
        }

        /// <summary>
        /// Check if the property value has been modified.
        /// This method is and should only be used for comparing property values
        /// with their original value and default value.
        /// </summary>
        /// <param name="propertyName">The property name.</param>
        /// <param name="origValue">The original/default value.</param>
        /// <param name="currValue">The current value.</param>
        /// <returns>True if the value has been modified.</returns>
        protected override bool CheckPropertyValueModification(
            string propertyName,
            object origValue,
            object currValue)
        {
            if (propertyName != "EmitterList" &&
                propertyName != "FieldList" &&
                propertyName != "Emitters" &&
                propertyName != "ChildEmitters")
            {
                return base.CheckPropertyValueModification(propertyName, origValue, currValue);
            }

            // Check if the value is a sequence of Guid.
            // If it is, the property is probably a data model collection, in which
            // the elements are the Guid of the data models.
            if (origValue is IEnumerable<Guid>)
            {
                var origGuidList = (IEnumerable<Guid>)origValue;
                var currGuidList = DataModelProxy.ExtractDataModelGuids(currValue) as IEnumerable<Guid>;
                if (currGuidList == null)
                {
                    return false;
                }

                // Unlike the original implementation in ViewModelBase, we want to perform
                // ordered comparison for EmitterList and PreviewList because we want to
                // mark the property as modified when user drags and reorders the emitter
                // or preview nodes from the project tree view.
                return !currGuidList.SequenceEqual(origGuidList);
            }
            else if (origValue is IEnumerable<EmitterViewModel>)
            {
                var origGuidList = (IEnumerable<EmitterViewModel>)origValue;
                var currGuidList = DataModelProxy.ExtractDataModelGuids(currValue) as IEnumerable<EmitterViewModel>;
                if (currGuidList == null)
                {
                    return false;
                }

                // Unlike the original implementation in ViewModelBase, we want to perform
                // ordered comparison for EmitterList and PreviewList because we want to
                // mark the property as modified when user drags and reorders the emitter
                // or preview nodes from the project tree view.
                return !currGuidList.SequenceEqual(origGuidList);
            }

            return false;
        }

        /// <summary>
        /// Create reserved shader and add it to the emitter.
        /// </summary>
        /// <param name="parameter">The IExecutable parameter.</param>
        private void OnCreateReservedShader(object parameter)
        {
            // Do not create it if a reserved shader already exists.
            if (this.DataModel.ReservedShader != null)
            {
                return;
            }

            var userDataInfo = parameter as UserDataInfo;
            if (userDataInfo == null)
            {
                return;
            }

            // コマンドを実行.
            CommandManager.Execute(new CreateReservedShaderCommand(this, userDataInfo));
        }

        /// <summary>
        /// カスタムアクションを作成.
        /// </summary>
        /// <param name="parameter">パラメータ.</param>
        private void OnCreateCustomAction(object parameter)
        {
            // すでにカスタムアクションが存在する場合、作成しない。
            if (this.DataModel.CustomActionData != null)
            {
                return;
            }

            // コマンドを実行.
            CommandManager.Execute(new CreateCustomActionCommand(this));
        }

        /// <summary>
        /// エミッタ拡張パラメータを作成.
        /// </summary>
        /// <param name="parameter">パラメータ.</param>
        private void OnCreateEmitterExtParams(object parameter)
        {
            // すでにカスタムアクションが存在する場合、作成しない。
            if (this.DataModel.EmitterExtParams != null)
            {
                return;
            }

            // コマンドを実行.
            CommandManager.Execute(new CreateEmitterExtParamsCommand(this));
        }

        /// <summary>
        /// コンテキストメニューアイテムの状態を評価する.
        /// </summary>
        /// <param name="parameter">パラメータ.</param>
        private void OnEvaluateMenuItems(object parameter)
        {
            var customActionMenuExec = this.CustomActionMenuExecutable as AnonymousExecutable;
            if (customActionMenuExec != null)
            {
                // カスタムアクションのインスタンスが作成されている場合、無効にする.
                customActionMenuExec.IsEnabled = this.DataModel.CustomActionData == null;
            }

            var emitterExtParamsMenuExec = this.EmitterExtParamsMenuExecutable as AnonymousExecutable;
            if (emitterExtParamsMenuExec != null)
            {
                // エミッタ拡張パラメータのインスタンスが作成されている場合、無効にする.
                emitterExtParamsMenuExec.IsEnabled = this.DataModel.EmitterExtParams == null;
            }
        }

        /// <summary>
        /// Handle CollectionChanged event for the children collection.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The event arguments.</param>
        private void OnChildrenCollectionChanged(
            object sender,
            NotifyCollectionChangedEventArgs e)
        {
            this.OnPropertyChanged(() => this.DataModel.EmitterList);
            this.OnPropertyChanged(() => this.DataModel.FieldList);
            this.OnPropertyChanged(() => this.DataModel.ActiveFieldList);
            this.OnPropertyChanged(() => this.DataModel.ReservedShader);
            this.OnPropertyChanged(() => this.DataModel.CustomActionData);
            this.OnPropertyChanged(() => this.DataModel.EmitterExtParams);
        }

        /// <summary>
        /// Handle UserDataUpdated event from the user data manager.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnUserDataUpdated(
            object sender,
            UserDataUpdatedEventArgs e)
        {
            if (e.IsUserDataTypeUpdated(UpdatedUserDataTypes.GeneralUserPage) == false)
            {
                // The updated user data is not for emitter data, bail out.
                return;
            }

            this.UpdateUserData(false);
        }

        /// <summary>
        /// Helper method for updating user data.
        /// </summary>
        /// <param name="onlyUpdateViewModels">
        /// True to update only the view models for to the existing user data models.
        /// Otherwise, remove all the user data models and update them.
        /// </param>
        private void UpdateUserData(bool onlyUpdateViewModels)
        {
            // First remove all the user page view models.
            var userPages = this.Children.Where(ch => ch is UserPageViewModel).ToArray();
            foreach (var userPage in userPages)
            {
                this.Children.Remove(userPage);
            }

            if (onlyUpdateViewModels == false)
            {
                // Update user data models.
                (this.Proxy as EmitterDataProxy).UpdateUserData();
            }

            // Create user page view models with the new user data models.
            foreach (UserPageData page in this.DataModel.UserDataList)
            {
                var userPageViewModel = new UserPageViewModel(this, page);
                this.Children.Add(userPageViewModel);
            }
        }

        /// <summary>
        /// フィールドビューモデルを作成する.
        /// </summary>
        /// <param name="parent">親ビューモデル.</param>
        /// <param name="fieldData">データモデル.</param>
        /// <returns>
        /// 成功した場合、データモデルに対応するビューモデルを返す.
        /// 失敗した場合、nullを返す.
        /// </returns>
        private FieldViewModel CreateFieldViewModelByType(
            HierarchyViewModel parent,
            FieldDataBase fieldData)
        {
            return FieldViewModelFactory.CreateFieldViewModel(parent, fieldData);
        }
    }
}
