﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.ViewModels.Extenders;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the emitter texture group.
    /// </summary>
    public class EmitterTextureGroupViewModel : PropertyPageViewModel<EmitterTextureGroupData>, IModificationFlagOwner, IPropertyPageOwner
    {
        /// <summary>Extender for manipulating child property pages.</summary>
        private PropertyPageOwnerViewModelExtender propertyPageOwnerExtender;

        /// <summary>View model for the emitter texture 0.</summary>
        private EmitterTextureViewModel textureVM0 = null;

        /// <summary>View model for the emitter texture 1.</summary>
        private EmitterTextureViewModel textureVM1 = null;

        /// <summary>View model for the emitter texture 2.</summary>
        private EmitterTextureViewModel textureVM2 = null;

        /// <summary>View model for the emitter texture common settings.</summary>
        private EmitterTextureCommonViewModel textureVMCommon = null;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterTextureGroupViewModel(HierarchyViewModel parent, EmitterTextureGroupData dataModel)
            : base(parent, dataModel)
        {
            this.propertyPageOwnerExtender = new PropertyPageOwnerViewModelExtender(this);

            this.SetChildViewModel(
                ref this.textureVM0,
                new EmitterTextureViewModel(this, dataModel.Texture0));

            this.SetChildViewModel(
                ref this.textureVM1,
                new EmitterTextureViewModel(this, dataModel.Texture1));

            this.SetChildViewModel(
                ref this.textureVM2,
                new EmitterTextureViewModel(this, dataModel.Texture2));

            this.SetChildViewModel(
                ref this.textureVMCommon,
                new EmitterTextureCommonViewModel(this, dataModel.CommonSettings));

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
        }

        /// <summary>
        /// The default property page view model to use
        /// on the first time the workspace node is selected.
        /// </summary>
        public HierarchyViewModel DefaultPropertyPageViewModel
        {
            get { return this.textureVM0; }
        }

        /// <summary>
        /// Get or set the selected property page.
        /// </summary>
        public HierarchyViewModel SelectedPropertyPage
        {
            get
            {
                return this.propertyPageOwnerExtender.SelectedPropertyPage;
            }

            set
            {
                this.propertyPageOwnerExtender.SelectedPropertyPage = value;
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// Get the texture 0 view model.
        /// </summary>
        public EmitterTextureViewModel Texture0
        {
            get { return this.textureVM0; }
        }

        /// <summary>
        /// Get the texture 1 view model.
        /// </summary>
        public EmitterTextureViewModel Texture1
        {
            get { return this.textureVM1; }
        }

        /// <summary>
        /// Get the texture 2 view model.
        /// </summary>
        public EmitterTextureViewModel Texture2
        {
            get { return this.textureVM2; }
        }

        /// <summary>
        /// Get the texture common settings view model.
        /// </summary>
        public EmitterTextureCommonViewModel TextureCommon
        {
            get { return this.textureVMCommon; }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel
        {
            get;
            private set;
        }

        /// <summary>
        /// 渡されたEmitterTextureViewModelが何番目のテクスチャのものかを判定する
        /// </summary>
        /// <param name="textureViewModel">テクスチャViewModel</param>
        /// <returns>0～2,見つからなかったら-1</returns>
        public int GetTextureTabNumber(EmitterTextureViewModel textureViewModel)
        {
            if (this.Texture0 == textureViewModel)
            {
                return 0;
            }

            if (this.Texture1 == textureViewModel)
            {
                return 1;
            }

            if (this.Texture2 == textureViewModel)
            {
                return 2;
            }

            return -1;
        }
    }
}
