﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using EffectMaker.BusinessLogic.CurveEditorParameters;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterScaleBasicViewModel.
    /// </summary>
    public class EmitterScaleBasicViewModel : PropertyGroupViewModel<EmitterScaleBasicData>, IModificationFlagOwner
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterScaleBasicViewModel(
            HierarchyViewModel parent, EmitterScaleBasicData dataModel)
            : base(parent, dataModel)
        {
            this.BasicScaleAnimationViewModel = new EmitterAnimationViewModel(
                this, dataModel.BasicScaleAnimation, InitialKeyValues.ScaleBasic);
            this.Children.Add(this.BasicScaleAnimationViewModel);

            this.OnShowCurveEditorDialog = new AnonymousExecutable(() =>
            {
                this.UpdateCurveEditor();
                WorkspaceRootViewModel.Instance.ShowCurveEditor();
            });

            // 基準値のエミッタ時間アニメの設定
            this.BasicScaleAnimationViewModel.PropertyChanged += (s, e) =>
            {
                this.OnPropertyChanged(() => this.IsBasicScaleCurveModified);
                if (IsRaisedProperty(e, () => this.BasicScaleAnimationViewModel.EnableAnimation))
                {
                    this.OnPropertyChanged(() => this.EnableAnimation);
                    this.UpdateCurveEditor();
                }
            };
            this.BasicScaleAnimationViewModel.SetDefaultSnapSetting("EmitterScaleBasicScale", snapValue: 3);
            this.UpdateCurveEditor();

            var mmfvm = new MultipleModificationFlagsViewModel(this);
            var basicScaleParamsProperties = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.ScaleBasisValue),
                ViewModelBase.NameOf(() => this.IsBasicScaleCurveModified));
            mmfvm.SetPropertyDictionary("ScaleBasisValueParams", basicScaleParamsProperties);
            mmfvm.SetModificationDictionary("ScaleBasisValueParams", new IModificationPropertyOwner[] { this.BasicScaleAnimationViewModel });

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = mmfvm;
        }

        /// <summary>
        /// 基準値のエミッタ時間アニメが変更されたか
        /// </summary>
        public bool IsBasicScaleCurveModified
        {
            get { return this.BasicScaleAnimationViewModel.IsModified;  }
        }

        /// <summary>
        /// カーブエディタを表示するExecutableを取得します。
        /// </summary>
        public IExecutable OnShowCurveEditorDialog { get; private set; }

        /// <summary>
        /// Gets or sets a value indicating whether enable animation.
        /// </summary>
        public bool EnableAnimation
        {
            get
            {
                return this.BasicScaleAnimationViewModel.EnableAnimation;
            }

            set
            {
                this.BasicScaleAnimationViewModel.EnableAnimation = value;
                this.UpdateCurveEditor();
                if (value)
                {
                    WorkspaceRootViewModel.Instance.ShowCurveEditor();
                }
            }
        }

        /// <summary>
        /// BasicScaleAnimationViewModel.
        /// </summary>
        public EmitterAnimationViewModel BasicScaleAnimationViewModel { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// プロパティ変更通知を強制的に発行します。
        /// </summary>
        public override void FirePropertyChanges()
        {
            base.FirePropertyChanges();
            this.UpdateCurveEditor();
        }

        /// <summary>
        /// カーブエディタとの接続状態を更新
        /// </summary>
        /// <param name="forceDisconnect">強制的に切断する時はtrue.</param>
        public void UpdateCurveEditor(bool forceDisconnect = false)
        {
            var editorParam = new EmitterAnimationEditorParameter
            {
                NormalizeAt = 1,
                LabelDigit = 2,
                LabelPrefix = "x",
                MaxLimit = 1000000.0f,
                MinLimit = 0.0f,
                DefaultZeroPin = -1,
                DefaultValue = InitialKeyValues.ScaleBasic[0],
                Channels = new List<string>
                    {
                        "X",
                        "Y",
                        "Z",
                    },
                DataContext = this.BasicScaleAnimationViewModel,
                AnimationName = "EmitterScaleBasic"
            };

            if (!forceDisconnect)
            {
                WorkspaceRootViewModel.Instance.ConnectCurveEditor(editorParam);
            }
            else
            {
                WorkspaceRootViewModel.Instance.DisconnectCurveEditor(editorParam);
            }
        }
    }
}
