﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Attributes;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterRotationBasicViewModel.
    /// </summary>
    public class EmitterRotationBasicViewModel : PropertyGroupViewModel<EmitterRotationBasicData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "RotationOrderVisibility",
            "AttenuationVisibility",
            "AttenuationInvisibility",
        };

        /// <summary>
        /// 回転順序の表示状態.
        /// </summary>
        private bool rotationOrderVisibility = false;

        /// <summary>
        /// 減衰率の表示状態.
        /// </summary>
        private bool attenuationVisibility = false;

        /// <summary>
        /// 親のビューモデル
        /// </summary>
        private EmitterRotationViewModel parentViewModel = null;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterRotationBasicViewModel(
            HierarchyViewModel parent, EmitterRotationBasicData dataModel)
            : base(parent, dataModel)
        {
            this.RotationTypeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterRotationTypeYZX, 4),
                new KeyValuePair<string, object>(Properties.Resources.EmitterRotationTypeXYZ, 5),
                new KeyValuePair<string, object>(Properties.Resources.EmitterRotationTypeZXY, 6),
            };

            this.parentViewModel = (EmitterRotationViewModel)this.Parent;

            // ファイル読み込み時にUI表示フラグが更新されるようにする
            this.parentViewModel.EnableRotationXGroup = this.EnableRotationAxisX;
            this.parentViewModel.EnableRotationYGroup = this.EnableRotationAxisY;
            this.parentViewModel.EnableRotationZGroup = this.EnableRotationAxisZ;
            this.RotationOrderVisibility = this.ShouldShowRotationOrder();
            this.AttenuationVisibility = this.ShouldShowAttenuation();

            var enableRotationAxisesProperties = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.EnableRotationAxisX),
                ViewModelBase.NameOf(() => dataModel.EnableRotationAxisY),
                ViewModelBase.NameOf(() => dataModel.EnableRotationAxisZ));

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            var mmfvm = new MultipleModificationFlagsViewModel(this);
            mmfvm.SetPropertyDictionary("EnableRotationAxises", enableRotationAxisesProperties);
            this.ModificationFlagViewModel = mmfvm;

            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// 回転軸Xが有効かどうか.
        /// </summary>
        [UseDataModelOriginalValue]
        public bool EnableRotationAxisX
        {
            get
            {
                return this.GetDataModelValue(() => this.EnableRotationAxisX);
            }

            set
            {
                this.SetDataModelValue(value, () => this.EnableRotationAxisX);

                // プロパティ変更通知.
                this.parentViewModel.EnableRotationXGroup = value;
                this.RotationOrderVisibility = this.ShouldShowRotationOrder();
                this.AttenuationVisibility = this.ShouldShowAttenuation();
            }
        }

        /// <summary>
        /// 回転軸Yが有効かどうか.
        /// </summary>
        [UseDataModelOriginalValue]
        public bool EnableRotationAxisY
        {
            get
            {
                return this.GetDataModelValue(() => this.EnableRotationAxisY);
            }

            set
            {
                this.SetDataModelValue(value, () => this.EnableRotationAxisY);

                // プロパティ変更通知.
                this.parentViewModel.EnableRotationYGroup = value;
                this.RotationOrderVisibility = this.ShouldShowRotationOrder();
                this.AttenuationVisibility = this.ShouldShowAttenuation();
            }
        }

        /// <summary>
        /// 回転軸Zが有効かどうか.
        /// </summary>
        [UseDataModelOriginalValue]
        public bool EnableRotationAxisZ
        {
            get
            {
                return this.GetDataModelValue(() => this.EnableRotationAxisZ);
            }

            set
            {
                this.SetDataModelValue(value, () => this.EnableRotationAxisZ);

                // プロパティ変更通知.
                this.parentViewModel.EnableRotationZGroup = value;
                this.RotationOrderVisibility = this.ShouldShowRotationOrder();
                this.AttenuationVisibility = this.ShouldShowAttenuation();
            }
        }

        /// <summary>
        /// 回転順序グループの表示状態.
        /// </summary>
        public bool RotationOrderVisibility
        {
            get { return this.rotationOrderVisibility; }
            set { this.SetValue(ref this.rotationOrderVisibility, value); }
        }

        /// <summary>
        /// 減衰率の表示状態.
        /// </summary>
        public bool AttenuationVisibility
        {
            get { return this.attenuationVisibility; }
            set { this.SetValue(ref this.attenuationVisibility, value); }
        }

        /// <summary>
        /// 回転タイプの項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> RotationTypeItems { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// 回転の状態を復元
        /// </summary>
        public override void FirePropertyChanges()
        {
            this.EnableRotationAxisX = this.EnableRotationAxisX;
            this.EnableRotationAxisY = this.EnableRotationAxisY;
            this.EnableRotationAxisZ = this.EnableRotationAxisZ;

            base.FirePropertyChanges();
        }

        /// <summary>
        /// 回転順序を表示するか.
        /// </summary>
        /// <returns>表示する場合、trueを返す.</returns>
        private bool ShouldShowRotationOrder()
        {
            int enableCount = 0;
            enableCount += this.EnableRotationAxisX ? 1 : 0;
            enableCount += this.EnableRotationAxisY ? 1 : 0;
            enableCount += this.EnableRotationAxisZ ? 1 : 0;

            return enableCount >= 2;
        }

        /// <summary>
        /// 減衰率を表示するかどうか.
        /// </summary>
        /// <returns>表示する場合、trueを返す.</returns>
        private bool ShouldShowAttenuation()
        {
            int enableCount = 0;
            enableCount += this.EnableRotationAxisX ? 1 : 0;
            enableCount += this.EnableRotationAxisY ? 1 : 0;
            enableCount += this.EnableRotationAxisZ ? 1 : 0;

            return enableCount > 0;
        }
    }
}
