﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using EffectMaker.BusinessLogic.CurveEditorParameters;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterEmitterTransformViewModel.
    /// </summary>
    public class EmitterEmitterTransformViewModel : PropertyGroupViewModel<EmitterEmitterTransformData>, IModificationFlagOwner
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterEmitterTransformViewModel(
            HierarchyViewModel parent, EmitterEmitterTransformData dataModel)
            : base(parent, dataModel)
        {
            this.EmitterScaleAnimationViewModel = new EmitterAnimationViewModel(
                this, dataModel.EmitterScaleAnimation, InitialKeyValues.EmitterScale);
            this.Children.Add(this.EmitterScaleAnimationViewModel);

            this.EmitterRotationAnimationViewModel = new EmitterAnimationViewModel(
                this, dataModel.EmitterRotationAnimation, InitialKeyValues.EmitterRotation);
            this.Children.Add(this.EmitterRotationAnimationViewModel);

            this.EmitterPositionAnimationViewModel = new EmitterAnimationViewModel(
                this, dataModel.EmitterPositionAnimation, InitialKeyValues.EmitterPosition);
            this.Children.Add(this.EmitterPositionAnimationViewModel);

            this.OnShowCurveEditorDialog = new AnonymousExecutable((p) =>
            {
                if (string.Equals(p, "Position"))
                {
                    this.UpdatePositionAnimation();
                }
                else if (string.Equals(p, "Rotation"))
                {
                    this.UpdateRotationAnimation();
                }
                else if (string.Equals(p, "Scale"))
                {
                    this.UpdateScaleAnimation();
                }

                WorkspaceRootViewModel.Instance.ShowCurveEditor();
            });

            // 位置
            this.EmitterPositionAnimationViewModel.PropertyChanged += (s, e) =>
            {
                this.OnPropertyChanged(() => this.IsPositionCurveModified);
                if (IsRaisedProperty(e, () => this.EmitterPositionAnimationViewModel.EnableAnimation))
                {
                    this.OnPropertyChanged(() => this.EnablePositionAnimation);
                    this.OnPropertyChanged(() => this.IsNoAnimationEnabled);
                    this.UpdatePositionAnimation();
                }
            };
            this.EmitterPositionAnimationViewModel.SetDefaultSnapSetting("EmitterTransformPosition", snapValue: 3);
            this.UpdatePositionAnimation();

            // スケール
            this.EmitterScaleAnimationViewModel.PropertyChanged += (s, e) =>
            {
                this.OnPropertyChanged(() => this.IsScaleCurveModified);
                if (IsRaisedProperty(e, () => this.EmitterScaleAnimationViewModel.EnableAnimation))
                {
                    this.OnPropertyChanged(() => this.EnableScaleAnimation);
                    this.OnPropertyChanged(() => this.IsNoAnimationEnabled);
                    this.UpdateScaleAnimation();
                }
            };
            this.EmitterScaleAnimationViewModel.SetDefaultSnapSetting("EmitterTransformScale", snapValue: 3);
            this.UpdateScaleAnimation();

            // 回転
            this.EmitterRotationAnimationViewModel.PropertyChanged += (s, e) =>
            {
                this.OnPropertyChanged(() => this.IsRotationCurveModified);
                if (IsRaisedProperty(e, () => this.EmitterRotationAnimationViewModel.EnableAnimation))
                {
                    this.OnPropertyChanged(() => this.EnableRotationAnimation);
                    this.OnPropertyChanged(() => this.IsNoAnimationEnabled);
                    this.UpdateRotationAnimation();
                }
            };
            this.EmitterRotationAnimationViewModel.SetDefaultSnapSetting("EmitterTransformRotation", snapValue: 4);
            this.UpdateRotationAnimation();

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            var mmfvm = new MultipleModificationFlagsViewModel(this);

            // 位置
            var emitterPositionParamsProperties = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.EmitterPosition),
                ViewModelBase.NameOf(() => dataModel.EmitterPositionRandom),
                ViewModelBase.NameOf(() => this.IsPositionCurveModified));
            mmfvm.SetPropertyDictionary("EmitterPositionParams", emitterPositionParamsProperties);
            mmfvm.SetModificationDictionary("EmitterPositionParams", new IModificationPropertyOwner[] { this.EmitterPositionAnimationViewModel });

            // 回転
            var emitterRotationParamsProperties = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.EmitterRotation),
                ViewModelBase.NameOf(() => dataModel.EmitterRotationRandom),
                ViewModelBase.NameOf(() => this.IsRotationCurveModified));
            mmfvm.SetPropertyDictionary("EmitterRotationParams", emitterRotationParamsProperties);
            mmfvm.SetModificationDictionary("EmitterRotationParams", new IModificationPropertyOwner[] { this.EmitterRotationAnimationViewModel });

            // スケール
            var emitterScaleParamsProperties = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.EmitterScale),
                ViewModelBase.NameOf(() => this.IsScaleCurveModified));
            mmfvm.SetPropertyDictionary("EmitterScaleParams", emitterScaleParamsProperties);
            mmfvm.SetModificationDictionary("EmitterScaleParams", new IModificationPropertyOwner[] { this.EmitterScaleAnimationViewModel });

            this.ModificationFlagViewModel = mmfvm;
        }

        /// <summary>
        /// 位置のエミッタ時間アニメが変更されたか
        /// </summary>
        public bool IsPositionCurveModified
        {
            get { return this.EmitterPositionAnimationViewModel.IsModified; }
        }

        /// <summary>
        /// 回転のエミッタ時間アニメが変更されたか
        /// </summary>
        public bool IsRotationCurveModified
        {
            get { return this.EmitterRotationAnimationViewModel.IsModified; }
        }

        /// <summary>
        /// スケールのエミッタ時間アニメが変更されたか
        /// </summary>
        public bool IsScaleCurveModified
        {
            get { return this.EmitterScaleAnimationViewModel.IsModified; }
        }

        /// <summary>
        /// 位置エミッタ時間アニメの有効/無効を取得または設定します。
        /// </summary>
        public bool EnablePositionAnimation
        {
            get
            {
                return this.EmitterPositionAnimationViewModel.EnableAnimation;
            }

            set
            {
                this.EmitterPositionAnimationViewModel.EnableAnimation = value;
                this.OnPropertyChanged(() => this.IsNoAnimationEnabled);
                this.UpdatePositionAnimation();
                if (value)
                {
                    WorkspaceRootViewModel.Instance.ShowCurveEditor();
                }
            }
        }

        /// <summary>
        /// 回転エミッタ時間アニメの有効/無効を取得または設定します。
        /// </summary>
        public bool EnableRotationAnimation
        {
            get
            {
                return this.EmitterRotationAnimationViewModel.EnableAnimation;
            }

            set
            {
                this.EmitterRotationAnimationViewModel.EnableAnimation = value;
                this.OnPropertyChanged(() => this.IsNoAnimationEnabled);
                this.UpdateRotationAnimation();
                if (value)
                {
                    WorkspaceRootViewModel.Instance.ShowCurveEditor();
                }
            }
        }

        /// <summary>
        /// スケールエミッタ時間アニメの有効/無効を取得または設定します。
        /// </summary>
        public bool EnableScaleAnimation
        {
            get
            {
                return this.EmitterScaleAnimationViewModel.EnableAnimation;
            }

            set
            {
                this.EmitterScaleAnimationViewModel.EnableAnimation = value;
                this.OnPropertyChanged(() => this.IsNoAnimationEnabled);
                this.UpdateScaleAnimation();
                if (value)
                {
                    WorkspaceRootViewModel.Instance.ShowCurveEditor();
                }
            }
        }

        /// <summary>
        /// 1つでも有効になっているエミッタ時間アニメがあるか否かを取得します。
        /// </summary>
        public bool IsNoAnimationEnabled
        {
            get
            {
                return !(this.EnablePositionAnimation ||
                    this.EnableRotationAnimation ||
                    this.EnableScaleAnimation);
            }
        }

        /// <summary>
        /// EmitterScaleAnimationViewModel.
        /// </summary>
        public EmitterAnimationViewModel EmitterScaleAnimationViewModel { get; private set; }

        /// <summary>
        /// EmitterRotationAnimationViewModel.
        /// </summary>
        public EmitterAnimationViewModel EmitterRotationAnimationViewModel { get; private set; }

        /// <summary>
        /// EmitterPositionAnimationViewModel.
        /// </summary>
        public EmitterAnimationViewModel EmitterPositionAnimationViewModel { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// カーブエディタを表示するExecutableを取得します。
        /// </summary>
        public IExecutable OnShowCurveEditorDialog { get; private set; }

        /// <summary>
        /// プロパティ変更通知を強制的に発行します。
        /// </summary>
        public override void FirePropertyChanges()
        {
            base.FirePropertyChanges();
            this.UpdatePositionAnimation();
            this.UpdateRotationAnimation();
            this.UpdateScaleAnimation();
        }

        /// <summary>
        /// 位置アニメのエディタ接続更新
        /// </summary>
        /// <param name="forceDisconnect">強制的に切断する時はtrue.</param>
        public void UpdatePositionAnimation(bool forceDisconnect = false)
        {
            var editorParam = new EmitterAnimationEditorParameter
            {
                NormalizeAt = 1.0f,
                LabelDigit = 2,
                MaxLimit = 1000000.0f,
                MinLimit = -1000000.0f,
                DefaultValue = InitialKeyValues.EmitterPosition[0],
                DefaultZeroPin = 3,
                Channels = new List<string> { "X", "Y", "Z", },
                DataContext = this.EmitterPositionAnimationViewModel,
                AnimationName = "EmitterTransformPosition",
            };

            if (!forceDisconnect)
            {
                WorkspaceRootViewModel.Instance.ConnectCurveEditor(editorParam);
            }
            else
            {
                WorkspaceRootViewModel.Instance.DisconnectCurveEditor(editorParam);
            }
        }

        /// <summary>
        /// スケールアニメのエディタ接続更新
        /// </summary>
        /// <param name="forceDisconnect">強制的に切断する時はtrue.</param>
        public void UpdateScaleAnimation(bool forceDisconnect = false)
        {
            var editorParam = new EmitterAnimationEditorParameter
            {
                NormalizeAt = 1,
                LabelDigit = 2,
                LabelPrefix = "x",
                MaxLimit = 1000000.0f,
                MinLimit = 0.0f,
                DefaultValue = InitialKeyValues.EmitterScale[0],
                DefaultZeroPin = -1,
                Channels = new List<string> { "X", "Y", "Z", },
                DataContext = this.EmitterScaleAnimationViewModel,
                AnimationName = "EmitterTransformScale",
            };

            if (!forceDisconnect)
            {
                WorkspaceRootViewModel.Instance.ConnectCurveEditor(editorParam);
            }
            else
            {
                WorkspaceRootViewModel.Instance.DisconnectCurveEditor(editorParam);
            }
        }

        /// <summary>
        /// 回転アニメのエディタ接続更新
        /// </summary>
        /// <param name="forceDisconnect">強制的に切断する時はtrue.</param>
        public void UpdateRotationAnimation(bool forceDisconnect = false)
        {
            var editorParam = new EmitterAnimationEditorParameter
            {
                NormalizeAt = 180.0f,
                LabelDigit = 2,
                MaxLimit = 1000000.0f,
                MinLimit = -1000000.0f,
                DefaultValue = InitialKeyValues.EmitterRotation[0],
                DefaultZeroPin = 3,
                Channels = new List<string> { "X", "Y", "Z", },
                DataContext = this.EmitterRotationAnimationViewModel,
                AnimationName = "EmitterTransformRotation",
            };

            if (!forceDisconnect)
            {
                WorkspaceRootViewModel.Instance.ConnectCurveEditor(editorParam);
            }
            else
            {
                WorkspaceRootViewModel.Instance.DisconnectCurveEditor(editorParam);
            }
        }
    }
}
