﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Generic;
using EffectMaker.BusinessLogic.CurveEditorParameters;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Attributes;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterEmissionTimingViewModel.
    /// </summary>
    public class EmitterEmissionTimingViewModel : PropertyGroupViewModel<EmitterEmissionTimingData>, IModificationFlagOwner
    {
        /// <summary>
        /// フラグが無視するプロパティ
        /// </summary>
        private static readonly string[] ignoreProperties = { "EnabledEmitRateTiming" };

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterEmissionTimingViewModel(
            HierarchyViewModel parent, EmitterEmissionTimingData dataModel)
            : base(parent, dataModel)
        {
            this.EmitterEmitTimeRateAnimationViewModel = new EmitterAnimationViewModel(
                this, dataModel.EmitTimeRateAnimation, InitialKeyValues.EmitTimeRate);
            this.Children.Add(this.EmitterEmitTimeRateAnimationViewModel);

            this.EmissionTimingIntervalTypeItems = new []
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterEmissionTypeTime, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterEmissionTypeDistance, 1),
            };

            this.OnShowCurveEditorDialog = new AnonymousExecutable(() =>
            {
                this.UpdateEmitTimeRateAnimation();
                WorkspaceRootViewModel.Instance.ShowCurveEditor();
            });

            // 放出レートのエミッタ時間アニメの設定
            this.EmitterEmitTimeRateAnimationViewModel.PropertyChanged += (s, e) =>
            {
                this.OnPropertyChanged(() => this.IsEmitTimeRateCurveModified);
                if (IsRaisedProperty(e, () => this.EmitterEmitTimeRateAnimationViewModel.EnableAnimation))
                {
                    this.OnPropertyChanged(() => this.EnableEmitTimeRateAnimation);
                    this.UpdateEmitTimeRateAnimation();
                }
            };
            this.EmitterEmitTimeRateAnimationViewModel.SetDefaultSnapSetting("EmitterEmissionEmitTimeRate", snapValue: 3);
            this.UpdateEmitTimeRateAnimation();

            var mmfvm = new MultipleModificationFlagsViewModel(this);
            var emitterEmitTimeRateParamsProperties = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.EmitTimeEmissionRate),
                ViewModelBase.NameOf(() => this.IsEmitTimeRateCurveModified));
            mmfvm.SetPropertyDictionary("EmitterEmitTimeEmissionRateParams", emitterEmitTimeRateParamsProperties);
            mmfvm.SetModificationDictionary("EmitterEmitTimeEmissionRateParams", new IModificationPropertyOwner[] { this.EmitterEmitTimeRateAnimationViewModel });

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = mmfvm;
            this.ModificationFlagViewModel.AddIgnoreProperties(ignoreProperties);
        }

        /// <summary>
        /// 放出レートのエミッタ時間アニメが変更されたかどうか
        /// </summary>
        public bool IsEmitTimeRateCurveModified
        {
            get { return this.EmitterEmitTimeRateAnimationViewModel.IsModified; }
        }

        /// <summary>
        /// 放出間隔タイプの項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> EmissionTimingIntervalTypeItems { get; private set; }

        /// <summary>
        /// 放出間隔タイプ
        /// </summary>
        [UseDataModelOriginalValue]
        public int EmitDistUnitType
        {
            get
            {
                return this.GetDataModelValue(() => this.EmitDistUnitType);
            }

            set
            {
                this.SetDataModelValue(value, () => this.EmitDistUnitType);
                this.RaiseEmitRateTimingEnable();
            }
        }

        /// <summary>
        /// 放出レートアニメの有効/無効を取得または設定します。
        /// </summary>
        public bool EnableEmitTimeRateAnimation
        {
            get
            {
                return this.EmitterEmitTimeRateAnimationViewModel.EnableAnimation;
            }

            set
            {
                this.EmitterEmitTimeRateAnimationViewModel.EnableAnimation = value;
                this.UpdateEmitTimeRateAnimation();
                if (value)
                {
                    WorkspaceRootViewModel.Instance.ShowCurveEditor();
                }
            }
        }

        /// <summary>
        /// このエミッタがチャイルドならtrue、そうじゃなかったらfalse.
        /// </summary>
        public bool IsChild
        {
            get
            {
                var evm = this.Parent.Parent.Parent as EmitterViewModel;
                if (evm != null)
                {
                    return true;
                }

                return false;
            }
        }

        /// <summary>
        /// このエミッタがチャイルドならfalse、そうじゃなかったらtrue.
        /// </summary>
        public bool IsParent
        {
            get { return !this.IsChild; }
        }

        /// <summary>
        /// 放出レートが有効か否かを取得します。
        /// </summary>
        public bool EnabledEmitRateTiming
        {
            get
            {
                var emitter = this.FindNearestParentOfType<EmitterViewModel>();
                var emitterShape = emitter.EmitterEmitterViewModel.EmitterEmitterShapeViewModel;
                var shapeType = emitterShape.EmitterType;
                bool disable = (shapeType == 2 || shapeType == 5 || shapeType == 6 || shapeType == 13 || shapeType == 15)
                    && emitterShape.EmitterShapeEmissionType == 0;
                return !disable;
            }
        }

        /// <summary>
        /// EmitterEmitTimeRateAnimationViewModel.
        /// </summary>
        public EmitterAnimationViewModel EmitterEmitTimeRateAnimationViewModel { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// カーブエディタを表示するExecutableを取得します。
        /// </summary>
        public IExecutable OnShowCurveEditorDialog { get; private set; }

        /// <summary>
        /// プロパティ変更通知を強制的に発行します。
        /// </summary>
        public override void FirePropertyChanges()
        {
            base.FirePropertyChanges();
            this.UpdateEmitTimeRateAnimation();
        }

        /// <summary>
        /// 放出レートの有効無効を更新します。
        /// </summary>
        public void RaiseEmitRateTimingEnable()
        {
            this.OnPropertyChanged(() => this.EnabledEmitRateTiming);
            EmitterAnimationViewModel.IsInitializing = true;
            this.UpdateEmitTimeRateAnimation();
            EmitterAnimationViewModel.IsInitializing = false;
        }

        /// <summary>
        /// 放出レートアニメのエディタ接続更新
        /// </summary>
        /// <param name="forceDisconnect">強制的に切断する時はtrue.</param>
        public void UpdateEmitTimeRateAnimation(bool forceDisconnect = false)
        {
            var editorParam = new EmitterAnimationEditorParameter
            {
                NormalizeAt = 1.0f,
                LabelDigit = 2,
                MaxLimit = 1000000.0f,
                MinLimit = 0.0f,
                DefaultValue = InitialKeyValues.EmitTimeRate[0],
                DefaultZeroPin = -1,
                Channels = new List<string>
                    {
                        "X",
                    },
                DataContext = this.EnabledEmitRateTiming && this.EmitDistUnitType == 0 ?
                    this.EmitterEmitTimeRateAnimationViewModel : null,
                AnimationName = "EmitterEmissionRate",
            };

            if (!forceDisconnect)
            {
                WorkspaceRootViewModel.Instance.ConnectCurveEditor(editorParam);
            }
            else
            {
                WorkspaceRootViewModel.Instance.DisconnectCurveEditor(editorParam);
            }
        }
    }
}
