﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.ProjectConfig;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Disposables;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Properties;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// エミッタカスタムシェーダビューモデル
    /// </summary>
    public class EmitterCustomShaderBasicViewModel : PropertyGroupViewModel<EmitterCustomShaderData>, IModificationFlagOwner
    {
        /// <summary>The flag indicating whether the initialization of this instance is finished or not.</summary>
        private bool initialized = false;

        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "EnableShaderAssembly",
            "EnableStreamOutSource",
            "EnableStreamOutAssembly",
            "IsVfx",
            "LatencyInfo",
        };

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        /// <param name="groupViewModel">グループViewModel</param>
        public EmitterCustomShaderBasicViewModel(
            HierarchyViewModel parent,
            EmitterCustomShaderData dataModel,
            EmitterCustomShaderGroupViewModel groupViewModel)
            : base(parent, dataModel)
        {
            this.OnCompileShaderExecutable =
                new AnonymousExecutable(this.OnCompileShader);

            this.OnGenerateShaderSourceExecutable =
                new AnonymousExecutable(this.OnGenerateShaderSource);

            this.OnGenerateShaderAssemblyExecutable =
                new AnonymousExecutable(this.OnGenerateShaderAssembly);

            this.GroupViewModel = groupViewModel;

            this.SetupSettingItems();

            // spec切替時に、EnableShaderAssemblyにbindしたUIImageButtonを更新する.
            SpecManager.CurrentSpecChanged += this.OnSpecChanged;

            // プロセスタイプ変更イベントを登録
            var emitterViewModel = ViewModelBase.GetParent<EmitterViewModel>(this);
            emitterViewModel.PropertyChanged += (s, e) =>
            {
                if (e.PropertyName == "ProcessType")
                {
                    this.OnPropertyChanged(() => this.EnableStreamOutSource);
                    this.OnPropertyChanged(() => this.EnableStreamOutAssembly);
                }
            };

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);

            this.initialized = true;
        }

        /// <summary>
        /// グループViewModel
        /// </summary>
        public EmitterCustomShaderGroupViewModel GroupViewModel { get; private set; }

        /// <summary>
        /// Executable for compiling shader in the emitter.
        /// </summary>
        public IExecutable OnCompileShaderExecutable { get; private set; }

        /// <summary>
        /// Executable for generating shader source code in the emitter.
        /// </summary>
        public IExecutable OnGenerateShaderSourceExecutable { get; private set; }

        /// <summary>
        /// Executable for generating shader assembly code in the emitter.
        /// </summary>
        public IExecutable OnGenerateShaderAssemblyExecutable { get; private set; }

        /// <summary>
        /// Get or set the available custom setting name and index.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> AvailableSettingItems { get; set; }

        /// <summary>
        /// Get or set the selected custom setting index.
        /// </summary>
        [UseDataModelOriginalValue]
        public int SelectedSettingIndex
        {
            get
            {
                return this.GetDataModelValue(() => this.SelectedSettingIndex);
            }

            set
            {
                this.SetDataModelValue(value, () => this.SelectedSettingIndex);
                this.GroupViewModel.RaiseSelectedIndexChanged();
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// シェーダアセンブリボタンの有効状態を取得します。
        /// プリコンパイルなスペックの場合のみ、アセンブリボタンを使用可能にします。
        /// </summary>
        public bool EnableShaderAssembly
        {
            get { return SpecManager.CurrentSpec.ShaderConversionOption.TargetIsPreCompile; }
        }

        /// <summary>
        /// コンピュートシェーダのソースボタンの有効状態を取得します。
        /// タイプがSOの場合のみ、アセンブリボタンを使用可能にします。
        /// </summary>
        public bool EnableStreamOutSource
        {
            get
            {
                var emitterViewModel = ViewModelBase.GetParent<EmitterViewModel>(this);
                return (emitterViewModel.ProcessType == 2);
            }
        }

        /// <summary>
        /// コンピュートシェーダのアセンブリボタンの有効状態を取得します。
        /// プリコンパイルなスペックの場合且つタイプがSOの場合のみ、アセンブリボタンを使用可能にします。
        /// </summary>
        public bool EnableStreamOutAssembly
        {
            get
            {
                var emitterViewModel = ViewModelBase.GetParent<EmitterViewModel>(this);
                return ((emitterViewModel.ProcessType == 2) && this.EnableShaderAssembly);
            }
        }

        /// <summary>
        /// シェーダ定義1の有効状態を取得します。
        /// </summary>
        public bool EnabledShaderDefinition1
        {
            get
            {
                var drawPathInfo = this.GetCurrentDrawPath();
                if (drawPathInfo == null)
                {
                    return false;
                }

                return !string.IsNullOrEmpty(drawPathInfo.ShaderCompileDef1);
            }
        }

        /// <summary>
        /// シェーダ定義2の有効状態を取得します。
        /// </summary>
        public bool EnabledShaderDefinition2
        {
            get
            {
                var drawPathInfo = this.GetCurrentDrawPath();
                if (drawPathInfo == null)
                {
                    return false;
                }

                return !string.IsNullOrEmpty(drawPathInfo.ShaderCompileDef2);
            }
        }

        /// <summary>
        /// シェーダの負荷情報を取得または設定します。
        /// </summary>
        public string LatencyInfo { get; set; }

        /// <summary>
        /// ソース・アセンブリの表示ボタンの動作でプリプロセスを有効にするか否かを取得または設定します。
        /// </summary>
        public bool EnablePreprocess { get; set; }

        /// <summary>
        /// VFX系のスペックか否かを取得します。
        /// </summary>
        public bool IsVfx
        {
            get { return SpecManager.CurrentSpec.BinaryHeader == "VFXB"; }
        }

        /// <summary>
        /// Update child view models with the current data model.
        /// This method is usually called when data model is modified, thus some child
        /// view models might need to be created or removed according to the data model.
        /// </summary>
        public override void UpdateChildViewModels()
        {
            // First remove all the child view models.
            this.GroupViewModel.Children.Clear();

            // Create the key value pair array that contains the name of the custom shader setting
            // and it's view model.
            int itemCount = 0;
            if (CustomShaderUserDataManager.Definition != null)
            {
                itemCount = CustomShaderUserDataManager.Definition.CustomShaderDefinitions.Count;
            }

            // Add the custom shader setting items.
            if (itemCount > 0)
            {
                int index = 0;
                foreach (CustomShaderDefinition def in
                    CustomShaderUserDataManager.Definition.CustomShaderDefinitions)
                {
                    var childViewModel = new EmitterCustomShaderSettingDataViewModel(
                        this,
                        this.DataModel.Settings[index],
                        index);

                    // Add a view model for the data model.
                    this.GroupViewModel.Children.Add(childViewModel);

                    // Advance the index.
                    ++index;
                }
            }

            base.UpdateChildViewModels();
        }

        /// <summary>
        /// Disposes the instance.
        /// </summary>
        public override void Dispose()
        {
            base.Dispose();
            SpecManager.CurrentSpecChanged -= this.OnSpecChanged;
        }

        /// <summary>
        /// シェーダ定義に関するプロパティを更新します。
        /// </summary>
        public void RaiseShaderDefinitionProperty()
        {
            this.OnPropertyChanged(() => this.EnabledShaderDefinition1);
            this.OnPropertyChanged(() => this.EnabledShaderDefinition2);
        }

        /// <summary>
        /// オプションを変更したときの処理を行います.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The event arguments.</param>
        protected override void OnOptionChanged(object sender, EventArgs e)
        {
            base.OnOptionChanged(sender, e);

            if (ViewModelBase.GetParent<EmitterSetViewModel>(this) == null)
            {
                // 削除したエミッタや、新規作成してアンドゥしたエミッタがある場合、親が不在のエミッタビューモデルに対してイベントハンドラが走る
                return;
            }

            this.SetupSettingItems();
            this.RaiseShaderDefinitionProperty();
        }

        /// <summary>
        /// インデックスだけをエクスポート
        /// </summary>
        /// <param name="container">Container to export values to.</param>
        protected override void ExportValues(IPushOnlyContainer<XElement> container)
        {
            var root = new XElement("CustomShaderSettings");
            root.Add(new XAttribute("index", this.SelectedSettingIndex));
            container.Add(root);
        }

        /// <summary>
        /// インデックスだけをインポート
        /// </summary>
        /// <param name="elements">Sequence of XML elements to import from.</param>
        protected override void ImportValues(IEnumerable<XElement> elements)
        {
            if (elements == null)
            {
                throw new ArgumentNullException("elements");
            }

            XElement root = elements.FirstOrDefault(x => x.Name == "CustomShaderSettings");
            if (root == null)
            {
                throw new FormatException("CustomShaderSettings");
            }

            int index;
            if (int.TryParse(root.Attribute("index").Value, out index))
            {
                this.SelectedSettingIndex = index;
            }
        }

        /// <summary>
        /// スペック切り替えで変化するプロパティをアップデートします。
        /// </summary>
        /// <param name="sender">使用しません</param>
        /// <param name="e">使用しません</param>
        private void OnSpecChanged(object sender, EventArgs e)
        {
            this.OnPropertyChanged(() => this.EnableShaderAssembly);
            this.OnPropertyChanged(() => this.IsVfx);
            this.OnPropertyChanged(() => this.EnableStreamOutSource);
            this.OnPropertyChanged(() => this.EnableStreamOutAssembly);

            // シェーダの負荷情報もクリア
            this.LatencyInfo = string.Empty;
            this.OnPropertyChanged(() => this.LatencyInfo);
        }

        /// <summary>
        /// セッティング列を設定します.
        /// </summary>
        /// <returns>True on success.</returns>
        private bool SetupSettingItems()
        {
            // Determine the UI culture.
            bool useEnglish = false;

            var culture = System.Threading.Thread.CurrentThread.CurrentUICulture;
            var cultureEnUs = System.Globalization.CultureInfo.CreateSpecificCulture("en-US");
            if (culture.Equals(cultureEnUs) == true)
            {
                useEnglish = true;
            }

            // First remove all the child view models.
            this.GroupViewModel.Children.Clear();

            // Create the key value pair array that contains the name of the custom shader setting
            // and it's view model.
            int itemCount = 0;
            if (CustomShaderUserDataManager.Definition != null)
            {
                itemCount = CustomShaderUserDataManager.Definition.CustomShaderDefinitions.Count;
            }

            var settingItems = new KeyValuePair<string, object>[itemCount + 1];

            // Add the item indicating not using any custom shader.
            settingItems[0] = new KeyValuePair<string, object>(Resources.EmitterCustomShaderNotUse, -1);

            // Add the custom shader setting items.
            if (itemCount > 0)
            {
                int index = 0;
                foreach (CustomShaderDefinition def in
                    CustomShaderUserDataManager.Definition.CustomShaderDefinitions)
                {
                    string caption = useEnglish == true && string.IsNullOrEmpty(def.NameEn) == false ? def.NameEn : def.Name;

                    // Create the item.
                    settingItems[index + 1] = new KeyValuePair<string, object>(caption, index);

                    // Add a view model for the data model.
                    this.GroupViewModel.Children.Add(new EmitterCustomShaderSettingDataViewModel(
                        this,
                        this.DataModel.Settings[index],
                        index));

                    // Advance the index.
                    ++index;
                }
            }

            this.AvailableSettingItems = settingItems;

            // インデックス外を選択していたときデフォルト値に変更
            if (this.SelectedSettingIndex >= itemCount)
            {
                this.SelectedSettingIndex = -1;
            }

            // Notify UI to update.
            if (this.initialized == true)
            {
                this.OnPropertyChanged(() => this.AvailableSettingItems);
                this.OnPropertyChanged(() => this.SelectedSettingIndex);

                this.GroupViewModel.RaiseViewModelsChanged();
                this.GroupViewModel.RaiseSelectedIndexChanged();
            }

            return true;
        }

        /// <summary>
        /// Compile shader in the emitter.
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnCompileShader(object parameter)
        {
            uint id = 0;
            var indexStr = parameter as string;
            if (!string.IsNullOrEmpty(indexStr))
            {
                uint.TryParse(indexStr, out id);
            }

            var emitterViewModel = ViewModelBase.GetParent<EmitterViewModel>(this);
            if (emitterViewModel == null)
            {
                return;
            }

            var completeCallback = new Action<ShaderCompileResult, string, string, string, string>(
                (result, vertexShader, fragmentShader, computeShader, latencyInfo) =>
            {
                if (result == ShaderCompileResult.Success)
                {
                    Logger.Log("LogView", LogLevels.Information, Properties.Resources.ShaderCompileSuccess);
                    this.LatencyInfo = latencyInfo;
                    this.OnPropertyChanged(() => this.LatencyInfo);
                }
                else if (result == ShaderCompileResult.Warning)
                {
                    Logger.Log("LogView", LogLevels.Warning, Properties.Resources.ShaderCompileWarning);
                    this.LatencyInfo = latencyInfo;
                    this.OnPropertyChanged(() => this.LatencyInfo);
                }
                else
                {
                    Logger.Log("LogView", LogLevels.Warning, Properties.Resources.ShaderCompileFailed);
                }
            });

            var helper = new ShaderBinaryHelper();
            helper.UserDefineIndex = id;
            helper.GenrateShaderAssemblyAsync(emitterViewModel.DataModel, completeCallback);
        }

        /// <summary>
        /// Generate shader source code in the emitter.
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnGenerateShaderSource(object parameter)
        {
            var shaderType = parameter as string;
            if (string.IsNullOrEmpty(shaderType) == true)
            {
                return;
            }

            uint id = 0;
            var splittedParam = shaderType.Split(',');
            if (splittedParam.Length == 2)
            {
                shaderType = splittedParam[0];
                if (!uint.TryParse(splittedParam[1], out id))
                {
                    return;
                }
            }

            var emitterViewModel =　ViewModelBase.GetParent<EmitterViewModel>(this);
            var emitterSetViewModel = ViewModelBase.GetParent<EmitterSetViewModel>(this);
            if (emitterViewModel == null || emitterSetViewModel == null)
            {
                return;
            }

            string vertexShader;
            string fragmentShader;
            string computeShader;
            bool result;

            // 一時的にプロジェクトコンフィグの値を書き換えてオプション設定を通知する
            var prjconfig = OptionStore.ProjectConfig;
            bool currentOption = prjconfig.EnableShaderPreprocessOption;
            prjconfig.EnableShaderPreprocessOption = this.EnablePreprocess;
            using (new AnonymousDisposable(() => prjconfig.EnableShaderPreprocessOption = currentOption))
            {
                // Generate shader assembly code.
                var helper = new ShaderBinaryHelper();
                helper.UserDefineIndex = id;
                result = helper.GenerateShaderSource(
                    emitterViewModel.DataModel,
                    out vertexShader,
                    out fragmentShader,
                    out computeShader);
            }

            if (result == false)
            {
                if (shaderType == "vertex")
                {
                    Logger.Log("LogView", LogLevels.Error, Properties.Resources.WarningFailedToGenerateVertexShaderSource);
                }
                else if (shaderType == "fragment")
                {
                    Logger.Log("LogView", LogLevels.Error, Properties.Resources.WarningFailedToGenerateFragmentShaderSource);
                }

                return;
            }

            // Compose the file path.
            string esetName = emitterSetViewModel.DataModel.Name;
            string emitterName = emitterViewModel.DataModel.Name;

            if (shaderType == "vertex")
            {
                string fileName = string.Format("{0}_{1}_{2}.vsh", esetName, emitterName, id);
                this.ShowShaderCodeWithTextEditor(fileName, vertexShader);
            }
            else if (shaderType == "fragment")
            {
                string fileName = string.Format("{0}_{1}_{2}.fsh", esetName, emitterName, id);
                this.ShowShaderCodeWithTextEditor(fileName, fragmentShader);
            }
            else if (shaderType == "compute")
            {
                string fileName = string.Format("{0}_{1}_{2}.csh", esetName, emitterName, id);
                this.ShowShaderCodeWithTextEditor(fileName, computeShader);
            }
        }

        /// <summary>
        /// Generate shader assembly code in the emitter.
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnGenerateShaderAssembly(object parameter)
        {
            var shaderType = parameter as string;
            if (string.IsNullOrEmpty(shaderType) == true)
            {
                return;
            }

            uint id = 0;
            var splittedParam = shaderType.Split(',');
            if (splittedParam.Length == 2)
            {
                shaderType = splittedParam[0];
                if (!uint.TryParse(splittedParam[1], out id))
                {
                    return;
                }
            }

            var emitterViewModel = ViewModelBase.GetParent<EmitterViewModel>(this);
            var emitterSetViewModel = ViewModelBase.GetParent<EmitterSetViewModel>(this);
            if (emitterViewModel == null || emitterSetViewModel == null)
            {
                return;
            }

            // Compose the file path.
            string esetName = emitterSetViewModel.DataModel.Name;
            string emitterName = emitterViewModel.DataModel.Name;

            string fileName = string.Format("{0}_{1}_{2}.asm", esetName, emitterName, id);

            var prjconfig = OptionStore.ProjectConfig;
            bool currentOption = prjconfig.EnableShaderPreprocessOption;
            prjconfig.EnableShaderPreprocessOption = this.EnablePreprocess;

            var completeCallback = new Action<ShaderCompileResult, string, string, string, string>(
                (result, vertexShader, fragmentShader, computeShader, latencyInfo) =>
            {
                prjconfig.EnableShaderPreprocessOption = currentOption;
                if (result == ShaderCompileResult.Success)
                {
                    Logger.Log("LogView", LogLevels.Information, Properties.Resources.ShaderCompileSuccess);
                }
                else if (result == ShaderCompileResult.Warning)
                {
                    Logger.Log("LogView", LogLevels.Warning, Properties.Resources.ShaderCompileWarning);
                }
                else
                {
                    Logger.Log("LogView", LogLevels.Warning, Properties.Resources.ShaderCompileFailed);
                    return;
                }

                if (shaderType == "vertex")
                {
                    this.ShowShaderCodeWithTextEditor(fileName, vertexShader);
                }
                else if (shaderType == "fragment")
                {
                    this.ShowShaderCodeWithTextEditor(fileName, fragmentShader);
                }
                else if (shaderType == "compute")
                {
                    this.ShowShaderCodeWithTextEditor(fileName, computeShader);
                }
            });

            var helper = new ShaderBinaryHelper();
            helper.UserDefineIndex = id;
            helper.GenrateShaderAssemblyAsync(emitterViewModel.DataModel, completeCallback);
        }

        /// <summary>
        /// 現在の描画パス情報を取得します。
        /// </summary>
        /// <returns>現在の描画パス</returns>
        private EffectMakerProjectConfig.DrawPath GetCurrentDrawPath()
        {
            var emitter = this.FindNearestParentOfType<EmitterViewModel>();
            if (emitter == null || emitter.EmitterBasicViewModel == null)
            {
                return null;
            }

            var drawPathIndex = emitter.EmitterBasicViewModel.EmitterBasicRenderViewModel.DrawPath;
            return OptionStore.ProjectConfig.DrawPaths.FirstOrDefault(p => p.Id == drawPathIndex);
        }

        /// <summary>
        /// Show shader code with external text editor.
        /// </summary>
        /// <param name="fileName">The temporary shader file name.</param>
        /// <param name="shaderCode">The shader code.</param>
        /// <returns>True on success.</returns>
        private bool ShowShaderCodeWithTextEditor(string fileName, string shaderCode)
        {
            // Compose temporary folder path.
            string filePath = Path.Combine(IOConstants.ExecutableFolderPath, "work\\shader");

            if (Directory.Exists(filePath) == false)
            {
                try
                {
                    Directory.CreateDirectory(filePath);
                }
                catch
                {
                    Logger.Log(LogLevels.Error, "EmitterCustomShaderGroupViewModel.ShowShaderCodeWithTextEditor : Failed creating temporary folder for generated shader code.");
                    return false;
                }
            }

            // Compose the full path for the shader file.
            filePath = Path.Combine(filePath, fileName);

            // Save the shader code to temporary file folder.
            try
            {
                File.WriteAllText(filePath, shaderCode);
            }
            catch
            {
                Logger.Log(LogLevels.Error, "EmitterCustomShaderGroupViewModel.ShowShaderCodeWithTextEditor : Failed saving generated shader code to temporary file.");
                return false;
            }

            // Show the shader code with the text editor set in the config.
            ApplicationIOManager.OpenTextFileWithExternalTextEditor(filePath);

            return true;
        }
    }
}
