﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows;
using System.Xml.Linq;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.RandomColor;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.ClipboardDataTypes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Disposables;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Editting;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.Properties;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterColorSettingsViewModel.
    /// </summary>
    public class EmitterColorSettingsViewModel : ColorPickerOwnerViewModel<EmitterColorSettingsData>, IModificationFlagOwner
    {
        /// <summary>
        /// カラー挙動 - 固定
        /// </summary>
        private const int ColorBehaviorTypeFix = 0;

        /// <summary>
        /// カラー挙動 - 8キー
        /// </summary>
        private const int ColorBehaviorType8Key = 2;

        /// <summary>
        /// カラー挙動 - ランダム
        /// </summary>
        private const int ColorBehaviorTypeRandom = 3;

        /// <summary>
        /// アルファ挙動 - 固定
        /// </summary>
        private const int AlphaBehaviorTypeFix = 0;

        /// <summary>
        /// アルファ挙動 - 8キー
        /// </summary>
        private const int AlphaBehaviorType8Key = 2;

        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "Color0ValueFixColorEditorVisibility",
            "Color0Gradation8KeyEditorVisibility",
            "Color0RandomColorEditorVisibility",
            "Color1ValueFixColorEditorVisibility",
            "Color1Gradation8KeyEditorVisibility",
            "Color1RandomColorEditorVisibility",
            "Alpha0ValueFixColorEditorVisibility",
            "Alpha0Gradation8KeyEditorVisibility",
            "Alpha1ValueFixColorEditorVisibility",
            "Alpha1Gradation8KeyEditorVisibility",
        };

        /// <summary>
        /// データ交換中か
        /// </summary>
        private bool isExchangingData;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterColorSettingsViewModel(
            HierarchyViewModel parent, EmitterColorSettingsData dataModel)
            : base(parent, dataModel)
        {
            this.BlockingFunc = () => this.isExchangingData;

            this.Color0SettingTypeItems = new[]
            {
                new KeyValuePair<string, object>(Resources.EmitterColorSettingTypeFix, ColorBehaviorTypeFix),
                new KeyValuePair<string, object>(Resources.EmitterColorSettingType8Key, ColorBehaviorType8Key),
                new KeyValuePair<string, object>(Resources.EmitterColorSettingTypeRandom, ColorBehaviorTypeRandom),
           };

            this.Color1SettingTypeItems = new[]
            {
                new KeyValuePair<string, object>(Resources.EmitterColorSettingTypeFix, ColorBehaviorTypeFix),
                new KeyValuePair<string, object>(Resources.EmitterColorSettingType8Key, ColorBehaviorType8Key),
                new KeyValuePair<string, object>(Resources.EmitterColorSettingTypeRandom, ColorBehaviorTypeRandom),
            };

            this.Alpha0SettingTypeItems = new[]
            {
                new KeyValuePair<string, object>(Resources.EmitterColorSettingTypeFix, AlphaBehaviorTypeFix),
                new KeyValuePair<string, object>(Resources.EmitterColorSettingType8Key, AlphaBehaviorType8Key),
            };

            this.Alpha1SettingTypeItems = new[]
            {
                new KeyValuePair<string, object>(Resources.EmitterColorSettingTypeFix, AlphaBehaviorTypeFix),
                new KeyValuePair<string, object>(Resources.EmitterColorSettingType8Key, AlphaBehaviorType8Key),
            };

            this.ColorAnimationLoopData0 = new EmitterAnimationLoopViewModel(this, dataModel.ColorAnimationLoopData0);
            this.Children.Add(this.ColorAnimationLoopData0);

            this.ColorAnimationLoopData1 = new EmitterAnimationLoopViewModel(this, dataModel.ColorAnimationLoopData1);
            this.Children.Add(this.ColorAnimationLoopData1);

            this.AlphaAnimationLoopData0 = new EmitterAnimationLoopViewModel(this, dataModel.AlphaAnimationLoopData0);
            this.Children.Add(this.AlphaAnimationLoopData0);

            this.AlphaAnimationLoopData1 = new EmitterAnimationLoopViewModel(this, dataModel.AlphaAnimationLoopData1);
            this.Children.Add(this.AlphaAnimationLoopData1);

            this.AlphaPresetExecutable = new AnonymousExecutable(this.AdaptPreset);

            this.CopyColorExecutable = new AnonymousExecutable(this.OnCopyColor);
            this.PasteColorExecutable = new AnonymousExecutable(this.OnPasteColor);

            this.OnGetParticleLifeExecutable = new AnonymousExecutable((o) =>
            {
                var param = o as float[];
                var emitterVm = this.FindNearestParentOfType<EmitterViewModel>();
                param[0] = emitterVm.EmitterParticleViewModel.EmitterParticleLifeViewModel.GetDataModelValue<int>("Life");
            });

            var mmfvm = new MultipleModificationFlagsViewModel(this);

            // Color 0
            this.ColorAnimationLoopData0.PropertyChanged +=
                (sender, args) => this.OnPropertyChanged(() => this.IsColorAnimationLoop0Modified);
            var emitterColor0Params = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.Color0BehaviorType),
                ViewModelBase.NameOf(() => dataModel.Color0Value),
                ViewModelBase.NameOf(() => dataModel.ColorAnimation0),
                ViewModelBase.NameOf(() => this.IsColorAnimationLoop0Modified),
                ViewModelBase.NameOf(() => dataModel.RandomColor0));
            mmfvm.SetPropertyDictionary("EmitterColor0Params", emitterColor0Params);
            mmfvm.SetModificationDictionary("EmitterColor0Params", new IModificationPropertyOwner[] { this.ColorAnimationLoopData0 });

            // Color 1
            this.ColorAnimationLoopData1.PropertyChanged +=
                (sender, args) => this.OnPropertyChanged(() => this.IsColorAnimationLoop1Modified);
            var emitterColor1Params = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.Color1BehaviorType),
                ViewModelBase.NameOf(() => dataModel.Color1Value),
                ViewModelBase.NameOf(() => dataModel.ColorAnimation1),
                ViewModelBase.NameOf(() => this.IsColorAnimationLoop1Modified),
                ViewModelBase.NameOf(() => dataModel.RandomColor1));
            mmfvm.SetPropertyDictionary("EmitterColor1Params", emitterColor1Params);
            mmfvm.SetModificationDictionary("EmitterColor1Params", new IModificationPropertyOwner[] { this.ColorAnimationLoopData1 });

            // Alpha 0
            this.AlphaAnimationLoopData0.PropertyChanged +=
                (sender, args) => this.OnPropertyChanged(() => this.IsAlphaAnimationLoop0Modified);
            var emitterAlpha0Params = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.Alpha0BehaviorType),
                ViewModelBase.NameOf(() => dataModel.Alpha0Value),
                ViewModelBase.NameOf(() => dataModel.AlphaAnimation0),
                ViewModelBase.NameOf(() => this.IsAlphaAnimationLoop0Modified));
            mmfvm.SetPropertyDictionary("EmitterAlpha0Params", emitterAlpha0Params);
            mmfvm.SetModificationDictionary("EmitterAlpha0Params", new IModificationPropertyOwner[] { this.AlphaAnimationLoopData0 });

            // Alpha 1
            this.AlphaAnimationLoopData1.PropertyChanged +=
                (sender, args) => this.OnPropertyChanged(() => this.IsAlphaAnimationLoop1Modified);
            var emitterAlpha1Params = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.Alpha1BehaviorType),
                ViewModelBase.NameOf(() => dataModel.Alpha1Value),
                ViewModelBase.NameOf(() => dataModel.AlphaAnimation1),
                ViewModelBase.NameOf(() => this.IsAlphaAnimationLoop1Modified));
            mmfvm.SetPropertyDictionary("EmitterAlpha1Params", emitterAlpha1Params);
            mmfvm.SetModificationDictionary("EmitterAlpha1Params", new IModificationPropertyOwner[] { this.AlphaAnimationLoopData1 });

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = mmfvm;
            this.ModificationFlagViewModel.AddIgnoreProperties(this.IgnoreColorPickerPropertyNames);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// Color0のアニメーションのループ情報が変更されているか
        /// </summary>
        public bool IsColorAnimationLoop0Modified
        {
            get { return this.ColorAnimationLoopData0.IsModified; }
        }

        /// <summary>
        /// Color1のアニメーションのループ情報が変更されているか
        /// </summary>
        public bool IsColorAnimationLoop1Modified
        {
            get { return this.ColorAnimationLoopData1.IsModified; }
        }

        /// <summary>
        /// Alpha0のアニメーションのループ情報が変更されているか
        /// </summary>
        public bool IsAlphaAnimationLoop0Modified
        {
            get { return this.AlphaAnimationLoopData0.IsModified; }
        }

        /// <summary>
        /// Alpha1のアニメーションのループ情報が変更されているか
        /// </summary>
        public bool IsAlphaAnimationLoop1Modified
        {
            get { return this.AlphaAnimationLoopData1.IsModified; }
        }

        #region SettingTypeItems

        /// <summary>カラー0挙動タイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> Color0SettingTypeItems { get; private set; }

        /// <summary>カラー1挙動タイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> Color1SettingTypeItems { get; private set; }

        /// <summary>アルファ0挙動タイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> Alpha0SettingTypeItems { get; private set; }

        /// <summary>アルファ1挙動タイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> Alpha1SettingTypeItems { get; private set; }

        #endregion

        #region カラー挙動タイプ

        /// <summary>
        /// カラー0挙動タイプを設定または取得します。
        /// </summary>
        [UseDataModelOriginalValue]
        public int Color0BehaviorType
        {
            get
            {
                return this.DataModel.Color0BehaviorType;
            }

            set
            {
                // if (this.Color0BehaviorType != value)
                {
                    var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(
                        ViewModelBase.NameOf(() => this.Color0BehaviorType), false, true);
                    this.TrySetMember(binder, value);

                    this.OnPropertyChanged(() => this.Color0ValueFixColorEditorVisibility);
                    this.OnPropertyChanged(() => this.Color0Gradation8KeyEditorVisibility);
                    this.OnPropertyChanged(() => this.Color0RandomColorEditorVisibility);
                    this.DisableColorPicker();
                }
            }
        }

        /// <summary>
        /// カラー0固定カラーコントロールを表示するか？
        /// </summary>
        public bool Color0ValueFixColorEditorVisibility
        {
            get { return this.Color0BehaviorType == EmitterColorSettingsViewModel.ColorBehaviorTypeFix; }
        }

        /// <summary>
        /// カラー0８キーアニメコントロールを表示するか？
        /// </summary>
        public bool Color0Gradation8KeyEditorVisibility
        {
            get { return this.Color0BehaviorType == EmitterColorSettingsViewModel.ColorBehaviorType8Key; }
        }

        /// <summary>
        /// カラー0ランダムカラーコントロールを表示するか？
        /// </summary>
        public bool Color0RandomColorEditorVisibility
        {
            get { return this.Color0BehaviorType == EmitterColorSettingsViewModel.ColorBehaviorTypeRandom; }
        }

        /// <summary>
        /// カラー1挙動タイプを設定または取得します。
        /// </summary>
        [UseDataModelOriginalValue]
        public int Color1BehaviorType
        {
            get
            {
                return this.DataModel.Color1BehaviorType;
            }

            set
            {
                // if (this.Color1BehaviorType != value)
                {
                    var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(
                        ViewModelBase.NameOf(() => this.Color1BehaviorType), false, true);
                    this.TrySetMember(binder, value);

                    this.OnPropertyChanged(() => this.Color1ValueFixColorEditorVisibility);
                    this.OnPropertyChanged(() => this.Color1Gradation8KeyEditorVisibility);
                    this.OnPropertyChanged(() => this.Color1RandomColorEditorVisibility);
                    this.DisableColorPicker();
                }
            }
        }

        /// <summary>
        /// カラー1固定カラーコントロールを表示するか？
        /// </summary>
        public bool Color1ValueFixColorEditorVisibility
        {
            get { return this.Color1BehaviorType == EmitterColorSettingsViewModel.ColorBehaviorTypeFix; }
        }

        /// <summary>
        /// カラー1８キーアニメコントロールを表示するか？
        /// </summary>
        public bool Color1Gradation8KeyEditorVisibility
        {
            get { return this.Color1BehaviorType == EmitterColorSettingsViewModel.ColorBehaviorType8Key; }
        }

        /// <summary>
        /// カラー1ランダムカラーコントロールを表示するか？
        /// </summary>
        public bool Color1RandomColorEditorVisibility
        {
            get { return this.Color1BehaviorType == EmitterColorSettingsViewModel.ColorBehaviorTypeRandom; }
        }

        #endregion

        #region アルファ挙動タイプ

        /// <summary>
        /// アルファ0挙動タイプを設定または取得します。
        /// </summary>
        [UseDataModelOriginalValue]
        public int Alpha0BehaviorType
        {
            get
            {
                return this.DataModel.Alpha0BehaviorType;
            }

            set
            {
                // if (this.Alpha0BehaviorType != value)
                {
                    var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(
                        ViewModelBase.NameOf(() => this.Alpha0BehaviorType), false, true);
                    this.TrySetMember(binder, value);

                    this.OnPropertyChanged(() => this.Alpha0ValueFixColorEditorVisibility);
                    this.OnPropertyChanged(() => this.Alpha0Gradation8KeyEditorVisibility);
                    this.DisableColorPicker();
                }
            }
        }

        /// <summary>
        /// アルファ1固定カラーコントロールを表示するか？
        /// </summary>
        public bool Alpha0ValueFixColorEditorVisibility
        {
            get { return this.Alpha0BehaviorType == EmitterColorSettingsViewModel.AlphaBehaviorTypeFix; }
        }

        /// <summary>
        /// アルファ1８キーアニメコントロールを表示するか？
        /// </summary>
        public bool Alpha0Gradation8KeyEditorVisibility
        {
            get { return this.Alpha0BehaviorType == EmitterColorSettingsViewModel.AlphaBehaviorType8Key; }
        }

        /// <summary>
        /// アルファ1挙動タイプを設定または取得します。
        /// </summary>
        [UseDataModelOriginalValue]
        public int Alpha1BehaviorType
        {
            get
            {
                return this.DataModel.Alpha1BehaviorType;
            }

            set
            {
                // if (this.Alpha1BehaviorType != value)
                {
                    var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(
                        ViewModelBase.NameOf(() => this.Alpha1BehaviorType), false, true);
                    this.TrySetMember(binder, value);

                    this.OnPropertyChanged(() => this.Alpha1ValueFixColorEditorVisibility);
                    this.OnPropertyChanged(() => this.Alpha1Gradation8KeyEditorVisibility);
                    this.DisableColorPicker();
                }
            }
        }

        /// <summary>
        /// アルファ1固定カラーコントロールを表示するか？
        /// </summary>
        public bool Alpha1ValueFixColorEditorVisibility
        {
            get { return this.Alpha1BehaviorType == EmitterColorSettingsViewModel.AlphaBehaviorTypeFix; }
        }

        /// <summary>
        /// アルファ1８キーアニメコントロールを表示するか？
        /// </summary>
        public bool Alpha1Gradation8KeyEditorVisibility
        {
            get { return this.Alpha1BehaviorType == EmitterColorSettingsViewModel.AlphaBehaviorType8Key; }
        }

        #endregion

        #region カラー値

        /// <summary>
        /// カラー0
        /// </summary>
        [UseDataModelOriginalValue]
        public ColorRgba Color0Value
        {
            get
            {
                return this.DataModel.Color0Value;
            }

            set
            {
                var name = ViewModelBase.NameOf(() => this.Color0Value);
                var binder = new EffectMakerSetMemberBinder(name, false, false);

                this.TrySetMember(binder, value);
                this.RefreshColorPicker(name, value);
            }
        }

        /// <summary>
        /// カラー1
        /// </summary>
        [UseDataModelOriginalValue]
        public ColorRgba Color1Value
        {
            get
            {
                return this.DataModel.Color1Value;
            }

            set
            {
                var name = ViewModelBase.NameOf(() => this.Color1Value);
                var binder = new EffectMakerSetMemberBinder(name, false, false);

                this.TrySetMember(binder, value);
                this.RefreshColorPicker(name, value);
            }
        }

        /// <summary>
        /// ランダムカラー0
        /// </summary>
        [UseDataModelOriginalValue]
        public RandomColorTable RandomColor0
        {
            get
            {
                return this.DataModel.RandomColor0;
            }

            set
            {
                var name = ViewModelBase.NameOf(() => this.RandomColor0);
                var binder = new EffectMakerSetMemberBinder(name, false, false);
                this.TrySetMember(binder, value);
            }
        }

        /// <summary>
        /// ランダムカラー1
        /// </summary>
        [UseDataModelOriginalValue]
        public RandomColorTable RandomColor1
        {
            get
            {
                return this.DataModel.RandomColor1;
            }

            set
            {
                var name = ViewModelBase.NameOf(() => this.RandomColor1);
                var binder = new EffectMakerSetMemberBinder(name, false, false);
                this.TrySetMember(binder, value);
            }
        }

        /// <summary>
        /// カラーアニメーション0
        /// </summary>
        [UseDataModelOriginalValue]
        public AnimationTableData ColorAnimation0
        {
            get
            {
                return this.DataModel.ColorAnimation0;
            }

            set
            {
                int count = this.ColorAnimation0.Count;
                var binder = new EffectMakerSetMemberBinder("ColorAnimation0", false, false);
                MessageBlockerWithSendBinaryOnce.ExecuteOnDemandReload(
                    () => this.TrySetMember(binder, value),
                    value.Count != count,
                    this.DataModel);
            }
        }

        /// <summary>
        /// カラーアニメーション1
        /// </summary>
        [UseDataModelOriginalValue]
        public AnimationTableData ColorAnimation1
        {
            get
            {
                return this.DataModel.ColorAnimation1;
            }

            set
            {
                int count = this.ColorAnimation1.Count;
                var binder = new EffectMakerSetMemberBinder("ColorAnimation1", false, false);
                MessageBlockerWithSendBinaryOnce.ExecuteOnDemandReload(
                    () => this.TrySetMember(binder, value),
                    value.Count != count,
                    this.DataModel);
            }
        }

        /// <summary>
        /// カラーアニメーション0データ
        /// </summary>
        public EmitterAnimationLoopViewModel ColorAnimationLoopData0 { get; private set; }

        /// <summary>
        /// カラーアニメーション1データ
        /// </summary>
        public EmitterAnimationLoopViewModel ColorAnimationLoopData1 { get; private set; }

        #endregion

        #region アルファ値
        /// <summary>
        /// アルファ0
        /// </summary>
        [UseDataModelOriginalValue]
        public float Alpha0Value
        {
            get
            {
                return this.DataModel.Alpha0Value;
            }

            set
            {
                var name = ViewModelBase.NameOf(() => this.Alpha0Value);
                var binder = new EffectMakerSetMemberBinder(name, false, false);

                this.TrySetMember(binder, value);
                this.RefreshColorPicker(name, value);
            }
        }

        /// <summary>
        /// アルファ1
        /// </summary>
        [UseDataModelOriginalValue]
        public float Alpha1Value
        {
            get
            {
                return this.DataModel.Alpha1Value;
            }

            set
            {
                var name = ViewModelBase.NameOf(() => this.Alpha1Value);
                var binder = new EffectMakerSetMemberBinder(name, false, false);

                this.TrySetMember(binder, value);
                this.RefreshColorPicker(name, value);
            }
        }

        /// <summary>
        /// アルファアニメーション0
        /// </summary>
        [UseDataModelOriginalValue]
        public AnimationTableData AlphaAnimation0
        {
           get
           {
               return this.DataModel.AlphaAnimation0;
           }

           set
           {
               int count = this.AlphaAnimation0.Count;
               var binder = new EffectMakerSetMemberBinder("AlphaAnimation0", false, false);
               MessageBlockerWithSendBinaryOnce.ExecuteOnDemandReload(
                   () => this.TrySetMember(binder, value),
                   value.Count != count,
                   this.DataModel);
           }
        }

        /// <summary>
        /// アルファアニメーション1
        /// </summary>
        [UseDataModelOriginalValue]
        public AnimationTableData AlphaAnimation1
        {
           get
           {
               return this.DataModel.AlphaAnimation1;
           }

           set
           {
               int count = this.AlphaAnimation1.Count;
               var binder = new EffectMakerSetMemberBinder("AlphaAnimation1", false, false);
               MessageBlockerWithSendBinaryOnce.ExecuteOnDemandReload(
                   () => this.TrySetMember(binder, value),
                   value.Count != count,
                   this.DataModel);
           }
        }

        /// <summary>
        /// アルファアニメーション0データ
        /// </summary>
        public EmitterAnimationLoopViewModel AlphaAnimationLoopData0 { get; private set; }

        /// <summary>
        /// アルファアニメーション1データ
        /// </summary>
        public EmitterAnimationLoopViewModel AlphaAnimationLoopData1 { get; private set; }

        #endregion

        /// <summary>
        /// アルファグラデーションにプリセットを適用する処理を取得します。
        /// </summary>
        public IExecutable AlphaPresetExecutable { get; private set; }

        /// <summary>
        /// パーティクルの寿命を取得するExecutableを取得します。
        /// </summary>
        public IExecutable OnGetParticleLifeExecutable { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// Gets the Copy executable instance.
        /// </summary>
        public IExecutable CopyColorExecutable { get; private set; }

        /// <summary>
        /// Gets the Paste executable instance.
        /// </summary>
        public IExecutable PasteColorExecutable { get; private set; }

        /// <summary>
        /// データの０と１を交換する
        /// </summary>
        /// <param name="kind">カラー要素</param>
        public void ExchangeData(ColorElementType kind)
        {
            MultiNodeEditUtil.DisableMultiNodeEdit = true;
            this.isExchangingData = true;
            using (new AnonymousDisposable(() =>
            {
                this.isExchangingData = false;
                MultiNodeEditUtil.DisableMultiNodeEdit = false;
            }))
            using (new MessageBlockerWithSendBinaryOnce(this.DataModel))
            using (new CommandCombiner())
            {
                if (kind == ColorElementType.Color)
                {
                    var tempBehaviorType0 = this.Color0BehaviorType;
                    this.Color0BehaviorType = this.Color1BehaviorType;
                    this.Color1BehaviorType = tempBehaviorType0;

                    var tempColorValue0 = (ColorRgba)this.Color0Value.Clone();
                    this.SetDataModelValue((ColorRgba)this.Color1Value.Clone(), () => this.Color0Value);
                    this.SetDataModelValue(tempColorValue0, () => this.Color1Value);

                    var tempColorAnimation0 = (AnimationTableData)this.ColorAnimation0.Clone();
                    this.SetDataModelValue((AnimationTableData)this.ColorAnimation1.Clone(), () => this.ColorAnimation0);
                    this.SetDataModelValue(tempColorAnimation0, () => this.ColorAnimation1);

                    var tempRandomColor0 = this.RandomColor0.Clone() as RandomColorTable;
                    this.SetDataModelValue((RandomColorTable)this.RandomColor1.Clone(), () => this.RandomColor0);
                    this.SetDataModelValue(tempRandomColor0, () => this.RandomColor1);

                    // 以下、ループ関係
                    var loopColor0 = this.ColorAnimationLoopData0;
                    var loopColor1 = this.ColorAnimationLoopData1;

                    var tempEnableLoop0 = loopColor0.DataModel.EnableLoop;
                    var tempEnableLoop1 = loopColor1.DataModel.EnableLoop;
                    loopColor0.SetDataModelValue(tempEnableLoop1, "EnableLoop");
                    loopColor1.SetDataModelValue(tempEnableLoop0, "EnableLoop");

                    var tempLoopFrame0 = loopColor0.DataModel.LoopFrame;
                    var tempLoopFrame1 = loopColor1.DataModel.LoopFrame;
                    loopColor0.SetDataModelValue(tempLoopFrame1, "LoopFrame");
                    loopColor1.SetDataModelValue(tempLoopFrame0, "LoopFrame");

                    var tempEnableStartRandom0 = loopColor0.DataModel.EnableStartRandom;
                    var tempEnableStartRandom1 = loopColor1.DataModel.EnableStartRandom;
                    loopColor0.SetDataModelValue(tempEnableStartRandom1, "EnableStartRandom");
                    loopColor1.SetDataModelValue(tempEnableStartRandom0, "EnableStartRandom");
                }
                else if (kind == ColorElementType.Alpha)
                {
                    var tempBehaviorType0 = this.Alpha0BehaviorType;
                    this.Alpha0BehaviorType = this.Alpha1BehaviorType;
                    this.Alpha1BehaviorType = tempBehaviorType0;

                    var tempAlphaValue0 = this.Alpha0Value;
                    this.SetDataModelValue(this.Alpha1Value, () => this.Alpha0Value);
                    this.SetDataModelValue(tempAlphaValue0, () => this.Alpha1Value);

                    var tempAlphaAnimation0 = (AnimationTableData)this.AlphaAnimation0.Clone();
                    this.SetDataModelValue((AnimationTableData)this.AlphaAnimation1.Clone(), () => this.AlphaAnimation0);
                    this.SetDataModelValue(tempAlphaAnimation0, () => this.AlphaAnimation1);

                    // 以下、ループ関係
                    var loopAlpha0 = this.AlphaAnimationLoopData0;
                    var loopAlpha1 = this.AlphaAnimationLoopData1;

                    var tempEnableLoop0 = loopAlpha0.DataModel.EnableLoop;
                    var tempEnableLoop1 = loopAlpha1.DataModel.EnableLoop;
                    loopAlpha0.SetDataModelValue(tempEnableLoop1, "EnableLoop");
                    loopAlpha1.SetDataModelValue(tempEnableLoop0, "EnableLoop");

                    var tempLoopFrame0 = loopAlpha0.DataModel.LoopFrame;
                    var tempLoopFrame1 = loopAlpha1.DataModel.LoopFrame;
                    loopAlpha0.SetDataModelValue(tempLoopFrame1, "LoopFrame");
                    loopAlpha1.SetDataModelValue(tempLoopFrame0, "LoopFrame");

                    var tempEnableStartRandom0 = loopAlpha0.DataModel.EnableStartRandom;
                    var tempEnableStartRandom1 = loopAlpha1.DataModel.EnableStartRandom;
                    loopAlpha0.SetDataModelValue(tempEnableStartRandom1, "EnableStartRandom");
                    loopAlpha1.SetDataModelValue(tempEnableStartRandom0, "EnableStartRandom");
                }
            }
        }

        #region PatchCopyData
        /// <summary>
        /// コピーデータにパッチを当てる
        /// </summary>
        /// <param name="dstTarget">変更対象</param>
        /// <param name="current">現在の状態</param>
        /// <param name="copyData">コピーデータ</param>
        /// <returns>パッチ済みデータ</returns>
        private static string PatchCopyData(ColorTargetType dstTarget, string current, ColorUnitClipboardData copyData)
        {
            //// ■処理内容
            ////   copyData.Body の copyData.Target 位置のパラメータを
            ////   current の target 位置に上書きする

            var srcXml = XElement.Parse(copyData.Body);
            var dstXml = XElement.Parse(current);
            var srcTarget = copyData.Target;

            Action<string> paste = parent =>
            {
                // BehaviorType
                {
                    var src = srcXml.Element(parent).Elements().Where(x => x.Name == "lp").FirstOrDefault(x => x.Attribute("name").Value == MakeCopyDataBehaviorTypeString(srcTarget));
                    var dst = dstXml.Element(parent).Elements().Where(x => x.Name == "lp").FirstOrDefault(x => x.Attribute("name").Value == MakeCopyDataBehaviorTypeString(dstTarget));

                    if ((src != null) && (dst != null))
                    {
                        dst.Attribute("value").Value = src.Attribute("value").Value;
                    }
                }

                Action<Func<ColorTargetType, string>> pasteElements = makeString =>
                {
                    var src = srcXml.Element(parent).Elements().Where(x => x.Name == "IXmlSerializable").FirstOrDefault(x => x.Attribute("name").Value == makeString(srcTarget));
                    var dst = dstXml.Element(parent).Elements().Where(x => x.Name == "IXmlSerializable").FirstOrDefault(x => x.Attribute("name").Value == makeString(dstTarget));

                    if ((src != null) && (dst != null))
                    {
                        dst.Elements().Remove();
                        dst.Add(src.Elements());
                    }
                };

                // Value
                pasteElements(MakeCopyDataValueString);

                // Animation
                pasteElements(MakeCopyDataAnimationString);

                // Random
                pasteElements(MakeCopyDataRandomString);

                // AnimationLoopData
                pasteElements(MakeCopyDataAnimationLoopDataString);
            };

            paste("data");
            paste("vm");

            return dstXml.ToString();
        }

        /// <summary>
        /// BehaviorType文字列の生成
        /// </summary>
        /// <param name="target">対象</param>
        /// <returns>文字列</returns>
        private static string MakeCopyDataBehaviorTypeString(ColorTargetType target)
        {
            return target.ToString() + "BehaviorType";
        }

        /// <summary>
        /// Value文字列の生成
        /// </summary>
        /// <param name="target">対象</param>
        /// <returns>文字列</returns>
        private static string MakeCopyDataValueString(ColorTargetType target)
        {
            return target.ToString() + "Value";
        }

        /// <summary>
        /// Animation文字列の生成
        /// </summary>
        /// <param name="target">対象</param>
        /// <returns>文字列</returns>
        private static string MakeCopyDataAnimationString(ColorTargetType target)
        {
            switch (target)
            {
                case ColorTargetType.Color0:
                    return "ColorAnimation0";
                case ColorTargetType.Color1:
                    return "ColorAnimation1";
                case ColorTargetType.Alpha0:
                    return "AlphaAnimation0";
                case ColorTargetType.Alpha1:
                    return "AlphaAnimation1";
            }

            throw new NotImplementedException();
        }

        /// <summary>
        /// AnimationLoopData文字列の生成
        /// </summary>
        /// <param name="target">対象</param>
        /// <returns>文字列</returns>
        private static string MakeCopyDataAnimationLoopDataString(ColorTargetType target)
        {
            switch (target)
            {
                case ColorTargetType.Color0:
                    return "ColorAnimationLoopData0";
                case ColorTargetType.Color1:
                    return "ColorAnimationLoopData1";
                case ColorTargetType.Alpha0:
                    return "AlphaAnimationLoopData0";
                case ColorTargetType.Alpha1:
                    return "AlphaAnimationLoopData1";
            }

            throw new NotImplementedException();
        }

        /// <summary>
        /// Random文字列の生成
        /// </summary>
        /// <param name="target">対象</param>
        /// <returns>文字列</returns>
        private static string MakeCopyDataRandomString(ColorTargetType target)
        {
            return "Random" + target.ToString();
        }

        #endregion

        /// <summary>
        /// プリセットを適用します。
        /// </summary>
        /// <param name="parameter">プリセットの適用先と適用パターンを表す文字列(プロパティ名, [A or B])</param>
        private void AdaptPreset(object parameter)
        {
            var args = parameter as string;
            if (string.IsNullOrEmpty(args))
            {
                return;
            }

            var strArray = args.Split(',');

            if (strArray.Length != 2)
            {
                return;
            }

            var binder = new EffectMakerSetMemberBinder(strArray[0], false, true);
            var table = new AnimationTableData();
            if (strArray[1] == "A")
            {
                table.AddKeyFrame(0, 1.0f, 1.0f, 1.0f, 1.0f, false);
                table.AddKeyFrame(100, 1.0f, 1.0f, 1.0f, 0.0f, false);
            }
            else if (strArray[1] == "B")
            {
                table.AddKeyFrame(0, 1.0f, 1.0f, 1.0f, 0.0f, false);
                table.AddKeyFrame(25, 1.0f, 1.0f, 1.0f, 1.0f, false);
                table.AddKeyFrame(75, 1.0f, 1.0f, 1.0f, 1.0f, false);
                table.AddKeyFrame(100, 1.0f, 1.0f, 1.0f, 0.0f, false);
            }
            else
            {
                return;
            }

            using (new MessageBlockerWithSendBinaryOnce(this.DataModel))
            {
                if (this.TrySetMember(binder, table))
                {
                    this.OnPropertyChanged(strArray[0]);
                }
            }
        }

        /// <summary>
        /// Called when the Copy executable is run.
        /// </summary>
        /// <param name="param">パラメータ</param>
        private void OnCopyColor(object param)
        {
            var copyData = new ColorUnitClipboardData
            {
                Target = (ColorTargetType)param,
                Body = this.ExportValuesAsXml()
            };

            Clipboard.SetData(ColorUnitClipboardData.ClipboardFormat, copyData);
        }

        /// <summary>
        /// Called when the Paste executable is run.
        /// </summary>
        /// <param name="param">パラメータ</param>
        private void OnPasteColor(object param)
        {
            if (Clipboard.ContainsData(ColorUnitClipboardData.ClipboardFormat) == false)
            {
                return;
            }

            var copyData = Clipboard.GetData(ColorUnitClipboardData.ClipboardFormat) as ColorUnitClipboardData;

            System.Diagnostics.Debug.Assert(
                ColorUtility.MakeColorElementType(copyData.Target) == ColorUtility.MakeColorElementType((ColorTargetType)param),
                "色要素が食い違っている");

            var current = this.ExportValuesAsXml();
            string newValue = PatchCopyData((ColorTargetType)param, current, copyData);

            CommandManager.Execute(new PasteGroupCommand(this, current, newValue));
        }
    }
}
