﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.Properties;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterBasicRenderViewModel.
    /// </summary>
    public class EmitterBasicRenderViewModel : PropertyGroupViewModel<EmitterBasicRenderData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "ZBufferTestVisibility",
            "ZBufferTestSettingVisibility",
            "AlphaTestVisibility",
            "AlphaTestSettingVisibility",
            "AlphaBlendVisibility",
            "DrawPathEnabled",
            "RenderPathTypeItems",
            "RenderPathTypeColors",
            "DrawPathWarningVisibility",
            "ZbufferAlphaTestTypeDesc",
            "DrawPathName",
            "ZSortCommentVisibility",
        };

        /// <summary>
        /// 描画パスの有効/無効です.
        /// </summary>
        private bool drawPathEnabled = false;

        /// <summary>
        /// 描画パス警告表示のOn/Off
        /// </summary>
        private bool drawPathWarningVisibility = false;

        /// <summary>
        /// 親のビューモデル
        /// </summary>
        private EmitterBasicViewModel parentViewModel = null;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterBasicRenderViewModel(
            HierarchyViewModel parent, EmitterBasicRenderData dataModel)
            : base(parent, dataModel)
        {
            // 描画パスのコンボボックスアイテムを設定
            this.SetupRenderPathTypeItems();

            this.ParticleSortTypeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterBasicParticleSortTypeNothing, 0),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterBasicParticleSortTypeAscend, 3),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterBasicParticleSortTypeDescend, 1),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterBasicParticleSortTypeZSort, 2),
            };

            this.DisplaySurfaceTypeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterDisplaySurfaceBoth, 0),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterDisplaySurfaceFront, 1),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterDisplaySurfaceBack, 2),
            };

            this.ZBufferTestPassTypeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterZBufferTestPassNever, 0),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterZBufferTestPassLess, 1),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterZBufferTestPassEqual, 2),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterZBufferTestPassLessEqual, 3),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterZBufferTestPassGreater, 4),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterZBufferTestPassNotEqual, 5),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterZBufferTestPassGreaterEqual, 6),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterZBufferTestPassAlways, 7),
            };

            this.AlphaTestPassTypeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterAlphaTestPassNever, 0),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterAlphaTestPassLess, 1),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterAlphaTestPassEqual, 2),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterAlphaTestPassLessEqual, 3),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterAlphaTestPassGreater, 4),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterAlphaTestPassNotEqual, 5),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterAlphaTestPassGreaterEqual, 6),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterAlphaTestPassAlways, 7),
            };

            this.BlendTypeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterBlendTypeGeneral, 0),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterBlendTypeAdd, 1),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterBlendTypeSub, 2),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterBlendTypeMul, 3),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterBlendTypeScreen, 4),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterBlendTypePreMultipliedAlpha, 5),
            };

            this.ZBufferAlphaTestTypeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterZBufATestNormal, 0),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterZBufATestIgnore, 1),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterZBufATestTransparent, 2),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterZBufATestOpaque, 3),
                new KeyValuePair<string, object>(
                    Properties.Resources.EmitterZBufATestCustom, 4),
            };

            this.parentViewModel = (EmitterBasicViewModel)this.Parent;

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// パーティクルソートタイプを取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> ParticleSortTypeItems { get; private set; }

        /// <summary>
        /// 描画パスタイプを取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> RenderPathTypeItems { get; private set; }

        /// <summary>
        /// 描画パスタイプの文字色を取得します.
        /// </summary>
        public Color[] RenderPathTypeColors { get; private set; }

        /// <summary>
        /// 表示面タイプを取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> DisplaySurfaceTypeItems { get; private set; }

        /// <summary>
        /// 深度テスト通過タイプを取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> ZBufferTestPassTypeItems { get; private set; }

        /// <summary>
        /// アルファテスト通過タイプを取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> AlphaTestPassTypeItems { get; private set; }

        /// <summary>
        /// ブレンドタイプを取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> BlendTypeItems { get; private set; }

        /// <summary>
        /// Zバッファ・αテストタイプを取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> ZBufferAlphaTestTypeItems { get; private set; }

        /// <summary>
        /// 描画パスの有効/無効を取得または設定します.
        /// </summary>
        public bool DrawPathEnabled
        {
            get { return this.drawPathEnabled; }
            set { this.SetValue(ref this.drawPathEnabled, value); }
        }

        /// <summary>
        /// 描画パス警告表示のOn/Offを取得または設定します.
        /// </summary>
        public bool DrawPathWarningVisibility
        {
            get { return this.drawPathWarningVisibility; }
            set { this.SetValue(ref this.drawPathWarningVisibility, value); }
        }

        /// <summary>
        /// 深度テストの有効/無効を取得または設定します.
        /// </summary>
        [UseDataModelOriginalValue]
        public bool EnableZBufferTest
        {
            get
            {
                return this.GetDataModelValue(() => this.EnableZBufferTest);
            }

            set
            {
                this.SetDataModelValue(value, () => this.EnableZBufferTest);

                // プロパティ変更通知.
                this.OnPropertyChanged(() => this.ZBufferTestSettingVisibility);
            }
        }

        /// <summary>
        /// アルファテストの有効/無効を取得または設定します.
        /// </summary>
        [UseDataModelOriginalValue]
        public bool EnableAlphaTest
        {
            get
            {
                return this.GetDataModelValue(() => this.EnableAlphaTest);
            }

            set
            {
                this.SetDataModelValue(value, () => this.EnableAlphaTest);

                // プロパティ変更通知.
                this.OnPropertyChanged(() => this.AlphaTestSettingVisibility);
            }
        }

        /// <summary>
        /// 半透明の有効/無効を取得または設定します.
        /// </summary>
        [UseDataModelOriginalValue]
        public bool EnableTransparency
        {
            get
            {
                return this.GetDataModelValue(() => this.EnableTransparency);
            }

            set
            {
                this.SetDataModelValue(value, () => this.EnableTransparency);

                // プロパティ変更通知.
                this.OnPropertyChanged(() => this.AlphaBlendVisibility);
            }
        }

        /// <summary>
        /// パーティクルソートを設定または取得します.
        /// </summary>
        [UseDataModelOriginalValue]
        public int ParticleSortOrder
        {
            get
            {
                return this.GetDataModelValue(() => this.ParticleSortOrder);
            }

            set
            {
                this.SetDataModelValue(value, () => this.ParticleSortOrder);

                // プロパティ変更通知.
                this.OnPropertyChanged(() => this.ZSortCommentVisibility);
            }
        }

        /// <summary>
        /// 深度テストの表示状態.
        /// </summary>
        public bool ZBufferTestVisibility
        {
            get
            {
                // Zバッファ・αテストが"カスタム"の時表示.
                return this.ZbufferAlphaTestType == 4;
            }
        }

        /// <summary>
        /// 深度テスト設定の表示状態.
        /// </summary>
        public bool ZBufferTestSettingVisibility
        {
            get
            {
                // Zバッファ・αテストが"カスタム"の時表示.
                return this.ZbufferAlphaTestType == 4 && this.EnableZBufferTest;
            }
        }

        /// <summary>
        /// アルファテストの表示状態.
        /// </summary>
        public bool AlphaTestVisibility
        {
            get
            {
                // Zバッファ・αテストが"カスタム"の時表示.
                return this.ZbufferAlphaTestType == 4;
            }
        }

        /// <summary>
        /// アルファテスト設定の表示状態.
        /// </summary>
        public bool AlphaTestSettingVisibility
        {
            get
            {
                // Zバッファ・αテストが"カスタム"の時表示.
                return this.ZbufferAlphaTestType == 4 && this.EnableAlphaTest;
            }
        }

        /// <summary>
        /// 半透明設定の表示状態.
        /// </summary>
        public bool TransparencyVisibility
        {
            get
            {
                // Zバッファ・αテストが"カスタム"の時だけ表示.
                return this.ZbufferAlphaTestType == 4;
            }
        }

        /// <summary>
        /// ブレンドタイプ設定の表示状態.
        /// </summary>
        public bool AlphaBlendVisibility
        {
            get
            {
                // Zバッファ・αテストが"抜き"もしくは"不透明"の時非表示.
                return this.ZbufferAlphaTestType == 0
                    || this.ZbufferAlphaTestType == 1
                    || (this.ZbufferAlphaTestType == 4 && this.EnableTransparency);
            }
        }

        /// <summary>
        /// "Zソート"によるパーティクルソートの説明文の表示/非表示
        /// </summary>
        public bool ZSortCommentVisibility
        {
            get
            {
                return this.ParticleSortOrder == 2 && !this.parentViewModel.EmitterBasicBasicViewModel.IsUsingCpu;
            }
        }

        /// <summary>
        /// Zバッファ・αテストの説明文.
        /// </summary>
        public string ZbufferAlphaTestTypeDesc
        {
            get
            {
                if (this.ZbufferAlphaTestType == 0)
                {
                    return Properties.Resources.EmitterZBufATestIgnoreComment;
                }
                else if (this.ZbufferAlphaTestType == 1)
                {
                    return Properties.Resources.EmitterZBufATestNormalComment;
                }
                else if (this.ZbufferAlphaTestType == 2)
                {
                    return Properties.Resources.EmitterZBufATestOpaqueComment;
                }
                else if (this.ZbufferAlphaTestType == 3)
                {
                    return Properties.Resources.EmitterZBufATestTransparentComment;
                }
                else if (this.ZbufferAlphaTestType == 4)
                {
                    return Properties.Resources.EmitterZBufATestCustomComment;
                }

                return string.Empty;
            }
        }

        /// <summary>
        /// Zバッファ・αテストタイプを設定または取得します.
        /// </summary>
        [UseDataModelOriginalValue]
        public int ZbufferAlphaTestType
        {
            get
            {
                return this.GetDataModelValue(() => this.ZbufferAlphaTestType);
            }

            set
            {
                this.SetDataModelValue(value, () => this.ZbufferAlphaTestType);

                // プロパティ変更通知.
                this.OnPropertyChanged(() => this.ZbufferAlphaTestTypeDesc);
                this.OnPropertyChanged(() => this.ZBufferTestVisibility);
                this.OnPropertyChanged(() => this.ZBufferTestSettingVisibility);
                this.OnPropertyChanged(() => this.AlphaTestVisibility);
                this.OnPropertyChanged(() => this.AlphaTestSettingVisibility);
                this.OnPropertyChanged(() => this.TransparencyVisibility);
                this.OnPropertyChanged(() => this.AlphaBlendVisibility);
            }
        }

        /// <summary>
        /// 描画パスインデックスを取得または設定します。
        /// </summary>
        [UseDataModelOriginalValue]
        public int DrawPath
        {
            get
            {
                return this.GetDataModelValue(() => this.DrawPath);
            }

            set
            {
                string drawPathName = string.Empty;

                // 描画パス名を更新
                foreach (KeyValuePair<string, object> item in this.RenderPathTypeItems)
                {
                    int i = (int)item.Value;
                    if (i == value)
                    {
                        drawPathName = item.Key;
                        break;
                    }
                }

                Debug.Assert(!this.RenderPathTypeItems.Any() || string.IsNullOrEmpty(drawPathName) == false, "描画パスインデックスが不正。");

                using (new CommandCombiner())
                {
                    this.SetDataModelValue(drawPathName, () => this.DataModel.DrawPathName);

                    //// データモデルの描画パスが設定されているとき
                    // 一致する描画パスをコンボボックスアイテムから検索
                    var drawPaths = OptionStore.ProjectConfig.DrawPaths;
                    var findItem = drawPaths.FirstOrDefault(item => item.Text == drawPathName);

                    if (findItem != null)
                    {
                        this.SetDataModelValue(findItem.ShaderCompileDef1, () => this.DataModel.DrawPathShaderDef1);
                        this.SetDataModelValue(findItem.ShaderCompileDef2, () => this.DataModel.DrawPathShaderDef2);
                    }

                    // 描画パスインデックスを更新
                    // ここでバイナリデータ全体がコンバートされるので、処理の最後で行う
                    this.SetDataModelValue(value, () => this.DrawPath);
                    var emitter = this.FindNearestParentOfType<EmitterViewModel>();
                    if (emitter != null && emitter.EmitterCustomShaderViewModel != null)
                    {
                        emitter.EmitterCustomShaderViewModel.EmitterCustomShaderBasicViewModel.RaiseShaderDefinitionProperty();
                    }
                }
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// Update child view models with the current data model.
        /// This method is usually called when data model is modified, thus some child
        /// view models might need to be created or removed according to the data model.
        /// </summary>
        public override void UpdateChildViewModels()
        {
            base.UpdateChildViewModels();
            this.SetupRenderPathTypeItems();
        }

        /// <summary>
        /// "Zソート"によるパーティクルソートの説明文の表示/非表示の更新
        /// </summary>
        public void UpdateZSortCommentVisibility()
        {
            this.OnPropertyChanged(() => this.ZSortCommentVisibility);
        }

        /// <summary>
        /// オプションを変更したときの処理を行います.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The event arguments.</param>
        protected override void OnOptionChanged(object sender, EventArgs e)
        {
            base.OnOptionChanged(sender, e);

            if (ViewModelBase.GetParent<EmitterSetViewModel>(this) == null)
            {
                // 削除したエミッタや、新規作成してアンドゥしたエミッタがある場合、親が不在のエミッタビューモデルに対してイベントハンドラが走る
                return;
            }

            this.SetupRenderPathTypeItems();
        }

        /// <summary>
        /// 描画パスのコンボボックスアイテムを設定します.
        /// </summary>
        private void SetupRenderPathTypeItems()
        {
            // 登録されている描画パスを取得
            var drawPaths = OptionStore.ProjectConfig.DrawPaths;

            // デフォルトの描画パスを取得
            var defaultDrawPath = OptionStore.ProjectConfig.DefaultDrawPath;

            // コンボボックスアイテムの文字色をリセット
            this.RenderPathTypeColors = null;

            // データモデルに設定された描画パスを取得
            string drawPathName = this.GetDataModelValue<string>(() => this.DataModel.DrawPathName);

            if (drawPaths.Count == 0)
            {
                //// オプションに描画パスが設定されていないとき
                if (string.IsNullOrEmpty(drawPathName))
                {
                    //// オプションに描画パスがなくて、データモデルの描画パスも設定されていないとき
                    //// コンボボックスのアイテムを空にして変更できないようにする
                    this.DrawPathEnabled = false;
                    this.RenderPathTypeItems = null;
                    this.DrawPathWarningVisibility = false;

                    this.RenderPathTypeItems = new KeyValuePair<string, object>[] { };
                }
                else
                {
                    //// オプションに描画パスがなくて、データモデルの描画パスだけ設定されているとき
                    //// コンボボックスアイテムをデータモデルの描画パスだけにして変更できないようにする
                    this.DrawPathEnabled = false;
                    this.DrawPathWarningVisibility = true;

                    int drawPathIndex = this.GetDataModelValue<int>(() => this.DrawPath);
                    this.RenderPathTypeItems = new KeyValuePair<string, object>[]
                    {
                        new KeyValuePair<string, object>(drawPathName, drawPathIndex)
                    };
                }
            }
            else
            {
                //// オプションに描画パスが設定されているとき
                this.DrawPathEnabled = true;

                // コンボボックスのアイテムを作成
                var renderPathTypeItems = new List<KeyValuePair<string, object>>(drawPaths.Count);

                // this.DrawPathを設定するときにthis.RenderPathTypeItemsにアクセスするので
                // 先に参照を設定しておかないとNullReferenceExceptionが出る
                this.RenderPathTypeItems = renderPathTypeItems;

                foreach (var path in drawPaths)
                {
                    var pair = new KeyValuePair<string, object>(path.Text, path.Id);
                    renderPathTypeItems.Add(pair);
                }

                if (string.IsNullOrEmpty(drawPathName))
                {
                    //// データモデルの描画パスが設定されていないとき
                    // 描画パスをデフォルトに設定
                    this.DrawPath = defaultDrawPath.Id;
                }
                else
                {
                    //// データモデルの描画パスが設定されているとき
                    // 一致する描画パスをコンボボックスアイテムから検索
                    var findItemByName = drawPaths.FirstOrDefault(item => item.Text.Equals(drawPathName));
                    var findItemById = drawPaths.FirstOrDefault(item => item.Id == this.DrawPath);

                    if (findItemByName != null)
                    {
                        //// 一致する描画パスがコンボボックスアイテムにあるとき、それを選択状態にする
                        this.DrawPath = findItemByName.Id;
                    }
                    else if (findItemById != null)
                    {
                        if (OptionStore.RuntimeOptions.IsBinaryConverting)
                        {
                            //// コマンドラインコンバート時は名前での一致を必須とするので、
                            //// 補正は行わずコンバータの処理でエラーとして弾く
                        }
                        else
                        {
                            //// 名前は一致しなくてもIDが一致する描画パスがある場合は警告を出して名前を変更
                            this.SetDataModelValue(findItemById.Text, () => this.DataModel.DrawPathName);
                            var diag = WorkspaceRootViewModel.Instance.Dialogs.OnShowWarningDialogExecutable;
                            if (diag != null)
                            {
                                diag.Execute(new[]
                                {
                                    Resources.WarningSameDrawPathIdButDifferntName,
                                    Properties.Resources.WarningCaption,
                                });
                            }
                        }
                    }
                    else
                    {
                        //// 一致する描画パスがコンボボックスアイテムにないとき
                        //// コンボボックスアイテムに描画パスを追加して、その描画パスだけ文字色を変える
                        this.DrawPathWarningVisibility = true;

                        renderPathTypeItems.Insert(
                            0,
                            new KeyValuePair<string, object>(drawPathName, this.DrawPath));

                        // 文字色を設定
                        this.RenderPathTypeColors = new Color[renderPathTypeItems.Count];
                        for (int i = 1; i < this.RenderPathTypeColors.Length; ++i)
                        {
                            this.RenderPathTypeColors[i] = Color.Black;
                        }

                        this.RenderPathTypeColors[0] = Color.Red;
                    }
                }
            }

            // Notify UI to update.
            var emitter = this.FindNearestParentOfType<EmitterViewModel>();
            if (emitter != null && emitter.EmitterCustomShaderViewModel != null)
            {
                emitter.EmitterCustomShaderViewModel.EmitterCustomShaderBasicViewModel.RaiseShaderDefinitionProperty();
            }

            this.OnPropertyChanged(() => this.RenderPathTypeItems);
            this.OnPropertyChanged(() => this.RenderPathTypeColors);
            this.OnPropertyChanged(() => this.DrawPathEnabled);
            this.OnPropertyChanged(() => this.DrawPath);
        }
    }
}
