﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Serialization;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.UILogic.TypeSerializers
{
    /// <summary>
    /// Type serializer for IXmlSerializable interface.
    /// </summary>
    public class XmlSerializableXmlTypeSerializer : IXmlTypeSerializer
    {
        /// <summary>
        /// The default encoding to use to serialize and deserialize.
        /// </summary>
        private static readonly Encoding DefaultEncoding = Encoding.UTF8;

        /// <summary>
        /// Stores the type cache.
        /// </summary>
        private Dictionary<string, Type> typeCache = new Dictionary<string, Type>();

        /// <summary>
        /// Checks whether the given type is supported or not.
        /// </summary>
        /// <param name="type">The type to check upon.</param>
        /// <returns>Returns true if type is supported, false otherwise.</returns>
        public bool IsSupportedType(Type type)
        {
            return typeof(IXmlSerializable).IsAssignableFrom(type);
        }

        /// <summary>
        /// Serializes to XML.
        /// </summary>
        /// <param name="name">The name of the variable to serialize.</param>
        /// <param name="value">The value to serialize.</param>
        /// <returns>Returns an XElement containing the serialized value.</returns>
        public XElement Serialize(string name, object value)
        {
            if (string.IsNullOrWhiteSpace(name))
            {
                throw new ArgumentException("name");
            }

            var serializable = value as IXmlSerializable;
            if (serializable == null)
            {
                return null;
            }

            byte[] data = null;

            try
            {
                using (var stream = new MemoryStream())
                {
                    var ser = new XmlSerializer(serializable.GetType());
                    ser.Serialize(stream, serializable);

                    var len = (int)stream.Length;
                    data = new byte[len];
                    stream.Position = 0;
                    stream.Read(data, 0, len);
                }

                var str = DefaultEncoding.GetString(data);

                return new XElement(
                    "IXmlSerializable",
                    new XAttribute("name", name),
                    new XAttribute("type", serializable.GetType().FullName),
                    XElement.Parse(str));
            }
            catch
            {
                return null;
            }
        }

        /// <summary>
        /// Deserializes from XML.
        /// </summary>
        /// <param name="xmlElement">XElement to get data from.</param>
        /// <param name="name">Name of the variable.</param>
        /// <param name="value">Deserialized value.</param>
        /// <returns>Returns true if deserialization was successful, false otherwise.</returns>
        public bool Deserialize(XElement xmlElement, out string name, out object value)
        {
            name = null;
            value = null;

            if (xmlElement == null)
            {
                throw new ArgumentNullException("xmlElement");
            }

            if (xmlElement.Name != "IXmlSerializable")
            {
                return false;
            }

            name = (string)xmlElement.Attribute("name");
            if (string.IsNullOrWhiteSpace(name))
            {
                return false;
            }

            var typeName = (string)xmlElement.Attribute("type");
            if (string.IsNullOrWhiteSpace(typeName))
            {
                return false;
            }

            Type serializableType = this.GetTypeFromFullname(typeName);
            if (serializableType == null)
            {
                return false;
            }

            try
            {
                var contentValue = xmlElement.Elements().First().ToString();

                using (var stream = new MemoryStream(DefaultEncoding.GetBytes(contentValue)))
                {
                    var ser = new XmlSerializer(serializableType);
                    value = ser.Deserialize(stream);
                }

                return true;
            }
            catch
            {
                return false;
            }
        }

        /// <summary>
        /// Retrieve a type application-wise.
        /// </summary>
        /// <param name="typeFullname">The fullname of the type.</param>
        /// <returns>Returns the type if found, null otherwise.</returns>
        private Type GetTypeFromFullname(string typeFullname)
        {
            // lookup in mscorlib.dll and the currently executing assembly
            Type type = Type.GetType(typeFullname);
            if (type != null)
            {
                return type;
            }

            // lookup in the local type cache
            if (this.typeCache.TryGetValue(typeFullname, out type))
            {
                return type;
            }

            // lookup in the assemblies loaded in the running application domain
            foreach (Assembly asm in AppDomain.CurrentDomain.GetAssemblies())
            {
                type = asm.GetType(typeFullname);
                if (type != null)
                {
                    this.typeCache.Add(typeFullname, type);
                    return type;
                }
            }

            // nothing found
            return null;
        }
    }
}
