﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Extensions;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// A CommandBase class that set and reset a whole DataModel of a ViewModel.
    /// </summary>
    [Alias("SetDataModel")]
    public class SetDataModelCommand : CommandBase
    {
        /// <summary>
        /// Stores the command stack target.
        /// </summary>
        private object commandStackTarget;

        /// <summary>
        /// Stores the ViewModel given as parameter.
        /// </summary>
        private HierarchyViewModel viewModel;

        /// <summary>
        /// Stores a clone of the DataModel of the ViewModel given as parameter.
        /// </summary>
        private DataModelBase currentDataModel;

        /// <summary>
        /// Stores the new DataModel given as parameter.
        /// </summary>
        private DataModelBase newDataModel;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public SetDataModelCommand()
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="viewModel">ビューモデル</param>
        /// <param name="newDataModel">新しいデータモデル</param>
        public SetDataModelCommand(HierarchyViewModel viewModel, DataModelBase newDataModel)
        {
            object[] args = new object[] { viewModel, newDataModel };

            this.Initialize(args);
        }

        /// <summary>
        /// Initializes the SetDataModel command.
        /// </summary>
        /// <param name="args">Command arguments.</param>
        /// <returns>Returns true on success, false otherwise.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 2)
            {
                return false;
            }

            this.viewModel = args[0] as HierarchyViewModel;
            if (this.viewModel == null)
            {
                return false;
            }

            var newDataModel = args[1] as DataModelBase;
            if (newDataModel == null)
            {
                return false;
            }

            using (new EnableDataModelCloneSetter())
            {
                this.currentDataModel = (DataModelBase)this.viewModel.Proxy.DataModel.Clone();
                this.newDataModel = (DataModelBase)newDataModel.Clone();
            }

            // コピー対象となったデータモデルからチャイルドエミッタを削除
            var emitterData = this.newDataModel as EmitterData;
            if (emitterData != null)
            {
                var childEmitterList = this.newDataModel.Children.OfType<EmitterData>().ToList();
                foreach (var emitter in childEmitterList)
                {
                    this.newDataModel.RemoveChild(emitter);
                }

                emitterData.EmitterList.Clear();
            }

            this.commandStackTarget =
                CommandBase.FindCommandStackTargetFromHierarchyObject(this.viewModel);

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// Gets the command target.
        /// </summary>
        /// <returns>Returns the command target.</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Sets the new DataModel into the ViewModel and sends PropertyChange requests.
        /// </summary>
        /// <returns>Returns true on success or throw an exception.</returns>
        public override bool Do()
        {
            return this.PasteCore(this.newDataModel);
        }

        /// <summary>
        /// Sets the previous DataModel into the ViewModel and sends PropertyChange requests.
        /// </summary>
        /// <returns>Returns true on success or throw an exception.</returns>
        public override bool Undo()
        {
            return this.PasteCore(this.currentDataModel);
        }

        /// <summary>
        /// ペースト処理のコアです。
        /// </summary>
        /// <param name="values">値</param>
        /// <returns>成功したらtrue,失敗したらfalse.</returns>
        private bool PasteCore(DataModelBase values)
        {
            ExportableViewModel.IsPasting = true;

            // DataModelの差し替え
            DataModelBase dataModel = this.viewModel.Proxy.DataModel;

            using (new EnableDataModelCloneSetter())
            {
                dataModel.SetWithoutGuid(values);
            }

            // Update the child view models with the new data model.
            this.viewModel.UpdateChildViewModels();

            // プロパティ変更を通知
            this.viewModel.FirePropertyChanges();

            // メニューの状態を更新
            WorkspaceRootViewModel.Instance.UpdateUIStates();

            // Send the whole binary again.
            using (new ForceRefreshBinary())
            {
                ViewerMessageHelper.FindPropertyParentDataAndSend(this.viewModel.Proxy.DataModel);
            }

            ExportableViewModel.IsPasting = false;

            // カーブエディタウィンドウの更新
            WorkspaceRootViewModel.Instance.UpdateCurveEditorWindow(this.viewModel);

            return true;
        }
    }
}
