﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// 名前変更コマンドです.
    /// </summary>
    [Alias("Rename")]
    public class RenameCommand : CommandBase
    {
        /// <summary>
        /// 名前を変更するビューモデル.
        /// </summary>
        private HierarchyViewModel viewModel = null;

        /// <summary>
        /// 変更後の名前.
        /// </summary>
        private string newName = string.Empty;

        /// <summary>
        /// 変更前の名前.
        /// </summary>
        private string oldName = string.Empty;

        /// <summary>コマンドスタックを持つターゲット.</summary>
        private object commandStackTarget = null;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RenameCommand()
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="viewModel">リネームするビューモデル</param>
        /// <param name="newName">リネーム後の名前</param>
        /// <param name="oldName">リネーム前の名前</param>
        public RenameCommand(HierarchyViewModel viewModel, string newName, string oldName)
        {
            object[] args = new object[] { viewModel, newName, oldName };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">
        /// The arguments for the command.
        /// args[0] : リネームするビューモデル
        /// args[1] : リネーム後の名前
        /// args[2] : リネーム前の名前
        /// </param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 3)
            {
                return false;
            }

            // 名前を変更するビューモデルを取得.
            this.viewModel = args[0] as HierarchyViewModel;
            if (this.viewModel == null)
            {
                return false;
            }

            // 変更後の名前を取得.
            this.newName = args[1] as string;
            if (string.IsNullOrEmpty(this.newName))
            {
                return false;
            }

            // 変更前の名前を取得
            this.oldName = args[2] as string;
            if (string.IsNullOrEmpty(this.oldName))
            {
                return false;
            }

            // 遡ってCommandStackOwnerAttributeを持つインスタンスを取得.
            var cmdStackTarget = this.viewModel as IHierarchyObject;
            this.commandStackTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(cmdStackTarget) as IHierarchyObject;

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            var dataModel = this.viewModel.Proxy.DataModel;

            // エミッタやエミッタセットのリネームは、まず削除を先に行う
            var eset = dataModel.FindSelfOrParentOfType<EmitterSetData>();
            if (eset != null)
            {
                ViewerMessageHelper.RequestDeleteEmitterSet(eset);
            }

            // リネームして
            this.viewModel.SetName(this.newName);

            // UIに反映して
            this.UpdatePageView();

            // リロード
            using (new ForceRefreshBinary())
            {
                ViewerMessageHelper.FindPropertyParentDataAndSend(dataModel);
            }

            WorkspaceRootViewModel.Instance.UpdateUIStates();

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            var dataModel = this.viewModel.Proxy.DataModel;

            // エミッタやエミッタセットのリネームは、まず削除を先に行う
            var eset = dataModel.FindSelfOrParentOfType<EmitterSetData>();
            if (eset != null)
            {
                ViewerMessageHelper.RequestDeleteEmitterSet(eset);
            }

            this.viewModel.SetName(this.oldName);
            this.UpdatePageView();

            // Send the whole binary again.
            using (new ForceRefreshBinary())
            {
                ViewerMessageHelper.FindPropertyParentDataAndSend(dataModel);
            }

            WorkspaceRootViewModel.Instance.UpdateUIStates();

            return true;
        }

        /// <summary>
        /// ページビュー中の名前を更新します。
        /// </summary>
        private void UpdatePageView()
        {
            var esetVm = this.viewModel as EmitterSetViewModel;
            if (esetVm != null)
            {
                esetVm.EmitterSetBasicViewModel.EmitterSetBasicBasicViewModel.RaiseNameChanged();
            }

            var prevFileVm = HierarchyObjectExtensions.FindChildrenRecursive(
                null,
                this.viewModel,
                (n, o) => o is PreviewBasicFileViewModel) as PreviewBasicFileViewModel;
            if (prevFileVm != null)
            {
                prevFileVm.RaiseNameChanged();
                prevFileVm.RaiseFilePathChanged();
            }
        }
    }
}
