﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// Command to remove reserved shader node from emitter.
    /// </summary>
    [Alias("RemoveReservedShader")]
    public class RemoveReservedShaderCommand : CommandBase
    {
        /// <summary>The emitter view model to remove reserved shader from.</summary>
        private EmitterViewModel parentViewModel = null;

        /// <summary>The reserved shader node view model to be removed.</summary>
        private ReservedShaderNodeViewModel reservedShaderViewModel = null;

        /// <summary>The index of the reserved shader view model in the parent's child list.</summary>
        private int childListIndex = 0;

        /// <summary>The command stack target.</summary>
        private IHierarchyObject commandStackTarget = null;

        /// <summary>The node to be selected after the reserved shader is removed</summary>
        private WorkspaceNodeViewModelBase selectNode = null;

        /// <summary>以前選択していたエミッタの処理タイプ</summary>
        private int previousUsingCpu = -1;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RemoveReservedShaderCommand()
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="reservedShaderViewModel">削除するエミッタプラグインのビューモデル</param>
        public RemoveReservedShaderCommand(ReservedShaderNodeViewModel reservedShaderViewModel)
        {
            object[] args = new object[] { reservedShaderViewModel };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">
        /// The arguments for the command.
        /// [0] : The reserved shader view model to remove.
        /// </param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            // Check the arguments.
            if (args == null || args.Length != 1)
            {
                return false;
            }

            // Get the view model to remove.
            this.reservedShaderViewModel = args[0] as ReservedShaderNodeViewModel;
            if (this.reservedShaderViewModel == null)
            {
                return false;
            }

            // Get the parent view model.
            this.parentViewModel = this.reservedShaderViewModel.Parent as EmitterViewModel;
            if (this.parentViewModel == null)
            {
                return false;
            }

            // Save the index of the reserved shader view model in the parent's child list.
            this.parentViewModel.GetIndexInChildList(
                this.reservedShaderViewModel,
                out this.childListIndex);

            // Find the target command stack by looping through the parents of the emitter
            // and find the first parent who has CommandStackOwnerAttribute defined.
            var cmdStackTarget = this.parentViewModel as IHierarchyObject;
            this.commandStackTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(cmdStackTarget) as IHierarchyObject;

            this.selectNode = this.parentViewModel as WorkspaceNodeViewModelBase;

            // Get the node to be selected after removal.
            this.selectNode = RemoveNodeCommandUtil.GetSelectNode(
                this.parentViewModel,
                this.reservedShaderViewModel);

            if (this.selectNode == null)
            {
                return false;
            }

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// Get the target command stack.
        /// </summary>
        /// <returns>The target command stack.</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // 親ノードを選択する.
                // (削除したはずのノードページが操作できてしまうことを防ぐため)
                // この操作は、ノードを削除する前に実行する必要がある.
                // またノードの削除処理は、一時的に親ノードから全てのチャイルドを削除するので
                // ここで選択するのは親ノードにする必要がある.
                this.parentViewModel.IsSelected = true;

                // Remove the view model from it's parent.
                this.parentViewModel.RemoveReservedShader(this.reservedShaderViewModel);

                WorkspaceRootViewModel.Instance.UpdateUIStates();

                // Change the selection state of the node.
                this.selectNode.IsSelected = true;

                this.previousUsingCpu =
                    this.parentViewModel.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu();
                this.parentViewModel.EmitterBasicViewModel.EmitterBasicBasicViewModel.IsUsingCpu = false;
            }

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // 元あったインデックスにストライプを戻す.
                bool insertResult = this.parentViewModel.InsertReservedShader(
                    this.reservedShaderViewModel,
                    this.childListIndex);

                if (insertResult == false)
                {
                    return false;
                }

                // このビューモデルのノードを選択する.
                // この操作はノードを追加した後に実行する必要がある.
                var node = this.reservedShaderViewModel as WorkspaceNodeViewModelBase;
                if (node != null)
                {
                    node.IsSelected = true;
                }

                this.parentViewModel.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu(
                    this.previousUsingCpu);

                OptionStore.TriggerOptionChangedEvent(null);
            }

            return true;
        }
    }
}
