﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// RemovePreviewCommand.
    /// </summary>
    [Alias("RemovePreview")]
    public class RemovePreviewCommand : CommandBase
    {
        /// <summary>
        /// 親ビューモデルです.
        /// </summary>
        private EmitterSetViewModel parentViewModel = null;

        /// <summary>
        /// プレビュービューモデルです.
        /// </summary>
        private PreviewViewModel previewViewModel = null;

        /// <summary>
        /// ビューモデルのインデックスです.
        /// </summary>
        private int origViewModelIndex = -1;

        /// <summary>
        /// データモデルのインデックスです.
        /// </summary>
        private int origDataModelIndex = -1;

        /// <summary>The command stack target.</summary>
        private object commandStackTarget = null;

        /// <summary>エミッタの削除後に選択するノードです.</summary>
        private WorkspaceNodeViewModelBase selectNode = null;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RemovePreviewCommand()
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="previewViewModel">削除するプレビューのビューモデル</param>
        public RemovePreviewCommand(PreviewViewModel previewViewModel)
        {
            object[] args = new object[] { previewViewModel };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">
        /// The arguments for the command.
        /// [0] : 削除するプレビューのビューモデル
        /// </param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            // 引数の数をチェック
            if (args == null || args.Length != 1)
            {
                return false;
            }

            // 削除するプレビューのビューモデルを取得.
            this.previewViewModel = args[0] as PreviewViewModel;
            if (this.previewViewModel == null)
            {
                return false;
            }

            // 親ビューモデルを取得
            this.parentViewModel = this.previewViewModel.Parent as EmitterSetViewModel;
            if (this.parentViewModel == null)
            {
                return false;
            }

            // 親エミッタセットにプレビューが1つしかないときは削除しない.
            if (this.parentViewModel.Proxy.PreviewList.Count <= 1)
            {
                return false;
            }

            // プレビューノードのインデックスを取得
            this.parentViewModel.GetPreviewIndex(
                this.previewViewModel,
                out this.origViewModelIndex,
                out this.origDataModelIndex);
            if (this.origViewModelIndex < 0 || this.origDataModelIndex < 0)
            {
                return false;
            }

            // エミッタの削除後に選択するノードを取得
            this.selectNode = RemoveNodeCommandUtil.GetSelectNode(this.parentViewModel, this.previewViewModel);
            if (this.selectNode == null)
            {
                return false;
            }

            // 遡ってCommandStackOwnerAttributeを持つインスタンスをチェックする
            var cmdStackTarget = this.parentViewModel as IHierarchyObject;
            this.commandStackTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(cmdStackTarget) as IHierarchyObject;

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // 親ノードを選択する.
                // (削除したはずのノードページが操作できてしまうことを防ぐため)
                // この操作は、ノードを削除する前に実行する必要がある.
                // またノードの削除処理は、一時的に親ノードから全てのチャイルドを削除するので
                // ここで選択するのは親ノードにする必要がある.
                this.parentViewModel.IsSelected = true;

                // プレビューを削除.
                bool removeResult = this.parentViewModel.RemovePreview(this.previewViewModel);
                if (removeResult == false)
                {
                    return false;
                }

                // ノードの選択を変更
                this.selectNode.IsSelected = true;
            }

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // 元あったインデックスにプレビューを戻す.
                bool insertResult = this.parentViewModel.InsertPreview(
                    this.previewViewModel,
                    this.origViewModelIndex,
                    this.origDataModelIndex);
                if (insertResult == false)
                {
                    return false;
                }

                this.previewViewModel.IsSelected = true;

                OptionStore.TriggerOptionChangedEvent(null);
            }

            return true;
        }
    }
}
