﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// ノード削除コマンドユーティリティ。
    /// </summary>
    public static class RemoveNodeCommandUtil
    {
        /// <summary>
        /// 削除後に選択するノードを取得します。
        /// </summary>
        /// <param name="parentNode">親ノード</param>
        /// <param name="removeNode">削除するノード</param>
        /// <returns>削除後に選択するノードを返します。</returns>
        public static WorkspaceNodeViewModelBase GetSelectNode(
            WorkspaceNodeViewModelBase parentNode,
            WorkspaceNodeViewModelBase removeNode)
        {
            WorkspaceNodeViewModelBase selectNode = null;

            // 次のノードを選択ノードに
            selectNode = GetNextNode(parentNode, removeNode);
            if (selectNode != null)
            {
                return selectNode;
            }

            // なければ前のエミッタを選択ノードに
            selectNode = GetPrevNode(parentNode, removeNode);
            if (selectNode != null)
            {
                return selectNode;
            }

            // なければ親ノードを選択ノードに
            return parentNode;
        }

        /// <summary>
        /// ノード順で前にあるノードを取得します。
        /// </summary>
        /// <param name="parentNode">親ノード</param>
        /// <param name="currentNode">ノード</param>
        /// <returns>前にあるノードを返します。</returns>
        public static WorkspaceNodeViewModelBase GetPrevNode(
            WorkspaceNodeViewModelBase parentNode,
            WorkspaceNodeViewModelBase currentNode)
        {
            bool find = false;
            WorkspaceNodeViewModelBase prevNode = null;

            foreach (IHierarchyObject child in parentNode.Children)
            {
                var node = child as WorkspaceNodeViewModelBase;
                if (node == null)
                {
                    continue;
                }

                if (node == currentNode)
                {
                    find = true;
                    break;
                }

                prevNode = node;
            }

            if (find == true)
            {
                if (prevNode != null)
                {
                    if (IsKindSame(currentNode, prevNode))
                    {
                        return prevNode;
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// ノード順で次にあるノードを取得します。
        /// </summary>
        /// <param name="parentNode">親ノード</param>
        /// <param name="currentNode">ノード</param>
        /// <returns>次にあるノードを返します。</returns>
        public static WorkspaceNodeViewModelBase GetNextNode(
            WorkspaceNodeViewModelBase parentNode,
            WorkspaceNodeViewModelBase currentNode)
        {
            bool find = false;
            WorkspaceNodeViewModelBase nextNode = null;

            foreach (IHierarchyObject child in parentNode.Children)
            {
                var node = child as WorkspaceNodeViewModelBase;
                if (node == null)
                {
                    continue;
                }

                if (find == true)
                {
                    nextNode = node;
                    break;
                }

                if (node == currentNode)
                {
                    find = true;
                }
            }

            if (find == true)
            {
                if (nextNode != null)
                {
                    if (IsKindSame(currentNode, nextNode))
                    {
                        return nextNode;
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// ノードの種別がおおまかに同じかどうかを判定します。
        /// </summary>
        /// <param name="nodeA">ノードA</param>
        /// <param name="nodeB">ノードB</param>
        /// <returns>種別が一緒ならtrue,違うならfalse.</returns>
        public static bool IsKindSame(
            WorkspaceNodeViewModelBase nodeA,
            WorkspaceNodeViewModelBase nodeB)
        {
            if (nodeA is FieldViewModel && nodeB is FieldViewModel)
            {
                return true;
            }

            if (nodeA is CustomActionViewModel && nodeB is CustomActionViewModel)
            {
                return true;
            }

            if (nodeA is ReservedShaderNodeViewModel && nodeB is ReservedShaderNodeViewModel)
            {
                return true;
            }

            if (nodeA is PreviewViewModel && nodeB is PreviewViewModel)
            {
                return true;
            }

            if (nodeA is EmitterViewModel && nodeB is EmitterViewModel)
            {
                return true;
            }

            if (nodeA is EmitterSetViewModel && nodeB is EmitterSetViewModel)
            {
                return true;
            }

            return false;
        }
    }
}
