﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// RemoveEmitterSetCommand.
    /// </summary>
    [Alias("RemoveEmitterSet")]
    public class RemoveEmitterSetCommand : CommandBase
    {
        /// <summary>
        /// 親ビューモデルです.
        /// </summary>
        private WorkspaceViewModel parentViewModel = null;

        /// <summary>
        /// エミッタセットビューモデルです.
        /// </summary>
        private EmitterSetViewModel emitterSetViewModel = null;

        /// <summary>
        /// エミッタセットビューモデルのインデックスです.
        /// </summary>
        private int origViewModelIndex = -1;

        /// <summary>
        /// エミッタセットデータモデルのインデックスです.
        /// </summary>
        private int origDataModelIndex = -1;

        /// <summary>エミッタセットの削除後に選択するノードです.</summary>
        private WorkspaceNodeViewModelBase selectNode = null;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RemoveEmitterSetCommand()
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="emitterSetViewModel">削除するエミッタセットのビューモデル</param>
        public RemoveEmitterSetCommand(EmitterSetViewModel emitterSetViewModel)
        {
            object[] args = new object[] { emitterSetViewModel };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">
        /// The arguments for the command.
        /// [0] : 削除するエミッタセットのビューモデル
        /// </param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            // 引数の数をチェック
            if (args == null || args.Length != 1)
            {
                return false;
            }

            // 削除するエミッタセットのビューモデルを取得
            this.emitterSetViewModel = args[0] as EmitterSetViewModel;
            if (this.emitterSetViewModel == null)
            {
                return false;
            }

            // 親ビューモデルを取得
            this.parentViewModel = this.emitterSetViewModel.Parent as WorkspaceViewModel;
            if (this.parentViewModel == null)
            {
                return false;
            }

            // ノードのインデックスを取得
            this.parentViewModel.GetEmitterSetIndex(
                this.emitterSetViewModel,
                out this.origViewModelIndex,
                out this.origDataModelIndex);
            if (this.origViewModelIndex < 0 || this.origDataModelIndex < 0)
            {
                return false;
            }

            // エミッタセットの削除後に選択するノードを取得
            this.selectNode = RemoveNodeCommandUtil.GetSelectNode(this.parentViewModel, this.emitterSetViewModel);
            if (this.selectNode == null)
            {
                return false;
            }

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return null;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            // 親ノードを選択する.
            // (削除したはずのノードページが操作できてしまうことを防ぐため)
            // この操作は、ノードを削除する前に実行する必要がある.
            // またノードの削除処理は、一時的に親ノードから全てのチャイルドを削除するので
            // ここで選択するのは親ノードにする必要がある.
            this.parentViewModel.IsSelected = true;

            this.emitterSetViewModel.DisconnectEmitterAnimetion();

            // エミッタセットを削除
            bool removeResult = this.parentViewModel.RemoveEmitterSet(this.emitterSetViewModel);

            // 削除できなかったとき、エミッターセットノードの選択を戻す.
            if (removeResult == false)
            {
                if (this.parentViewModel.IsExpanded == true)
                {
                    this.emitterSetViewModel.IsSelected = true;
                    this.emitterSetViewModel.UpdateEmitterAnimetion();
                }

                return false;
            }

            // ノードの選択を変更
            this.selectNode.IsSelected = true;
            if (this.selectNode is EmitterSetViewModel)
            {
                ((EmitterSetViewModel)this.selectNode).UpdateEmitterAnimetion();
            }

            ViewerMessageHelper.RequestDeleteEmitterSet(this.emitterSetViewModel.DataModel);

            // FileMonitorManagerによるファイル監視を解除するため、EmitterSetのリソースを解放する。
            // リソースの解放時に、ファイル監視も解除される。
            emitterSetViewModel.Dispose();

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            // 元あったインデックスにエミッタセットを戻す.
            bool insertResult = this.parentViewModel.InsertEmitterSet(
                this.emitterSetViewModel,
                this.origViewModelIndex,
                this.origDataModelIndex);
            if (insertResult == false)
            {
                return false;
            }

            this.emitterSetViewModel.UpdateEmitterAnimetion();
            this.emitterSetViewModel.IsSelected = true;
            OptionStore.TriggerOptionChangedEvent(null);
            ViewerMessageHelper.SendEmitterSet(this.emitterSetViewModel.DataModel);

            return true;
        }
    }
}
