﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// RemoveCustomActionCommand.
    /// </summary>
    [Alias("RemoveCustomAction")]
    public class RemoveCustomActionCommand : CommandBase
    {
        /// <summary>親のビューモデルです.</summary>
        private EmitterViewModel parentViewModel = null;

        /// <summary>
        /// カスタムアクションのビューモデルです.
        /// </summary>
        private CustomActionViewModel customActionViewModel;

        /// <summary>The index of the emitter view model.</summary>
        private int origViewModelIndex = -1;

        /// <summary>コマンドスタックを持つターゲット.</summary>
        private object commandStackTarget = null;

        /// <summary>カスタムアクションの削除後に選択するノードです.</summary>
        private WorkspaceNodeViewModelBase selectNode = null;

        /// <summary>以前選択していたエミッタの処理タイプ</summary>
        private int previousUsingCpu = -1;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RemoveCustomActionCommand()
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="customActionViewModel">削除するカスタムアクションのビューモデル</param>
        public RemoveCustomActionCommand(CustomActionViewModel customActionViewModel)
        {
            object[] args = new object[] { customActionViewModel };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">
        /// The arguments for the command.
        /// [0] : 削除するカスタムアクションのビューモデル
        /// </param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            // 引数の数をチェック
            if (args == null || args.Length != 1)
            {
                return false;
            }

            // RemoveCustomActionCommandビューモデルを取得
            this.customActionViewModel = args[0] as CustomActionViewModel;
            if (this.customActionViewModel == null)
            {
                return false;
            }

            // Get the parent view model.
            this.parentViewModel = this.customActionViewModel.Parent as EmitterViewModel;
            if (this.parentViewModel == null)
            {
                return false;
            }

            // Find and save the index of the emitter.
            this.parentViewModel.GetCustomActionIndex(this.customActionViewModel, out this.origViewModelIndex);
            if (this.origViewModelIndex < 0)
            {
                return false;
            }

            // 遡ってCommandStackOwnerAttributeを持つインスタンスを取得.
            var cmdStackTarget = this.parentViewModel as IHierarchyObject;
            this.commandStackTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(cmdStackTarget) as IHierarchyObject;

            this.selectNode = this.parentViewModel as WorkspaceNodeViewModelBase;

            // カスタムアクションの削除後に選択するノードを取得
            this.selectNode = RemoveNodeCommandUtil.GetSelectNode(this.parentViewModel, this.customActionViewModel);
            if (this.selectNode == null)
            {
                return false;
            }

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // 親ノードを選択する.
                // (削除したはずのノードページが操作できてしまうことを防ぐため)
                // この操作は、ノードを削除する前に実行する必要がある.
                // またノードの削除処理は、一時的に親ノードから全てのチャイルドを削除するので
                // ここで選択するのは親ノードにする必要がある.
                this.parentViewModel.IsSelected = true;

#if false // Test For SendBinary Packet --->
                // Get the emitter set data.
                var emitterSetData = ViewModelBase.GetParent<EmitterSetViewModel>(
                    this.parentViewModel as ViewModelBase).Proxy.DataModel as EmitterSetData;

            {
                var converterInfo = new DataModelValueConverterInfo();
                converterInfo.SendDeleteMessage(emitterSetData);
                converterInfo.SendRemoveBinaryMessage(emitterSetData);
            }
#endif // Test For SendBinary Packet <---

                // Remove the view model from it's parent.
                this.parentViewModel.RemoveCustomAction(this.customActionViewModel);

#if false // Test For SendBinary Packet --->
            {
                var converterInfo = new DataModelValueConverterInfo();
                converterInfo.SendEmitterSetReloadMessage(emitterSetData);
                converterInfo.SendCreateMessage(emitterSetData);
            }
#endif // Test For SendBinary Packet <---

                WorkspaceRootViewModel.Instance.UpdateUIStates();

                // ノードの選択を変更
                this.selectNode.IsSelected = true;

                this.previousUsingCpu =
                    this.parentViewModel.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu();
                this.parentViewModel.EmitterBasicViewModel.EmitterBasicBasicViewModel.IsUsingCpu = false;
            }

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // 元あったインデックスにストライプを戻す.
                bool insertResult = this.parentViewModel.InsertCustomAction(
                    this.customActionViewModel,
                    this.origViewModelIndex);
                if (insertResult == false)
                {
                    return false;
                }

#if false // Test For SendBinary Packet --->
                // Get the emitter set data.
                var emitterSetData = ViewModelBase.GetParent<EmitterSetViewModel>(
                    this.parentViewModel as ViewModelBase).Proxy.DataModel as EmitterSetData;

            {
                var converterInfo = new DataModelValueConverterInfo();
                converterInfo.SendDeleteMessage(emitterSetData);
                converterInfo.SendRemoveBinaryMessage(emitterSetData);

                converterInfo.SendEmitterSetReloadMessage(emitterSetData);
                converterInfo.SendRemoveBinaryMessage(emitterSetData);
            }
#endif // Test For SendBinary Packet <---

                // このビューモデルのノードを選択する.
                // この操作はノードを追加した後に実行する必要がある.
                var node = this.customActionViewModel as WorkspaceNodeViewModelBase;
                if (node != null)
                {
                    node.IsSelected = true;
                }

                this.parentViewModel.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu(
                    this.previousUsingCpu);
                OptionStore.TriggerOptionChangedEvent(null);
            }

            return true;
        }
    }
}
