﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.BusinessLogic.Commands
{
    /// <summary>
    /// CreatePreviewCommand.
    /// </summary>
    [Alias("CreatePreview")]
    public class CreatePreviewCommand : CommandBase
    {
        /// <summary>
        /// 親ビューモデルです.
        /// </summary>
        private EmitterSetViewModel parentViewModel;

        /// <summary>
        /// プレビュービューモデルです.
        /// </summary>
        private PreviewViewModel previewViewModel;

        /// <summary>
        /// プレビュー名です.
        /// </summary>
        private string previewName;

        /// <summary>The command stack target.</summary>
        private object commandStackTarget = null;

        /// <summary>以前選択していたノード.</summary>
        private WorkspaceNodeViewModelBase previousSelectedNode = null;

        /// <summary>
        /// 挿入先のインデックス
        /// </summary>
        private int insertIndex = -1;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public CreatePreviewCommand()
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="parentViewModel">親ビューモデル</param>
        /// <param name="previewName">プレビュー名</param>
        /// <param name="insertIndex">挿入するインデックス</param>
        public CreatePreviewCommand(EmitterSetViewModel parentViewModel, string previewName, int insertIndex = -1)
        {
            object[] args = new object[] { parentViewModel, previewName, insertIndex };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">The arguments for the command.</param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || (args.Length != 2 && args.Length != 3))
            {
                return false;
            }

            // 親ビューモデルを記録
            this.parentViewModel = args[0] as EmitterSetViewModel;
            if (this.parentViewModel == null)
            {
                return false;
            }

            // プレビュー名を記録
            this.previewName = args[1] as string;
            if (string.IsNullOrEmpty(this.previewName))
            {
                return false;
            }

            // 挿入先インデックスの追加
            if (args.Length == 3)
            {
                this.insertIndex = (int)args[2];
            }

            // 遡ってCommandStackOwnerAttributeを持つインスタンスをチェックする
            var cmdStackTarget = this.parentViewModel as IHierarchyObject;
            this.commandStackTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(cmdStackTarget) as IHierarchyObject;

            this.previousSelectedNode = this.parentViewModel as WorkspaceNodeViewModelBase;

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // プレビューを作成.
                this.previewViewModel = this.parentViewModel.CreateNewPreview(this.previewName);
                if (this.previewViewModel == null)
                {
                    return false;  // プレビューの作成に失敗.
                }

                // プレビューを親ノードに追加.
                bool addResult = this.parentViewModel.AddPreview(this.previewViewModel);
                if (addResult == false)
                {
                    return false;
                }

                this.SwapLastToIndex();

                // 作成したプレビューを選択状態に.
                var node = this.previewViewModel as WorkspaceNodeViewModelBase;
                if (node != null)
                {
                    node.IsSelected = true;
                }
            }

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // 親ノードを選択する.
                // (削除したはずのプレビューページが操作できてしまうことを防ぐため)
                // この操作は、ノードを削除する前に実行する必要がある.
                if (this.previousSelectedNode != null)
                {
                    this.previousSelectedNode.IsSelected = true;
                }

                // ノードの変更を元に戻す.
                bool removeResult = this.parentViewModel.RemovePreview(this.previewViewModel);
                if (removeResult == false)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// Redo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Redo()
        {
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // ノードの変更をやり直す.
                bool addResult = this.parentViewModel.AddPreview(this.previewViewModel);
                if (addResult == false)
                {
                    return false;
                }

                this.SwapLastToIndex();
                OptionStore.TriggerOptionChangedEvent(null);
            }

            return true;
        }

        /// <summary>
        /// 挿入先が指定されている場合、末尾から挿入先にフィールドを移動します。
        /// </summary>
        private void SwapLastToIndex()
        {
            if (this.insertIndex == -1)
            {
                return;
            }

            using (new SuppressNodeTypeOrdering(this.parentViewModel))
            {
                int sourceIndex = this.parentViewModel.Children.IndexOf(this.previewViewModel);

                ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;
                this.parentViewModel.Children.RemoveAt(sourceIndex);
                this.parentViewModel.Children.Insert(this.insertIndex + 1, this.previewViewModel);
                ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;

                this.parentViewModel.Proxy.MoveChildDataModel(
                    this.previewViewModel.DataModel,
                    this.insertIndex + 1 - sourceIndex);
            }
        }
    }
}
