﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.BusinessLogic.Commands
{
    /// <summary>
    /// CreateFieldCommand.
    /// </summary>
    [Alias("CreateField")]
    public class CreateFieldCommand : CommandBase
    {
        /// <summary>
        /// 親のビューモデルです.
        /// </summary>
        private EmitterViewModel parentViewModel;

        /// <summary>
        /// フィールドのビューモデル.
        /// 実際には、ランダムなどの具象クラスが作成される.
        /// </summary>
        private FieldViewModel fieldViewModel;

        /// <summary>フィールドのタイプ</summary>
        private FieldTypes fieldType;

        /// <summary>The command stack target.</summary>
        private IHierarchyObject commandStackTarget = null;

        /// <summary>以前選択していたノード.</summary>
        private WorkspaceNodeViewModelBase previousSelectedNode = null;

        /// <summary>挿入するインデックス</summary>
        private int insertIndex = -1;

        /// <summary>以前選択していたエミッタの処理タイプ</summary>
        private int previousUsingCpu = -1;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public CreateFieldCommand()
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="emitterViewModel">親のエミッタ</param>
        /// <param name="emitterSetName">作成するフィールドのタイプ</param>
        /// <param name="insertIndex">挿入するインデックス</param>
        public CreateFieldCommand(HierarchyViewModel emitterViewModel, FieldTypes fieldType, int insertIndex = -1)
        {
            object[] args = new object[] { emitterViewModel, fieldType, insertIndex };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">The arguments for the command.</param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || (args.Length != 2 && args.Length != 3))
            {
                return false;
            }

            // 親のエミッタを記録.
            this.parentViewModel = args[0] as EmitterViewModel;
            if (this.parentViewModel == null)
            {
                return false;
            }

            // 作成するフィールドのタイプを記録.
            if (args[1] is FieldTypes == false)
            {
                return false;
            }

            this.fieldType = (FieldTypes)args[1];

            // 挿入先インデックスの取得
            if (args.Length == 3)
            {
                this.insertIndex = (int)args[2];
            }

            // 遡ってCommandStackOwnerAttributeを持つインスタンスをチェックする.
            var cmdStackTarget = this.parentViewModel as IHierarchyObject;
            this.commandStackTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(cmdStackTarget) as IHierarchyObject;

            this.previousSelectedNode = this.parentViewModel as WorkspaceNodeViewModelBase;

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス.</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // フィールドの作成.
                this.fieldViewModel = this.parentViewModel.CreateNewField(this.fieldType);
                if (this.fieldViewModel == null)
                {
                    return false; // ノードの作成に失敗.
                }

                // フィールドを親ノードに追加.
                bool addResult = this.parentViewModel.AddField(this.fieldViewModel);
                if (addResult == false)
                {
                    return false;
                }

                this.SwapLastToIndex();

                // 作成したフィールドを選択状態にする.
                var node = this.fieldViewModel as WorkspaceNodeViewModelBase;
                if (node != null)
                {
                    node.IsSelected = true;
                }

                this.previousUsingCpu =
                    this.parentViewModel.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu();

                WorkspaceRootViewModel.Instance.UpdateUIStates();
            }

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // 親ノードを選択する.
                // (削除したはずのフィールドページが操作できてしまうことを防ぐため)
                // この操作は、ノードを削除する前に実行する必要がある.
                if (this.previousSelectedNode != null)
                {
                    var node = this.fieldViewModel as WorkspaceNodeViewModelBase;
                    if (node != null && node.IsSelected)
                    {
                        this.previousSelectedNode.IsSelected = true;
                    }
                }

                // ノードの変更を元に戻す.
                bool removeResult = this.parentViewModel.RemoveField(this.fieldViewModel);
                if (removeResult == false)
                {
                    return false;
                }

                this.parentViewModel.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu(
                    this.previousUsingCpu);

                WorkspaceRootViewModel.Instance.UpdateUIStates();
            }

            return true;
        }

        /// <summary>
        /// Redo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Redo()
        {
            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // ノードの変更をやり直す.
                bool addResult = this.parentViewModel.AddField(this.fieldViewModel);
                if (addResult == false)
                {
                    return false;
                }

                this.SwapLastToIndex();

                this.previousUsingCpu =
                    this.parentViewModel.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu();

                OptionStore.TriggerOptionChangedEvent(null);
                WorkspaceRootViewModel.Instance.UpdateUIStates();
            }

            return true;
        }

        /// <summary>
        /// 挿入先が指定されている場合、末尾から挿入先にフィールドを移動します。
        /// </summary>
        private void SwapLastToIndex()
        {
            if (this.insertIndex == -1)
            {
                return;
            }

            using (new SuppressNodeTypeOrdering(this.parentViewModel))
            {
                int sourceIndex = this.parentViewModel.Children.IndexOf(this.fieldViewModel);

                this.parentViewModel.Children.RemoveAt(sourceIndex);
                this.parentViewModel.Children.Insert(this.insertIndex + 1, this.fieldViewModel);

                this.parentViewModel.Proxy.MoveChildDataModel(
                    this.fieldViewModel.Proxy.DataModel,
                    this.insertIndex + 1 - sourceIndex);
            }
        }
    }
}
