﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using EffectMaker.BusinessLogic.DataModelOperation;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.Properties;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.BusinessLogic.Commands
{
    /// <summary>
    /// CreateEmitterExtParamsCommand.
    /// </summary>
    [Alias("CreateEmitterExtParams")]
    public class CreateEmitterExtParamsCommand : CommandBase
    {
        /// <summary>親のビューモデルです.</summary>
        private EmitterViewModel parentViewModel = null;

        /// <summary>エミッタ拡張パラメータのビューモデルです.</summary>
        private EmitterExtParamsViewModel emitterExtParamsViewModel = null;

        /// <summary>The command stack target.</summary>
        private IHierarchyObject commandStackTarget = null;

        /// <summary>以前選択していたノード.</summary>
        private WorkspaceNodeViewModelBase previousSelectedNode = null;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public CreateEmitterExtParamsCommand()
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="parentViewModel">親のエミッタ</param>
        public CreateEmitterExtParamsCommand(EmitterViewModel parentViewModel)
        {
            object[] args = new object[] { parentViewModel };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">The arguments for the command.</param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 1)
            {
                return false;
            }

            // エミッタ拡張パラメータが読み込めていない時は失敗にする
            if (EmitterExtParamsUserDataManager.Definition == null)
            {
                Logger.Log("LogView", LogLevels.Warning, Resources.WarningAbortCreateEmitterExtParams);
                return false;
            }

            // 親のエミッタを記録
            if (args[0] is EmitterViewModel)
            {
                this.parentViewModel = args[0] as EmitterViewModel;
            }

            if (this.parentViewModel == null)
            {
                return false;
            }

            // 遡ってCommandStackOwnerAttributeを持つインスタンスをチェックする
            var cmdStackTarget = this.parentViewModel as IHierarchyObject;
            this.commandStackTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(cmdStackTarget) as IHierarchyObject;

            this.previousSelectedNode = this.parentViewModel as WorkspaceNodeViewModelBase;

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            bool addResult = false;
            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // エミッタ拡張パラメータノードを作成する.
                this.emitterExtParamsViewModel = this.parentViewModel.CreateNewEmitterExtParams();
                if (this.emitterExtParamsViewModel == null)
                {
                    return false;
                }

                // エミッタ拡張パラメータを親ノードの追加.
                addResult = this.parentViewModel.AddEmitterExtParams(this.emitterExtParamsViewModel);

                // 作成したエミッタ拡張パラメータを選択状態にする.
                var node = this.emitterExtParamsViewModel as WorkspaceNodeViewModelBase;
                if (node != null)
                {
                    node.IsSelected = true;
                }

                WorkspaceRootViewModel.Instance.UpdateUIStates();
            }

            return addResult;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // エミッタ拡張パラメータノードを選択している場合、
                // 親エミッタノードを選択する.(削除したはずのカスタムアクション
                // ページが操作できてしまうことを防ぐため)
                // この操作は、ノードを削除する前に実行する必要がある.
                if (this.previousSelectedNode != null)
                {
                    var node = this.emitterExtParamsViewModel as WorkspaceNodeViewModelBase;
                    if (node != null && node.IsSelected)
                    {
                        this.previousSelectedNode.IsSelected = true;
                    }
                }

                // ノードの変更を元に戻す.
                bool removeResult = this.parentViewModel.RemoveEmitterExtParams(this.emitterExtParamsViewModel);
                if (removeResult == false)
                {
                    return false;
                }

                WorkspaceRootViewModel.Instance.UpdateUIStates();
            }

            return true;
        }

        /// <summary>
        /// Redo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Redo()
        {
            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // ノードの変更をやり直す.
                bool addResult = this.parentViewModel.AddEmitterExtParams(this.emitterExtParamsViewModel);
                if (addResult == false)
                {
                    return false;
                }

                OptionStore.TriggerOptionChangedEvent(null);
                WorkspaceRootViewModel.Instance.UpdateUIStates();
            }

            return true;
        }
    }
}
