﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.Serializer;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Extensions;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.Properties;
using EffectMaker.UILogic.ViewModels;
using EffectMaker.UILogic.ViewModels.IO;

namespace EffectMaker.BusinessLogic.Commands
{
    /// <summary>
    /// CreateEmitterCommand.
    /// </summary>
    [Alias("CreateEmitter")]
    public class CreateEmitterCommand : CommandBase
    {
        /// <summary>親ビューモデルです.</summary>
        private IEmitterOwnerViewModel parentViewModel = null;

        /// <summary>エミッタビューモデルです.</summary>
        private EmitterViewModel emitterViewModel = null;

        /// <summary>The name for the created emitter.</summary>
        private string emitterName = string.Empty;

        /// <summary>The command stack target.</summary>
        private object commandStackTarget = null;

        /// <summary>以前選択していたノード.</summary>
        private WorkspaceNodeViewModelBase previousSelectedNode = null;

        /// <summary>適用するプリセットのファイルパスとエミッタ名</summary>
        private KeyValuePair<string, string> presetInfo;

        /// <summary>挿入するインデックス</summary>
        private int insertIndex = -1;

        /// <summary>以前選択していたエミッタの処理タイプ</summary>
        private int previousUsingCpu = -1;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public CreateEmitterCommand()
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="parentViewModel">親ビューモデル</param>
        /// <param name="emitterName">エミッタ名</param>
        /// <param name="object">適用するプリセットのファイルパスとエミッタ名</param>
        /// <param name="insertIndex">挿入するインデックス</param>
        public CreateEmitterCommand(HierarchyViewModel parentViewModel, string emitterName, object presetInfo, int insertIndex = -1)
        {
            object[] args = new object[] { parentViewModel, emitterName, presetInfo, insertIndex };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">The arguments for the command.</param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || (args.Length != 3 && args.Length != 4))
            {
                return false;
            }

            // 親ビューモデルを記録
            this.parentViewModel = args[0] as IEmitterOwnerViewModel;
            if (this.parentViewModel == null)
            {
                return false;
            }

            // エミッタ名を記録
            this.emitterName = args[1] as string;
            if (string.IsNullOrEmpty(this.emitterName) == true)
            {
                return false;
            }

            // 適用するプリセットのファイルパスとエミッタ名を取得
            if (args[2] == null)
            {
                this.presetInfo = new KeyValuePair<string, string>();
            }
            else
            {
                this.presetInfo = (KeyValuePair<string, string>)args[2];
            }

            if (args.Length == 4)
            {
                this.insertIndex = (int)args[3];
            }

            // 遡ってCommandStackOwnerAttributeを持つインスタンスをチェックする
            var cmdStackTarget = this.parentViewModel as IHierarchyObject;
            this.commandStackTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(cmdStackTarget) as IHierarchyObject;

            this.previousSelectedNode = this.parentViewModel as WorkspaceNodeViewModelBase;

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            EmitterData presetEmitter = null;

            // Get the emitter set view model.
            var parentEsetViewModel = ViewModelBase.GetParent<EmitterSetViewModel>(
                this.parentViewModel as ViewModelBase);

            using (new MessageBlockerWithSendBinaryOnce(parentEsetViewModel.DataModel))
            {
                if (!string.IsNullOrEmpty(this.presetInfo.Key)
                    && !string.IsNullOrEmpty(this.presetInfo.Value))
                {
                    WorkspaceViewModel workspaceVm = WorkspaceRootViewModel.Instance.WorkspaceViewModel;
                    presetEmitter = workspaceVm.AdaptPresetForEmitter(this.presetInfo.Key, this.presetInfo.Value);
                    if (presetEmitter == null)
                    {
                        return false;
                    }

                    if (this.parentViewModel is EmitterViewModel)
                    {
                        // 子エミッタの削除
                        if (presetEmitter.EmitterList.Any())
                        {
                            foreach (var childEmitter in presetEmitter.EmitterList)
                            {
                                presetEmitter.RemoveChild(childEmitter);
                            }

                            presetEmitter.EmitterList.Clear();

                            Logger.Log(
                                "LogView,Console",
                                LogLevels.Warning,
                                Resources.CreateEmitterCommandDeleteChildrenEmitters);
                        }

                        // 継承設定の初期化
                        using (new EnableDataModelCloneSetter())
                        {
                            presetEmitter.EmitterBasicSettingData.EmitterBasicInheritanceData.SetWithoutGuid(
                                new EmitterBasicInheritanceData());
                        }
                    }
                }

                // エミッタを作成
                this.emitterViewModel = this.parentViewModel.CreateNewEmitter(this.emitterName, presetEmitter);
                if (this.emitterViewModel == null)
                {
                    return false;
                }

                // エミッタを親ノードに追加
                bool addResult = this.parentViewModel.AddEmitter(this.emitterViewModel);
                if (addResult == false)
                {
                    return false;
                }

                // プリセットから作成した時のためにアセットをロケートする
                if (presetEmitter != null)
                {
                    var tmpPath = parentEsetViewModel.FilePath;
                    parentEsetViewModel.FilePath = Path.GetDirectoryName(this.presetInfo.Key);

                    string esetFullPath = null;

                    if (string.IsNullOrEmpty(parentEsetViewModel.FilePath) == false)
                    {
                        esetFullPath = Path.Combine(parentEsetViewModel.FilePath, parentEsetViewModel.FileName + IOConstants.EmitterSetFileExtension);
                    }

                    AssetsManager.CheckAndLoadAssetsFile(
                        this.emitterViewModel,
                        esetFullPath,
                        parentEsetViewModel.FileName);

                    parentEsetViewModel.FilePath = tmpPath;

                    // 初期値から変更されたタブの色付け
                    this.emitterViewModel.ModificationFlagViewModel.UpdateDefaultValues();
                }

                this.SwapLastToIndex();

                // チャイルドエミッタを作成した場合はRaiseCanUsingGpu
                var parentEmitter = this.parentViewModel as EmitterViewModel;
                if (parentEmitter != null)
                {
                    this.previousUsingCpu =
                        parentEmitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu();
                }

                // 編集フラグをクリア
                this.emitterViewModel.ModificationFlagViewModel.ClearChildModificationFlags();
                this.emitterViewModel.ModificationFlagViewModel.ClearModificationFlags();

                // 作成したエミッタを選択状態にする.
                var node = this.emitterViewModel as WorkspaceNodeViewModelBase;
                if (node != null)
                {
                    node.IsSelected = true;
                }

                WorkspaceRootViewModel.Instance.UpdateUIStates();
            }

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            // Get the emitter set view model.
            var parentEsetViewModel = ViewModelBase.GetParent<EmitterSetViewModel>(
                this.parentViewModel as ViewModelBase);

            using (new MessageBlockerWithSendBinaryOnce(parentEsetViewModel.DataModel))
            {
                // エミッタノードを選択している場合、親エミッタノードを選択する.
                // (削除したはずのエミッタページが操作できてしまうことを防ぐため)
                // この操作は、ノードを削除する前に実行する必要がある.
                if (this.previousSelectedNode != null)
                {
                    var node = this.emitterViewModel as WorkspaceNodeViewModelBase;
                    if (node != null && node.IsSelected)
                    {
                        this.previousSelectedNode.IsSelected = true;
                    }
                }

                // ノードの変更を元に戻す.
                bool removeResult = this.parentViewModel.RemoveEmitter(this.emitterViewModel);
                if (removeResult == false)
                {
                    return false;
                }

                // チャイルドエミッタを作成した場合はRaiseCanUsingGpu
                var parentEmitter = this.parentViewModel as EmitterViewModel;
                if (parentEmitter != null)
                {
                    parentEmitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu(
                        this.previousUsingCpu);
                }

                WorkspaceRootViewModel.Instance.UpdateUIStates();
            }

            return true;
        }

        /// <summary>
        /// Redo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Redo()
        {
            // Get the emitter set view model.
            var parentEsetViewModel = ViewModelBase.GetParent<EmitterSetViewModel>(
                this.parentViewModel as ViewModelBase);

            using (new MessageBlockerWithSendBinaryOnce(parentEsetViewModel.DataModel))
            {
                // ノードの変更をやり直す.
                bool addResult = this.parentViewModel.AddEmitter(this.emitterViewModel);
                if (addResult == false)
                {
                    return false;
                }

                this.SwapLastToIndex();

                // チャイルドエミッタを作成した場合はRaiseCanUsingGpu
                var parentEmitter = this.parentViewModel as EmitterViewModel;
                if (parentEmitter != null)
                {
                    this.previousUsingCpu =
                        parentEmitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu();
                }

                OptionStore.TriggerOptionChangedEvent(null);
                WorkspaceRootViewModel.Instance.UpdateUIStates();
            }

            return true;
        }

        /// <summary>
        /// 挿入先が指定されている場合、末尾から挿入先にフィールドを移動します。
        /// </summary>
        private void SwapLastToIndex()
        {
            if (this.insertIndex == -1)
            {
                return;
            }

            var parentOwnerViewModel = (HierarchyViewModel)this.parentViewModel;
            using (new SuppressNodeTypeOrdering(parentOwnerViewModel))
            {
                int sourceIndex = parentOwnerViewModel.Children.IndexOf(this.emitterViewModel);

                parentOwnerViewModel.Children.RemoveAt(sourceIndex);
                parentOwnerViewModel.Children.Insert(this.insertIndex + 1, this.emitterViewModel);

                parentOwnerViewModel.Proxy.MoveChildDataModel(
                    this.emitterViewModel.DataModel,
                    this.insertIndex + 1 - sourceIndex);
            }
        }
    }
}
