﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;
using TexUtils = nw.g3d.iftexutil;

namespace EffectMaker.TextureManager
{
    /// <summary>
    /// TexUtilsプロキシです。
    /// </summary>
    internal class TexUtilsProxy
    {
        /// <summary>
        /// コンバータの実体です。
        /// </summary>
        private static TexUtils.Converter converter = null;

        /// <summary>
        /// ロック用オブジェクトです。
        /// </summary>
        private static readonly object thisLock = new object();

        /// <summary>
        /// 3dTextureConverterTexUtils.dllがあるフォルダのリストです。
        /// </summary>
        private static readonly List<string> basePathes = new List<string>();

        /// <summary>
        /// 初期化処理を行います。
        /// </summary>
        public static void SafeInitialize()
        {
            // ロック前に早期リターンできるならする
            if (converter != null)
            {
                return;
            }

            lock (thisLock)
            {
                if (converter != null)
                {
                    return;
                }

                // コンバータを初期化
                converter = new TexUtils.Converter();
                converter.Initialize(basePathes.ToArray());
            }
        }

        /// <summary>
        /// 破棄処理を行います。
        /// </summary>
        public static void Destroy()
        {
            // ロック前に早期リターンできるならする
            if (converter == null)
            {
                return;
            }

            lock (thisLock)
            {
                if (converter == null)
                {
                    return;
                }

                // コンバータを破棄
                converter.Destroy();
                converter = null;
            }
        }

        /// <summary>
        /// コンバータがターゲットとしているプラットフォームを取得します。
        /// </summary>
        /// <param name="texture">テクスチャタイプ</param>
        /// <returns>プラットフォーム名を返します。</returns>
        public static string GetPlatformName(textureType texture)
        {
            SafeInitialize();

            return converter.GetPlatformName(texture);
        }

        /// <summary>
        /// 3dTextureConverterTexUtils.dllがあるフォルダを追加します。
        /// </summary>
        /// <param name="basePath">3dTextureConverterTexUtils.dllがあるフォルダ</param>
        public static void AddBasePath(string basePath)
        {
            basePathes.Add(basePath);
        }

        /// <summary>
        /// 1D,2Dテクスチャデータを取得します。
        /// </summary>
        /// <param name="texture">テクスチャ</param>
        /// <param name="streams">ストリーム</param>
        /// <returns>テクスチャデータを返します。</returns>
        public static TextureData[] ConvertTo1d2dStream(textureType texture, List<G3dStream> streams)
        {
            SafeInitialize();

            return converter.ConvertTo1d2dStream(texture, streams);
        }

        /// <summary>
        /// 2Dアレイテクスチャデータを取得します。
        /// </summary>
        /// <param name="texture">テクスチャ</param>
        /// <param name="streams">ストリーム</param>
        /// <returns>テクスチャアレイデータを返します。</returns>
        public static TextureData[][] ConvertTo2dArrayStream(textureType texture, List<G3dStream> streams)
        {
            SafeInitialize();

            return converter.ConvertTo2dArrayStream(texture, streams);
        }
    }
}
