﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.EffectCombinerEditor;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.RuntimeOptions;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic;
using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Utility;
using EffectMaker.ShaderCodeGeneratorGeneric;

using CompileShaderAssemblyCallback = System.Action<EffectMaker.DataModelLogic.ShaderCompileResult, string, string, string, string>;
using CompileShaderCompleteCallback = System.Action<System.Collections.Generic.List<long>, System.Action<bool>, EffectMaker.Foundation.Debugging.Profiling.ProfileTimer, EffectMaker.Foundation.Debugging.Profiling.ProfileTimer, bool, bool, byte[], long, int, long, int, System.Action<System.IO.Stream>>;

namespace EffectMaker.SpecGeneric.Shader
{
    /// <summary>
    /// Genericスペック向けのシェーダコンバートを管理するクラスです。
    /// Eft2ベースのGenericとVfxベースのGenericから派生したスペックの両方に対応します。
    /// </summary>
    public partial class ShaderManager : IShaderManager
    {
        /// <summary>The semaphore to prevent multiple threads to access the shader converter together.</summary>
        private static readonly SemaphoreSlim AsyncSemaphore = new SemaphoreSlim(1, 1);

        /// <summary>シェーダコンバートのキュー管理オブジェクト</summary>
        private static readonly AsyncTaskQueueJustOne AsyncAdjuster = new AsyncTaskQueueJustOne();

        /// <summary>
        /// Gfxのシェーダコンバータを使うか否かを取得します。
        /// </summary>
        private static bool UsingGfx
        {
            get { return !string.IsNullOrEmpty(SpecManager.CurrentSpec.ShaderConversionOption.ConverterOption); }
        }

        /// <summary>
        /// ユーザー定義文字列の何番目を利用するかを設定・取得
        /// </summary>
        public uint UserDefineIndex { get; set; }

        /// <summary>
        /// Generate shader source code in the emitter.
        /// </summary>
        /// <param name="emitter">The emitter.</param>
        /// <param name="vertexShader">The generated vertex shader.</param>
        /// <param name="fragmentShader">The generated fragment shader.</param>
        /// <param name="computeShader">The generated compute shader.</param>
        /// <returns>True on success.</returns>
        public bool GenerateShaderSource(EmitterData emitter,
                                         out string vertexShader,
                                         out string fragmentShader,
                                         out string computeShader)
        {
            vertexShader = null;
            fragmentShader = null;
            computeShader = null;

            var context = this.GenerateShaderSourceInternal(emitter,
                                                            OptionStore.RuntimeOptions.ShaderConvertOption);

            vertexShader = context.VertexShaderSource;
            fragmentShader = context.FragmentShaderSource;
            computeShader = context.ComputeShaderSource;

            return context.IsSuccess;
        }

        /// <summary>
        /// Compile shader in the emitter and generate shader assembly.
        /// </summary>
        /// <param name="emitter">The emitter.</param>
        /// <param name="completeCallback">The callback function when completed.</param>
        public async void GenrateShaderAssemblyAsync(EmitterData emitter,
                                                     CompileShaderAssemblyCallback completeCallback)
        {
            if (completeCallback == null)
            {
                return;
            }

            string vertexShaderAssembly = string.Empty;
            string fragmentShaderAssembly = string.Empty;
            string computeShaderAssembly = string.Empty;
            string latencyInfo = string.Empty;
            ShaderConversionInputData conversionInputData;

            bool isSuccess = true;
            List<ShaderCompileErrorInfo> compileErrors = new List<ShaderCompileErrorInfo>();

            if (UsingGfx)
            {
                isSuccess = GenerateGfxShaderCore(
                    emitter,
                    this.UserDefineIndex,
                    false,
                    out vertexShaderAssembly,
                    out fragmentShaderAssembly,
                    out computeShaderAssembly,
                    out latencyInfo,
                    out conversionInputData);
            }
            else
            {
                // EmitterDataからShaderConversionInputDataを用意する
                var conversionInputDataList = PrepareEmitterForShaderConversion(emitter);
                if (conversionInputDataList == null || conversionInputDataList.Count <= 0)
                {
                    completeCallback(ShaderCompileResult.Error, null, null, null, null);
                    return;
                }
                conversionInputData = conversionInputDataList[0];

                // There can only be one thread that uses the shader converter at one time.
                await AsyncSemaphore.WaitAsync();

                var context = await this.AsyncCompileShaderAssemblyWrapper(
                    conversionInputData,
                    OptionStore.RuntimeOptions.ShaderConvertOption);

                AsyncSemaphore.Release();

                ShowShaderCompileError(context.CompileErrors);

                isSuccess = context.IsSuccess;
                compileErrors = context.CompileErrors;
                latencyInfo = context.LatencyInfo;

                // EFT2Genericだとアセンブリは取れないが、検証用にソースをゲット
                vertexShaderAssembly = context.VertexShaderSource;
                fragmentShaderAssembly = context.FragmentShaderSource;
                computeShaderAssembly = context.ComputeShaderSource;
            }

            if (isSuccess == false)
            {
                completeCallback(
                    ShaderCompileResult.Error,
                    vertexShaderAssembly,
                    fragmentShaderAssembly,
                    computeShaderAssembly,
                    latencyInfo);
            }
            else if (compileErrors.Count > 0)
            {
                completeCallback(
                    ShaderCompileResult.Warning,
                    vertexShaderAssembly,
                    fragmentShaderAssembly,
                    computeShaderAssembly,
                    latencyInfo);
            }
            else
            {
                completeCallback(
                    ShaderCompileResult.Success,
                    vertexShaderAssembly,
                    fragmentShaderAssembly,
                    computeShaderAssembly,
                    latencyInfo);
            }
        }

        /// <summary>
        /// Write shader binary data.
        /// </summary>
        /// <param name="binaryName">バイナリー名</param>
        /// <param name="stream">バイナリーストリーム</param>
        /// <param name="enableAsync">True to compile shader asynchronously.</param>
        /// <param name="completeCallback">The callback to execute when finished.</param>
        /// <param name="emitterContexts">エミッタのバイナリーコンテキスト</param>
        /// <param name="handleShaderCompileComplete"></param>
        /// <returns>True on success.</returns>
        public bool WriteShaderResourcesImpl(string binaryName,
                                             Stream stream,
                                             string shaderBinaryOutputFilePath,
                                             string computeShaderBinaryOutputFilePath,
                                             bool enableAsync,
                                             Action<bool> completeCallback,
                                             IEnumerable<WriteBinaryDataContext> emitterContexts,
                                             CompileShaderCompleteCallback handleShaderCompileComplete)
        {
            // GFX準拠ではないランタイムなら空バイナリを付ける
            if (!UsingGfx)
            {
                // 空のシェーダーアレイバイナリを付ける
                var shda = new BinaryStructHeader { Tag = "SHDA" };
                shda.Write(stream);

                // バイナリのケツをシメる
                completeCallback(true);

                return true;
            }

            if (enableAsync)
            {
                AsyncAdjuster.RunOrQueue(
                    binaryName,
                    () => WriteGfxShaderResourcesCore(
                        stream,
                        shaderBinaryOutputFilePath,
                        computeShaderBinaryOutputFilePath,
                        completeCallback,
                        emitterContexts,
                        this.UserDefineIndex,
                        handleShaderCompileComplete),
                    () => handleShaderCompileComplete(
                        null,
                        completeCallback,
                        null,
                        null,
                        false, // isSuccessful
                        false, // shaderCompileFailed
                        null,  // shaderBinary
                        0,     // shaderBinarySize
                        1,     // shaderCount
                        0,
                        0,
                        null)); // writeShaderIndicesAction

                return true;
            }

            return WriteGfxShaderResourcesCore(
                stream,
                shaderBinaryOutputFilePath,
                computeShaderBinaryOutputFilePath,
                completeCallback,
                emitterContexts,
                this.UserDefineIndex,
                handleShaderCompileComplete);
        }

        /// <summary>
        /// The internal method for generating shader source code.
        /// </summary>
        /// <param name="emitter">The input.</param>
        /// <param name="option">The option for the shader conversion.</param>
        /// <returns>The result of the source code generation.</returns>
        private ShaderCompileContext GenerateShaderSourceInternal(
            EmitterData emitter,
            ShaderConvertOption option)
        {
            bool result;
            string vertexShader;
            string fragmentShader;
            string computeShader;
            string latencyInfo = string.Empty;
            ShaderConversionInputData conversionInputData;

            if (UsingGfx)
            {
                result = GenerateGfxShaderCore(
                    emitter,
                    this.UserDefineIndex,
                    true,
                    out vertexShader,
                    out fragmentShader,
                    out computeShader,
                    out latencyInfo,
                    out conversionInputData);
            }
            else
            {
                // EmitterDataからShaderConversionInputDataを用意する
                var conversionInputDataList = PrepareEmitterForShaderConversion(emitter);
                if (conversionInputDataList == null || conversionInputDataList.Count <= 0)
                {
                    return null;
                }
                conversionInputData = conversionInputDataList[0];

                using (var converter = new ShaderCodeGeneratorGeneric.ShaderCodeGeneratorGeneric(ShaderBinaryHelper.Handle))
                {
                    // Set shader source code to the shader converter.
                    converter.SetShaderCodes(CopyShaderCodeForShaderConversion(false));
                    conversionInputData.UserDefineIndex = this.UserDefineIndex;
                    result = converter.GenerateShaderCode(
                        conversionInputData,
                        out vertexShader,
                        out fragmentShader,
                        out computeShader);
                }
            }

            return new ShaderCompileContext(
                result,
                new List<ShaderCompileErrorInfo>(),
                vertexShader,
                fragmentShader,
                computeShader,
                latencyInfo,
                conversionInputData);
        }

        /// <summary>
        /// Wraps CompileShaderAssemblyInternal() to make it run on another thread. (asychronously)
        /// </summary>
        /// <param name="conversionInputData">The input data for the conversion.</param>
        /// <param name="option">The option for the shader conversion.</param>
        /// <returns>The compile result.</returns>
        private Task<ShaderCompileContext> AsyncCompileShaderAssemblyWrapper(
            ShaderConversionInputData conversionInputData,
            ShaderConvertOption option)
        {
            return Task.Run(() => this.CompileShaderAssemblyInternal(conversionInputData, option));
        }

        /// <summary>
        /// The internal method for compiling shader assembly.
        /// </summary>
        /// <param name="conversionInputData">The input data for the conversion.</param>
        /// <param name="option">The option for the shader conversion.</param>
        /// <returns>The compile result.</returns>
        private ShaderCompileContext CompileShaderAssemblyInternal(
            ShaderConversionInputData conversionInputData,
            ShaderConvertOption option)
        {
            bool result;
            string vertexShader;
            string fragmentShader;
            string computeShader;
            List<ShaderCompileErrorInfo> errorList;

            using (var converter = new ShaderCodeGeneratorGeneric.ShaderCodeGeneratorGeneric(ShaderBinaryHelper.Handle))
            {
                // Set shader source code to the shader converter.
                converter.SetShaderCodes(CopyShaderCodeForShaderConversion(false));
                conversionInputData.UserDefineIndex = this.UserDefineIndex;
                result = converter.GenerateShaderAssembly(
                    conversionInputData,
                    out vertexShader,
                    out fragmentShader,
                    out computeShader,
                    out errorList);
            }

            return new ShaderCompileContext(
                result,
                errorList,
                vertexShader,
                fragmentShader,
                computeShader,
                null,
                conversionInputData);
        }

        /// <summary>
        /// Copy shader source code for shader conversion.
        /// </summary>
        /// <param name="usingGfx">Gfxを使う場合はtrue,そうでない場合はfalse.</param>
        /// <returns>The shader code for conversion.</returns>
        private static ShaderCodeListCs CopyShaderCodeForShaderConversion(bool usingGfx)
        {
            var shaderCodes = new ShaderCodeListCs();

            if (!usingGfx)
            {
                // Eft2の時はソースコードをマネージャから取得してガッチャンコ
                shaderCodes.SpecDecSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                    ShaderTypes.SpecDeclarationTargetShader);

                shaderCodes.ParticleDecVshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                    ShaderTypes.ParticleDeclarationVertexShader);

                shaderCodes.ParticleDecFshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                    ShaderTypes.ParticleDeclarationFragmentShader);

                shaderCodes.StreamOutDecVshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                    ShaderTypes.StreamOutDeclarationVertexShader);

                shaderCodes.ParticleGlslSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                    ShaderTypes.ParticleDeclarationShader);

                shaderCodes.VertexShaderSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                    ShaderTypes.ParticleVertexShader);

                shaderCodes.FragShaderSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                    ShaderTypes.ParticleFragmentShader);

                shaderCodes.StreamOutVshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                    ShaderTypes.StreamOutVertexShader);
            }
            else
            {
                // Vfxの時はインクルードで解決
                shaderCodes.SpecDecSrc =
                    "#include \"" + SpecManager.CurrentSpec.ShaderConversionOption.SpecDeclarationTarget + "\"" + Environment.NewLine;
                shaderCodes.ParticleGlslSrc = "#include \"eft_ParticleDeclaration.glsl\"" + Environment.NewLine;
                shaderCodes.ParticleDecVshSrc = "#include \"eft_ParticleDeclaration.vsh\"" + Environment.NewLine;
                shaderCodes.ParticleDecFshSrc = "#include \"eft_ParticleDeclaration.fsh\"" + Environment.NewLine;
                shaderCodes.VertexShaderSrc = "#include \"eft_Particle.vsh\"" + Environment.NewLine;
                shaderCodes.FragShaderSrc = "#include \"eft_Particle.fsh\"" + Environment.NewLine;
                shaderCodes.StreamOutDecVshSrc = "#include \"eft_StreamOutDeclaration.vsh\"" + Environment.NewLine;
                shaderCodes.StreamOutVshSrc = "#include \"eft_StreamOut.vsh\"" + Environment.NewLine;
            }

            shaderCodes.GeneralVshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.GeneralVertexShader);

            shaderCodes.GeneralFshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.GeneralFragmentShader);

            shaderCodes.GeneralCshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.GeneralComputeShader);

            shaderCodes.CustomVshSrc = CustomShaderUserDataManager.GetCustomShaderSourceCodeArray(
                ShaderTypes.CustomVertexShader);

            shaderCodes.CustomFshSrc = CustomShaderUserDataManager.GetCustomShaderSourceCodeArray(
                ShaderTypes.CustomFragmentShader);

            shaderCodes.ReservedVshSrc = CustomShaderUserDataManager.GetReservedShaderSourceCodeArray(
                ShaderTypes.ReservedVertexShader);

            shaderCodes.ReservedFshSrc = CustomShaderUserDataManager.GetReservedShaderSourceCodeArray(
                ShaderTypes.ReservedFragmentShader);

            return shaderCodes;
        }

        /// <summary>
        /// Display error log and shader code when shader compile failed.
        /// </summary>
        /// <param name="errorList">The error information list.</param>
        private static void ShowShaderCompileError(List<ShaderCompileErrorInfo> errorList)
        {
            if (errorList.Count <= 0)
            {
                return;
            }

            // 30個以下になるように調整
            PathUtility.DeleteSubDirectoriesOverNum(IOConstants.ShaderErrorLogFolderPath, 30);

            // Save files in [Executable Folder]/Converter/shader_error/PID
            string outputFolderPath = Path.Combine(
                IOConstants.ShaderErrorLogFolderPath, Process.GetCurrentProcess().Id.ToString());

            // 空のフォルダを作成
            if (IOUtility.SafeCreateEmptyDirectory(outputFolderPath) == false)
            {
                return;
            }

            var filePathsForTextEditor = new List<string>();

            // Save error log and shader source code to file.
            int index = 0;
            foreach (ShaderCompileErrorInfo error in errorList)
            {
                string errorLog = error.EmitterName + Environment.NewLine + error.ErrorLog;

                int retryCount = 0;
                bool retry = false;
                do
                {
                    string filePath = Path.Combine(
                        outputFolderPath,
                        string.Format("{0}_log{1}", error.EmitterName, index));

                    retry = false;
                    try
                    {
                        File.WriteAllText(filePath + ".log", errorLog, Encoding.UTF8);
                        File.WriteAllText(filePath + ".vsh", error.VertexShader, Encoding.UTF8);
                        File.WriteAllText(filePath + ".fsh", error.FragmentShader, Encoding.UTF8);
                    }
                    catch
                    {
                        // Failed to save the files as the file name, try
                        retry = true;
                        ++retryCount;
                        ++index;
                    }

                    // Do not show the files in the text editor if the message is just an warning.
                    if (error.IsWarning == false &&
                        retry == false)
                    {
                        // Remember the error log file path, so we can open it
                        // with external text editor later.
                        filePathsForTextEditor.Add(filePath + ".log");
                    }
                }
                while (retry == true && retryCount < 10);

                ++index;
            }

            if (OptionStore.RuntimeOptions.IsCommandLineMode == false)
            {
                // Open the error log files with external text editor.
                foreach (string filePath in filePathsForTextEditor)
                {
                    ApplicationIOManager.OpenTextFileWithExternalTextEditor(filePath);
                }
            }
        }

        /// <summary>
        /// Prepare emitter binary data for shader conversion.
        /// </summary>
        /// <param name="emitter">The emitter data.</param>
        /// <returns>The data for shader conversion.</returns>
        private static List<ShaderConversionInputData> PrepareEmitterForShaderConversion(EmitterData emitter)
        {
            using (var stream = new MemoryStream())
            {
                // Find the binary data for the data model.
                DataModelBase dataModel = emitter;

                var binaryData = BinaryConversionInfoManager.FindBinaryData(dataModel);

                while (binaryData == null && dataModel.Parent != null)
                {
                    dataModel = dataModel.Parent;
                    binaryData = BinaryConversionInfoManager.FindBinaryData(dataModel);
                }

                if (binaryData == null)
                {
                    return null;
                }

                Stream childStream;
                IEnumerable<WriteBinaryDataContext> emitterContexts;
                if (!WriteBinaryDataSession.WriteBinaryForPrepareEmitter(
                    emitter,
                    stream,
                    binaryData,
                    out childStream,
                    out emitterContexts))
                {
                    return null;
                }

                // Make input data list for shader conversion.
                var conversionInputDataList =
                    MakeShaderConversionInputDataList(childStream, emitterContexts);

                // Do we have any emitter?
                if (conversionInputDataList.Count <= 0)
                {
                    return null;
                }

                return conversionInputDataList;
            }
        }

        /// <summary>
        /// Make a list of input data for shader conversion.
        /// </summary>
        /// <param name="stream">セッションのストリーム</param>
        /// <param name="emitterContexts">WriteBinaryDataContexts of emitters.</param>
        /// <returns>The list of input data for shader conversion.</returns>
        private static List<ShaderConversionInputData> MakeShaderConversionInputDataList(
            Stream stream,
            IEnumerable<WriteBinaryDataContext> emitterContexts)
        {
            uint binHeaderSize = BinaryStructHeader.Size;

            var conversionInputDataList = new List<ShaderConversionInputData>();

            // Prepare the information needed for shader binary conversion.
            foreach (WriteBinaryDataContext context in emitterContexts)
            {
                var tmpHeader = new byte[binHeaderSize];
                var tmpBuffer = new byte[context.WriteSize - binHeaderSize];

                stream.Seek(context.WritePosition, SeekOrigin.Begin);
                stream.Read(tmpHeader, 0, tmpHeader.Length);

                stream.Seek(context.WritePosition + (long)binHeaderSize, SeekOrigin.Begin);
                stream.Read(tmpBuffer, 0, tmpBuffer.Length);

                // Get the effect combiner project path set in the emitter data.
                var emitter = (EmitterData)context.BinaryStruct.DataModel;

                // Get the ID of the reserved shader the emitter uses.
                int reservedShaderIndex = -1;
                if (emitter.ActiveReservedShader != null &&
                    emitter.ActiveReservedShader.PageData != null &&
                    emitter.ActiveReservedShader.PageData.ContentsData != null)
                {
                    reservedShaderIndex = ReservedShaderUserDataManager.FindReservedShaderIndexByUserData(
                        emitter.ActiveReservedShader.PageData.ContentsData);
                }

                // Get the effect combiner project path set in the emitter data.
                string combinerShaderCode = string.Empty;
                string combinerProjectPath = emitter.EmitterCombinerData.EmitterCombinerEditorData.CombinerEditorProjectPath;

                if (OptionStore.ProjectConfig.IsEftCombinerEditorEnabled == true &&
                    string.IsNullOrEmpty(combinerProjectPath) == false)
                {
                    var combiner = EffectCombinerCommunicationManager.CommunicationBridge;
                    if (combiner != null)
                    {
                        combinerShaderCode = combiner.GetShaderSourceCode(combinerProjectPath);
                    }
                }

                // Create the shader conversion input data.
                var data = new ShaderConversionInputData()
                {
                    BinaryHeader = tmpHeader,
                    EmitterBinary = tmpBuffer,
                    OverrideShaderCode = combinerShaderCode,
                    ReservedShaderIndex = reservedShaderIndex,
                };

                conversionInputDataList.Add(data);
            }

            return conversionInputDataList;
        }

        /// <summary>
        /// Holds result and data generated when the shader is compiled.
        /// </summary>
        private class ShaderCompileContext
        {
            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="isSuccess">The result of the shader compilation.</param>
            /// <param name="errors">The error messages returned from the compiler.</param>
            /// <param name="shaderBinary">The compiled shader binary.</param>
            /// <param name="shaderCount">The number of shaders being compiled.</param>
            /// <param name="inputDataList">The shader conversion input data list.</param>
            public ShaderCompileContext(
                bool isSuccess,
                List<ShaderCompileErrorInfo> errors,
                byte[] shaderBinary,
                int shaderCount,
                List<ShaderConversionInputData> inputDataList)
            {
                this.IsSuccess = isSuccess;
                this.CompileErrors = errors;
                this.ShaderBinary = shaderBinary;
                this.ShaderCount = shaderCount;
                this.InputDataList = inputDataList;
                this.VertexShaderSource = string.Empty;
                this.FragmentShaderSource = string.Empty;
                this.LatencyInfo = string.Empty;
            }

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="isSuccess">The result of the shader compilation.</param>
            /// <param name="errors">The error messages returned from the compiler.</param>
            /// <param name="vertexSource">The source code of the vertex shader.</param>
            /// <param name="fragmentSource">The source code of the fragment shader.</param>
            /// <param name="fragmentSource">The source code of the compute shader.</param>
            /// <param name="latencyInfo">シェーダの負荷情報</param>
            /// <param name="inputData">The shader conversion input data.</param>
            public ShaderCompileContext(
                bool isSuccess,
                List<ShaderCompileErrorInfo> errors,
                string vertexSource,
                string fragmentSource,
                string computeShader,
                string latencyInfo,
                ShaderConversionInputData inputData)
            {
                this.IsSuccess = isSuccess;
                this.CompileErrors = errors;
                this.ShaderBinary = null;
                this.ShaderCount = 0;
                this.InputDataList = new List<ShaderConversionInputData>() { inputData };
                this.VertexShaderSource = vertexSource;
                this.FragmentShaderSource = fragmentSource;
                this.ComputeShaderSource = computeShader;
                this.LatencyInfo = latencyInfo;
            }

            /// <summary>
            /// Get the flag indicating whether the shader being compiled successfully.
            /// </summary>
            public bool IsSuccess { get; private set; }

            /// <summary>
            /// Get the error messages returned from the shader compiler.
            /// </summary>
            public List<ShaderCompileErrorInfo> CompileErrors { get; private set; }

            /// <summary>
            /// Get the generated shader binary.
            /// </summary>
            public byte[] ShaderBinary { get; private set; }

            /// <summary>
            /// Get the number of shaders being compiled.
            /// </summary>
            public int ShaderCount { get; private set; }

            /// <summary>
            /// Get the shader conversion input data list.
            /// </summary>
            public List<ShaderConversionInputData> InputDataList { get; private set; }

            /// <summary>
            /// Get the source code of the vertex shader.
            /// </summary>
            public string VertexShaderSource { get; private set; }

            /// <summary>
            /// Get the source code of the fragment shader.
            /// </summary>
            public string FragmentShaderSource { get; private set; }

            /// <summary>
            /// Get the source code of the compute shader.
            /// </summary>
            public string ComputeShaderSource { get; private set; }

            /// <summary>
            /// シェーダの負荷情報を取得します。
            /// </summary>
            public string LatencyInfo { get; private set; }
        }
    }
}
