﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Text;
using EffectMaker.BusinessLogic.GfxToolsUtility;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.BusinessLogic.Synchronization;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.SpecGeneric.Shader
{
    #if false
    // --------------------------------------------------------------------------------
    // DLL 版の ShaderConverterExecuter です。
    //
    // 並列実行したとき、アプリ終了時に ShaderCompiler.dll_unloaded のエラーが
    // 発生するため、無効にしています。
    //
    // 関連タスク: SIGLO-60671
    // --------------------------------------------------------------------------------

    /// <summary>
    /// シェーダコンバータの実行管理クラスです。
    /// </summary>
    public class ShaderConverterExecuter
    {
        #region Singleton Members

        /// <summary>
        /// シングルトンインスタンスです。
        /// </summary>
        private static ShaderConverterExecuter singletonInstance;

        /// <summary>
        /// シングルトンインスタンスを取得します。
        /// </summary>
        public static ShaderConverterExecuter Instance
        {
            get
            {
                if (singletonInstance == null)
                {
                    singletonInstance = new ShaderConverterExecuter();
                }

                return singletonInstance;
            }
        }

        #endregion

        /// <summary>
        /// シェーダコンバータの実行インスタンスです。
        /// </summary>
        private ShaderConverterWrapper shaderConverter;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderConverterExecuter()
        {
            string processId = Process.GetCurrentProcess().Id.ToString();

            string globalIntermediateDirectoryPath = Path.Combine(IOConstants.ExecutableFolderPath, @"Converter\shader_intermediate");
            string localIntermediateDirectoryPath = Path.Combine(globalIntermediateDirectoryPath, processId);

            string globalErrorLogDirectoryPath = Path.Combine(IOConstants.ExecutableFolderPath, @"Converter\shader_error");
            string localErrorLogDirectoryPath = Path.Combine(globalErrorLogDirectoryPath, processId);

            // 実行に必要なフォルダを作成
            using (new GlobalSyncSection())
            {
                // 古いデータを削除
                IOUtility.DeleteExpiredFileSystemEntries(globalIntermediateDirectoryPath, TimeSpan.FromDays(3.0));
                IOUtility.DeleteExpiredFileSystemEntries(globalErrorLogDirectoryPath, TimeSpan.FromDays(3.0));

                // 中間生成物配置フォルダを作成
                IOUtility.SafeCreateTemporaryDirectory(globalIntermediateDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForGlobal);
                IOUtility.SafeCreateTemporaryDirectory(localIntermediateDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForMyProcess);

                // エラーログ配置フォルダを作成
                // エラーログファイルは確認用に残しておくためForGlobalで作成する
                IOUtility.SafeCreateTemporaryDirectory(globalErrorLogDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForGlobal);
                IOUtility.SafeCreateTemporaryDirectory(localErrorLogDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForGlobal);
            }

            this.DllFilePath = Path.Combine(IOConstants.ToolsDirectoryPath, @"Graphics\GraphicsTools\ShaderConverter.dll");

            this.IntermediateDirectoryPath = localIntermediateDirectoryPath;
            this.ErrorLogDirecotoryPath = localErrorLogDirectoryPath;

            this.shaderConverter = new ShaderConverterWrapper();
            this.shaderConverter.Initialize(this.DllFilePath);
        }

        /// <summary>
        /// シェーダコンバータのDLLファイルパスを取得します。
        /// </summary>
        public string DllFilePath { get; private set; }

        /// <summary>
        /// シェーダコンバータの中間生成物配置フォルダパスを取得します。
        /// </summary>
        public string IntermediateDirectoryPath { get; private set; }

        /// <summary>
        /// シェーダコンバータのエラーログ配置フォルダパスを取得します。
        /// </summary>
        public string ErrorLogDirecotoryPath { get; private set; }

        /// <summary>
        /// ShaderConverterのDLLを実行する関数です。
        /// </summary>
        /// <param name="args">ツールに付けるオプション</param>
        /// <param name="standardOutput">ツールの標準出力</param>
        /// <returns>コンバート結果を返します。</returns>
        public ConvertResult Execute(string args)
        {
            // シェーダがプリコンパイル可能かチェック
            if (SpecManager.CurrentSpec.ShaderConversionOption.TargetIsPreCompile == false)
            {
                ConvertResult res = new ConvertResult()
                {
                    ResultCode = -1
                };

                return res;
            }

            // ツールの作業フォルダがあるかチェック
            {
                string[] workingDirectories = new string[]
                {
                    this.IntermediateDirectoryPath,
                    this.ErrorLogDirecotoryPath
                };

                foreach (string path in workingDirectories)
                {
                    if (Directory.Exists(path) == false)
                    {
                        ConvertResult res = new ConvertResult()
                        {
                            ResultCode = -1,
                            ErrorLog   = string.Format("Temporary folder not found at {0}.", path)
                        };

                        return res;
                    }
                }
            }

            // DLLファイルがあるかチェック
            if (File.Exists(this.DllFilePath) == false)
            {
                ConvertResult res = new ConvertResult()
                {
                    ResultCode = -1,
                    ErrorLog   = string.Format("DLL file not found at {0}.", this.DllFilePath)
                };

                return res;
            }

            // コンバートを実行
            var resExecute = this.shaderConverter.Execute(args);

            ConvertResult result = new ConvertResult()
            {
                ResultCode     = resExecute.ResultCode,
                InformationLog = resExecute.DefaultLog,
                WarningLog     = resExecute.WarningLog,
                ErrorLog       = resExecute.ErrorLog,
            };

            Logger.Log(LogLevels.Information, "{0} {1}", Path.GetFileName(this.DllFilePath), args);

            // コンバートに成功したときも警告を表示する
            if (resExecute.ResultCode == 0 && string.IsNullOrEmpty(resExecute.WarningLog) == false)
            {
                Logger.Log("Console,LogView", LogLevels.Warning, "ShaderCovnerter Warnings:");
                Logger.Log("Console,LogView", LogLevels.Warning, resExecute.WarningLog.Trim());
            }

            return result;
        }

        /// <summary>
        /// コンバート結果です。
        /// </summary>
        public class ConvertResult
        {
            /// <summary>
            /// リザルトコードを取得または設定します。
            /// </summary>
            public int ResultCode { get; set; }

            /// <summary>
            /// 情報ログの内容を取得または設定します。
            /// </summary>
            public string InformationLog { get; set; }

            /// <summary>
            /// 警告ログの内容を取得または設定します。
            /// </summary>
            public string WarningLog { get; set; }

            /// <summary>
            /// エラーログの内容を取得または設定します。
            /// </summary>
            public string ErrorLog { get; set; }

            /// <summary>
            /// 全てのログをまとめた内容を取得します。
            /// </summary>
            /// <returns>全てのログをまとめた内容</returns>
            public string GetAllLog()
            {
                StringBuilder log = new StringBuilder();

                log.AppendLine(" -- Information Log --");
                log.AppendLine(this.InformationLog);
                log.AppendLine();

                log.AppendLine(" -- Warning Log --");
                log.AppendLine(this.WarningLog);
                log.AppendLine();

                log.AppendLine(" -- Error Log --");
                log.AppendLine(this.ErrorLog);
                log.AppendLine();

                return log.ToString();
            }
        }
    }

    #else
    // --------------------------------------------------------------------------------
    // EXE 版の ShaderConverterExecuter です。
    // --------------------------------------------------------------------------------

    /// <summary>
    /// シェーダコンバータの実行管理クラスです。
    /// </summary>
    public class ShaderConverterExecuter
    {
        #region Singleton Members

        /// <summary>
        /// シングルトンインスタンスです。
        /// </summary>
        private static ShaderConverterExecuter singletonInstance;

        /// <summary>
        /// シングルトンインスタンスを取得します。
        /// </summary>
        public static ShaderConverterExecuter Instance
        {
            get
            {
                if (singletonInstance == null)
                {
                    singletonInstance = new ShaderConverterExecuter();
                }

                return singletonInstance;
            }
        }

        #endregion

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderConverterExecuter()
        {
            string processId = Process.GetCurrentProcess().Id.ToString();

            string globalIntermediateDirectoryPath = Path.Combine(IOConstants.ExecutableFolderPath, @"Converter\shader_intermediate");
            string localIntermediateDirectoryPath = Path.Combine(globalIntermediateDirectoryPath, processId);

            string globalErrorLogDirectoryPath = Path.Combine(IOConstants.ExecutableFolderPath, @"Converter\shader_error");
            string localErrorLogDirectoryPath = Path.Combine(globalErrorLogDirectoryPath, processId);

            // 実行に必要なフォルダを作成
            using (new GlobalSyncSection())
            {
                // 古いデータを削除
                IOUtility.DeleteExpiredFileSystemEntries(globalIntermediateDirectoryPath, TimeSpan.FromDays(3.0));
                IOUtility.DeleteExpiredFileSystemEntries(globalErrorLogDirectoryPath, TimeSpan.FromDays(3.0));

                // 中間生成物配置フォルダを作成
                IOUtility.SafeCreateTemporaryDirectory(globalIntermediateDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForGlobal);
                IOUtility.SafeCreateTemporaryDirectory(localIntermediateDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForMyProcess);

                // エラーログ配置フォルダを作成
                // エラーログファイルは確認用に残しておくためForGlobalで作成する
                IOUtility.SafeCreateTemporaryDirectory(globalErrorLogDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForGlobal);
                IOUtility.SafeCreateTemporaryDirectory(localErrorLogDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForGlobal);
            }

            this.ExecutableFilePath = Path.Combine(IOConstants.ToolsDirectoryPath, @"Graphics\GraphicsTools\ShaderConverter.exe");

            this.IntermediateDirectoryPath = localIntermediateDirectoryPath;
            this.ErrorLogDirecotoryPath = localErrorLogDirectoryPath;
        }

        /// <summary>
        /// シェーダコンバータのEXEファイルパスを取得します。
        /// </summary>
        public string ExecutableFilePath { get; private set; }

        /// <summary>
        /// シェーダコンバータの中間生成物配置フォルダパスを取得します。
        /// </summary>
        public string IntermediateDirectoryPath { get; private set; }

        /// <summary>
        /// シェーダコンバータのエラーログ配置フォルダパスを取得します。
        /// </summary>
        public string ErrorLogDirecotoryPath { get; private set; }

        /// <summary>
        /// ShaderConverterのDLLを実行する関数です。
        /// </summary>
        /// <param name="args">ツールに付けるオプション</param>
        /// <param name="standardOutput">ツールの標準出力</param>
        /// <returns>コンバート結果を返します。</returns>
        public ConvertResult Execute(string args)
        {
            // シェーダがプリコンパイル可能かチェック
            if (SpecManager.CurrentSpec.ShaderConversionOption.TargetIsPreCompile == false)
            {
                ConvertResult res = new ConvertResult()
                {
                    ResultCode = -1
                };

                return res;
            }

            // ツールの作業フォルダがあるかチェック
            {
                string[] workingDirectories = new string[]
                {
                    this.IntermediateDirectoryPath,
                    this.ErrorLogDirecotoryPath
                };

                foreach (string path in workingDirectories)
                {
                    if (Directory.Exists(path) == false)
                    {
                        ConvertResult res = new ConvertResult()
                        {
                            ResultCode = -1,
                            ErrorLog   = string.Format("Temporary folder not found at {0}.", path)
                        };

                        return res;
                    }
                }
            }

            // 実行ファイルがあるかチェック
            if (File.Exists(this.ExecutableFilePath) == false)
            {
                ConvertResult res = new ConvertResult()
                {
                    ResultCode = -1,
                    ErrorLog   = string.Format("Executable file not found at {0}.", this.ExecutableFilePath)
                };

                return res;
            }

            string workingDirectoryPath = Path.GetDirectoryName(this.ExecutableFilePath);

            // コンバートを実行
            var resExecute = GfxToolExecutionUtility.ExecuteExeFile(this.ExecutableFilePath, workingDirectoryPath, args);

            StringBuilder warnings = new StringBuilder();

            // 標準出力のテキストから警告を抽出
            {
                string[] lines = resExecute.StandardOutputLog.Split(new char[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries);

                foreach (string line in lines)
                {
                    if (line.StartsWith("Warning "))
                    {
                        warnings.AppendLine(line);
                    }
                }
            }

            ConvertResult result = new ConvertResult()
            {
                ResultCode     = resExecute.ResultCode,
                InformationLog = resExecute.StandardOutputLog,
                WarningLog     = warnings.ToString(),
                ErrorLog       = resExecute.StandardErrorLog,
            };

            Logger.Log(LogLevels.Information, "{0} {1}", Path.GetFileName(this.ExecutableFilePath), args);

            // コンバートに成功したときも警告を表示する
            if (resExecute.ResultCode == 0 && string.IsNullOrEmpty(result.WarningLog) == false)
            {
                Logger.Log("Console,LogView", LogLevels.Warning, "ShaderCovnerter Warnings:");
                Logger.Log("Console,LogView", LogLevels.Warning, result.WarningLog.Trim());
            }

            return result;
        }

        /// <summary>
        /// コンバート結果です。
        /// </summary>
        public class ConvertResult
        {
            /// <summary>
            /// リザルトコードを取得または設定します。
            /// </summary>
            public int ResultCode { get; set; }

            /// <summary>
            /// 情報ログの内容を取得または設定します。
            /// </summary>
            public string InformationLog { get; set; }

            /// <summary>
            /// 警告ログの内容を取得または設定します。
            /// </summary>
            public string WarningLog { get; set; }

            /// <summary>
            /// エラーログの内容を取得または設定します。
            /// </summary>
            public string ErrorLog { get; set; }

            /// <summary>
            /// 全てのログをまとめた内容を取得します。
            /// </summary>
            /// <returns>全てのログをまとめた内容</returns>
            public string GetAllLog()
            {
                StringBuilder log = new StringBuilder();

                log.AppendLine(" -- Information Log --");
                log.AppendLine(this.InformationLog);
                log.AppendLine();

                log.AppendLine(" -- Warning Log --");
                log.AppendLine(this.WarningLog);
                log.AppendLine();

                log.AppendLine(" -- Error Log --");
                log.AppendLine(this.ErrorLog);
                log.AppendLine();

                return log.ToString();
            }
        }
    }

    #endif
}
