﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Runtime.Remoting.Channels;
using System.Runtime.Remoting.Channels.Ipc;

namespace EffectMaker.ProcessBridge
{
    /// <summary>
    /// EffectMakerBridge通信クライアントです。
    /// </summary>
    public class BridgeClient
    {
        /// <summary>
        /// ファイナライザです。
        /// </summary>
        ~BridgeClient()
        {
            Debug.Assert(this.clientChannel == null, "未破棄");
        }

        /// <summary>
        /// 通信クライアントを初期化します。
        /// </summary>
        /// <returns>初期化に成功したときtrue、それ以外はfalseを返します。</returns>
        public bool Initialize()
        {
            Debug.Assert(this.clientChannel == null, "二重初期化");

            // クライアントチャンネルを作成
            this.clientChannel = new IpcClientChannel();
            ChannelServices.RegisterChannel(this.clientChannel, true);

            // リモートオブジェクトを取得
            string url = string.Format("ipc://{0}/{1}", BridgeRemoteObject.IpcChannelName, BridgeRemoteObject.IpcUrl);
            this.remoteObject = Activator.GetObject(typeof(BridgeRemoteObject), url) as BridgeRemoteObject;

            // 接続チェック
            try
            {
                var commandType = this.remoteObject.CommandType;
            }
            catch (Exception)
            {
                this.Release();
                return false;
            }

            return true;
        }

        /// <summary>
        /// 通信クライアントを破棄します。
        /// </summary>
        public void Release()
        {
            if (this.remoteObject != null)
            {
                this.remoteObject = null;
            }

            if (this.clientChannel != null)
            {
                ChannelServices.UnregisterChannel(this.clientChannel);
                this.clientChannel = null;
            }
        }

        /// <summary>
        /// ファイルを読み込み処理を呼び出します。
        /// </summary>
        /// <param name="paths">ファイルパスリスト</param>
        /// <returns>処理が成功したときtrue、それ以外はfalseを返します。</returns>
        public bool CallLoadFiles(string[] paths)
        {
            Debug.Assert(clientChannel != null, "未初期化");

            try
            {
                remoteObject.Paths = new List<string>(paths);
                remoteObject.CommandType = BridgeCommandType.LoadFiles;

                remoteObject.TriggerProcess();
            }
            catch (Exception)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// クライアントチャンネル
        /// </summary>
        private IpcClientChannel clientChannel;

        /// <summary>
        /// リモートオブジェクト
        /// </summary>
        private BridgeRemoteObject remoteObject;
    }
}
