﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Generic;
using System.Linq;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelLogic.Utilities
{
    /// <summary>
    /// バイナリコンバートセッションのプロファイラです。
    /// </summary>
    public class WriteBinaryDataSessionProfiler
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public WriteBinaryDataSessionProfiler()
        {
        }

        /// <summary>
        /// エミッタセットとエミッタのプロファイルの有効/無効を取得または設定します。
        /// </summary>
        public bool EsetProfileEnabled { get; set; }

        /// <summary>
        /// バイナリサイズを取得します。
        /// </summary>
        public long BinarySize { get; private set; }

        /// <summary>
        /// エミッタセットバイナリの合計サイズを取得します。
        /// </summary>
        public long EsetBinarySize { get; private set; }

        /// <summary>
        /// テクスチャのバイナリブロック情報を取得または設定します。
        /// </summary>
        public BinaryBlockInfo TextureBlockInfo { get; } = new BinaryBlockInfo();

        /// <summary>
        /// Primitive のバイナリブロック情報を取得または設定します。
        /// </summary>
        public BinaryBlockInfo PrimitiveBlockInfo { get; } = new BinaryBlockInfo();

        /// <summary>
        /// G3dPrimitive のバイナリブロック情報を取得または設定します。
        /// </summary>
        public BinaryBlockInfo G3dPrimitiveBlockInfo { get; } = new BinaryBlockInfo();

        /// <summary>
        /// Shader のバイナリブロック情報を取得または設定します。
        /// </summary>
        public BinaryBlockInfo ShaderBinBlockInfo { get; } = new BinaryBlockInfo();

        /// <summary>
        /// ComputeShader のバイナリブロック情報を取得または設定します。
        /// </summary>
        public BinaryBlockInfo ComputeShaderBinBlockInfo { get; } = new BinaryBlockInfo();

        /// <summary>
        /// ShaderSource のバイナリブロック情報を取得または設定します。
        /// </summary>
        public BinaryBlockInfo ShaderSrcBlockInfo { get; } = new BinaryBlockInfo();

        /// <summary>
        /// プロファイルの出力の行頭部文字列を取得します。
        /// </summary>
        private static string ProfileHeader
        {
            get { return SpecManager.CurrentSpec.BinaryHeader == "EFTB" ? "[Eft2Conv]" : "[VfxConv]"; }
        }

        /// <summary>
        /// セッションを終了するときの処理を行います。
        /// </summary>
        /// <param name="contextMap">バイナリ構造データ</param>
        /// <param name="binarySize">バイナリサイズ</param>
        public void OnFinalizeSession(Dictionary<DataModelBase, WriteBinaryDataContext> contextMap, long binarySize)
        {
            if (IOConstants.DetailCommandLineOutputMode)
            {
                this.OutputEmitterSetAndEmitterSize(contextMap);
            }

            if (this.EsetProfileEnabled)
            {
                var esetContextPairs = from p in contextMap
                                       where p.Key is EmitterSetData
                                       select p;

                this.EsetBinarySize = esetContextPairs.Sum(p => p.Value.WriteSize);
            }

            this.BinarySize = binarySize;
        }

        /// <summary>
        /// Output the binary size of emitter sets and emitters.
        /// </summary>
        /// <param name="contextMap">バイナリ構造データ</param>
        private void OutputEmitterSetAndEmitterSize(Dictionary<DataModelBase, WriteBinaryDataContext> contextMap)
        {
            var esetContextPairs = from p in contextMap
                                   where p.Key is EmitterSetData
                                   select p;

            var emitterContextPairs = from p in contextMap
                                      where p.Key is EmitterData
                                      select p;

            foreach (KeyValuePair<DataModelBase, WriteBinaryDataContext> pair in esetContextPairs)
            {
                EmitterSetData eset = (EmitterSetData)pair.Key;
                Logger.Log(
                    LogLevels.Profile,
                    "{0} Emitter set '{2}' binary size : {1} bytes",
                    ProfileHeader,
                    pair.Value.WriteSize,
                    eset.Name);
            }

            foreach (KeyValuePair<DataModelBase, WriteBinaryDataContext> pair in emitterContextPairs)
            {
                EmitterData emitter = (EmitterData)pair.Key;
                Logger.Log(
                    LogLevels.Profile,
                    "{0} Emitter '{2}' binary size : {1} bytes",
                    ProfileHeader,
                    pair.Value.WriteSize,
                    emitter.Name);
            }
        }

        /// <summary>
        /// バイナリブロック情報を格納するクラスです。
        /// </summary>
        public class BinaryBlockInfo
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public BinaryBlockInfo()
            {
            }

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="count">リソースの個数</param>
            /// <param name="size">コンバート後のサイズ</param>
            /// <param name="time">コンバートにかかった時間</param>
            public BinaryBlockInfo(int count, long size, double time)
            {
                this.Set(count, size, time);
            }

            /// <summary>
            /// リソースの個数を取得または設定します。
            /// </summary>
            public int Count { get; set; }

            /// <summary>
            /// コンバート後のサイズを取得または設定します。
            /// </summary>
            public long Size { get; set; }

            /// <summary>
            /// コンバートにかかった時間を取得または設定します。
            /// </summary>
            public double Time { get; set; }

            /// <summary>
            /// バイナリブロック情報を設定します。
            /// </summary>
            /// <param name="count">リソースの個数</param>
            /// <param name="size">コンバート後のサイズ</param>
            /// <param name="time">コンバートにかかった時間</param>
            public void Set(int count, long size, double time)
            {
                this.Count = count;
                this.Size = size;
                this.Time = time;
            }
        }
    }
}
