﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.BinaryHeaders.Helpers;
using EffectMaker.BusinessLogic.BinaryResourceWriters;
using EffectMaker.BusinessLogic.BinaryResourceWriters.Primitive;
using EffectMaker.BusinessLogic.BinaryResourceWriters.Shader;
using EffectMaker.BusinessLogic.BinaryResourceWriters.Texture;
using EffectMaker.BusinessLogic.EffectCombinerEditor;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.BinaryData;
using EffectMaker.DataModelLogic.Extensions;
using EffectMaker.Foundation.Debugging.Profiling;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.BusinessLogic.Manager;

namespace EffectMaker.DataModelLogic.Utilities
{
    /// <summary>
    /// Manages and holds data when writing data model to binary data.
    /// </summary>
    public class WriteBinaryDataSession
    {
        /// <summary>
        /// バイナリ構造ヘッダのアライメントサイズです。
        /// </summary>
        public const int BinaryStructHeaderAlignment = 16;

        /// <summary>
        /// バイナリサイズです。
        /// </summary>
        public long BinarySize { get; private set; } = 0;

        /// <summary>The list of sessions that haven't finished their tasks.</summary>
        private static readonly List<WriteBinaryDataSession> Instances =
            new List<WriteBinaryDataSession>();

        /// <summary>
        /// The dictionary that maps the data model of binary structures to the context instances.
        /// </summary>
        private readonly Dictionary<DataModelBase, WriteBinaryDataContext> contextMap =
            new Dictionary<DataModelBase, WriteBinaryDataContext>();

        /// <summary>
        /// The tagged binary fields that had written their data to the stream.
        /// </summary>
        private readonly Dictionary<string, List<WriteBinaryFieldInfo>> taggedBinaryFields =
            new Dictionary<string, List<WriteBinaryFieldInfo>>();

        /// <summary>
        /// A stack that stores the tagged binary fields that is being written.
        /// </summary>
        private readonly Stack<WriteBinaryFieldInfo> taggedBinaryFieldStack =
            new Stack<WriteBinaryFieldInfo>();

        /// <summary>The name of the binary file.</summary>
        private string binaryName = string.Empty;

        /// <summary>The stream that the binary data is being written to.</summary>
        private Stream stream = null;

        /// <summary>The asset type for the session.</summary>
        private AssetTypes sessionAssetType;

        /// <summary>True to enable asynchronous session.</summary>
        private bool isAsync = true;

        /// <summary>The callback function called when the asynchronous session is finished.</summary>
        private Action<bool, bool> sessionCompleteCallback = null;

        /// <summary>The position of the next bit of the end of binary file header.</summary>
        private long positionAfterBinaryFileHeader = 0;

        /// <summary>The binary file header.</summary>
        private BinaryFileHeader binaryFileHeader = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="stream">
        /// The stream for writing the binary data.
        /// DO NOT close the stream if planning on using asynchronous session.
        /// The stream will be closed by the session when all the data is written.
        /// </param>
        /// <param name="assetType">The asset type for th session.</param>
        /// <param name="binaryName">The binary name.</param>
        /// <param name="completeCallback">
        /// The callback function called when the session is finished.
        /// If the session is canceled the boolean parameter pass to the function is set to true.
        /// </param>
        /// <param name="enableAsync">True to enable asynchronous session.</param>
        /// <remarks>
        /// For now, asynchronous session only works for shader compile of the emitter sets.
        /// The rest of the operations are still executed synchronously.
        /// (binary data of the data models, textures, primitives, shader source code, ... etc.)
        /// </remarks>
        public WriteBinaryDataSession(
            Stream stream,
            AssetTypes assetType,
            string binaryName,
            Action<bool, bool> completeCallback = null,
            bool enableAsync = true)
        {
            if (stream == null)
            {
                throw new ArgumentException("The stream is mandatory, the stream in the argument is null.");
            }

            if (stream.CanSeek == false || stream.CanWrite == false)
            {
                throw new ArgumentException("The stream is invalid because it cannot be sought or written.");
            }

            // Store the instance to prevent it from getting
            // garbage collected because no one reference it.
            Instances.Add(this);

            this.binaryName = binaryName;
            this.sessionAssetType = assetType;
            this.stream = stream;
            this.sessionCompleteCallback = completeCallback;
            this.isAsync = enableAsync;
            this.Profiler = new WriteBinaryDataSessionProfiler();

            // シェーダバイナリを別ファイルで出力する場合のファイルパス
            this.ShaderBinaryOutputFilePath = string.Empty;
            this.ComputeShaderBinaryOutpufFilePath = string.Empty;
            this.TextureBinaryOutputFilePath = string.Empty;
            this.G3dPrimitiveBinaryOutputFilePath = string.Empty;

            // 通常はバイナリを出力するセッションごとにテーブルをクリアする
            if (OptionStore.RuntimeOptions.IsSubBinaryConverting == false)
            {
                TextureManager.Instance.ClearGuidTable();
                PrimitiveManager.Instance.ClearGuidTable();
            }

            this.WriteBinaryFileHeader();
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="stream">The stream to write binary data to.</param>
        private WriteBinaryDataSession(Stream stream)
        {
            // Store the instance to prevent it from getting
            // garbage collected because no one reference it.
            Instances.Add(this);
            this.stream = stream;
            this.Profiler = new WriteBinaryDataSessionProfiler();
        }

        /// <summary>
        /// Get the output stream for the session.
        /// </summary>
        public Stream Stream
        {
            get { return this.stream; }
        }

        /// <summary>
        /// プロファイラを取得します。
        /// </summary>
        public WriteBinaryDataSessionProfiler Profiler { get; private set; }

        /// <summary>
        /// 通常シェーダバイナリを別ファイルとして出力するときのファイルパス
        /// </summary>
        public string ShaderBinaryOutputFilePath { get; set; }

        /// <summary>
        /// コンピュートシェーダバイナリを別ファイルとして出力するときのファイルパス
        /// </summary>
        public string ComputeShaderBinaryOutpufFilePath { get; set; }

        /// <summary>
        /// テクスチャバイナリを別ファイルとして出力するときのファイルパス
        /// </summary>
        public string TextureBinaryOutputFilePath { get; set; }

        /// <summary>
        /// G3Dプリミティブバイナリを別ファイルとして出力するときのファイルパス
        /// </summary>
        public string G3dPrimitiveBinaryOutputFilePath { get; set; }

        /// <summary>
        /// 全テクスチャをname tableに記録するかしないか
        /// </summary>
        public bool ForceToWriteAllTexturesToNameTable { get; set; }

        /// <summary>
        /// エミッタセットのファイルパスを含めるかどうか
        /// </summary>
        public bool IncludeEmitterSetFilePathArray { get; set; }

        /// <summary>
        /// Create a session for writing the given binary field instance to the stream.
        /// </summary>
        /// <param name="binaryField">The binary field.</param>
        /// <param name="stream">The stream to write the binary data to.</param>
        /// <returns>True on success.</returns>
        public static bool WriteBinaryField(
            BinaryFieldInstance binaryField,
            Stream stream)
        {
            var session = new WriteBinaryDataSession(stream);

            // Write binary field.
            if (binaryField.WriteBinaryData(session) == false)
            {
                session.CancelSession();
                return false;
            }

            //// Do not call session.EndSession(), because we only need the binary data of the field.

            return true;
        }

        /// <summary>
        /// シェーダのコンバート前にエミッタから必要な情報を抽出します。
        /// </summary>
        /// <param name="emitter"></param>
        /// <param name="parentStream"></param>
        /// <param name="binaryData"></param>
        /// <param name="childStream"></param>
        /// <param name="contexts"></param>
        /// <returns></returns>
        public static bool WriteBinaryForPrepareEmitter(
            EmitterData emitter,
            Stream parentStream,
            BinaryStructInstance binaryData,
            out Stream childStream,
            out IEnumerable<WriteBinaryDataContext> contexts)
        {
            var session = new WriteBinaryDataSession(parentStream);

            childStream = session.stream;

            session.WriteBinaryFileHeader();

            // Write binary data.
            if (binaryData.WriteBinaryData(session) == false)
            {
                session.CancelSession();
                contexts = null;
                return false;
            }

            // Do not call session.EndSession(), because we only need the emitter binary.
            session.WriteBinaryHeaders();

            contexts =
                from ctx in session.contextMap.Values
                where ctx.BinaryStruct.DataModel.Guid == emitter.Guid
                select ctx;

            return true;
        }

        /// <summary>
        /// End the session.
        /// </summary>
        /// <returns>True on success.</returns>
        public bool EndSession()
        {
            if (this.stream == null)
            {
                // The session is probably canceled, no need to process anymore.
                this.FinalizeSession(false);
                return false;
            }

            if (this.contextMap == null)
            {
                // This session is only used to carry the stream through the binary data
                // structures and fields while writing binary data, the rest of the process
                // can be skipped.
                this.FinalizeSession(false);
                return true;
            }

            // Create the callback to finish the session.
            var actionToFinishSession = new Action<bool>((shaderCompileFailed) =>
            {
                if (this.binaryFileHeader.WriteFileSize != null)
                {
                    this.binaryFileHeader.WriteFileSize();
                    this.BinarySize = this.binaryFileHeader.BinarySize;
                }

                if (this.sessionCompleteCallback != null)
                {
                    this.sessionCompleteCallback(true, shaderCompileFailed);
                }

                this.FinalizeSession(true);
            });

            // Write binary structure headers.
            this.WriteBinaryHeaders();

            if (this.sessionAssetType == AssetTypes.EmitterSet)
            {
                if ((OptionStore.RuntimeOptions.IsCommandLineMode == false && OptionStore.ProjectConfig.IncludeEmitterSetFilePathArrayInEditorMode == true) ||
                    this.IncludeEmitterSetFilePathArray == true)
                {
                    if (this.WriteEmitterSetFilePathArray() == false)
                    {
                        this.FinalizeSession(false);
                        return false;
                    }
                }

                if (SpecManager.CurrentSpec.TexturePackingOption != null)
                {
                    if (this.WriteGfxTextureResources() == false)
                    {
                        this.FinalizeSession(false);
                        return false;
                    }
                }
                else
                {
                    if (this.WriteTextureResources() == false)
                    {
                        this.FinalizeSession(false);
                        return false;
                    }
                }

                if (this.WritePrimitiveResources() == false)
                {
                    this.FinalizeSession(false);
                    return false;
                }

                if (SpecManager.CurrentSpec.BinaryHeader == "VFXB")
                {
                    if (this.WriteG3dPrimitiveResource() == false)
                    {
                        this.FinalizeSession(false);
                        return false;
                    }
                }

                if (this.WriteShaderCodes() == false)
                {
                    this.FinalizeSession(false);
                    return false;
                }

                if (this.WriteShaderResources(this.isAsync, actionToFinishSession) == false)
                {
                    this.FinalizeSession(false);
                    return false;
                }
            }
            else if (this.sessionAssetType == AssetTypes.ViewerData)
            {
                if (this.WriteViewerTextureResources() == false)
                {
                    this.FinalizeSession(false);
                    return false;
                }

                // Finalize the session.
                actionToFinishSession(true);
            }
            else if (this.sessionAssetType == AssetTypes.ViewerModel)
            {
                if (this.WriteModelResources() == false)
                {
                    this.FinalizeSession(false);
                    return false;
                }

                // Finalize the session.
                actionToFinishSession(true);
            }
            else
            {
                // Finalize the session.
                actionToFinishSession(true);
            }

            return true;
        }

        /// <summary>
        /// Cancel the session without writing the binary data.
        /// </summary>
        public void CancelSession()
        {
            this.FinalizeSession(false);
        }

        /// <summary>
        /// Called when a binary structure is about to be written to the stream.
        /// </summary>
        /// <param name="binaryStruct">The binary structure.</param>
        public void BeginWriteBinaryStructure(BinaryStructInstance binaryStruct)
        {
            if (this.contextMap == null)
            {
                return;
            }

            if (binaryStruct == null ||
                binaryStruct.Definition == null ||
                binaryStruct.Definition.HasBinaryHeader == false)
            {
                return;
            }

            // Find the context for the root binary data structure.
            if (this.contextMap.ContainsKey(binaryStruct.DataModel) == true)
            {
                throw new ArgumentException("The binary structure has already been written to the binary data.");
            }

            // Update the child list and parent references for the data model.
            binaryStruct.DataModel.UpdateKinship();

            // Save the current stream position. (the position of the binary structure header)
            long pos = this.stream.Position;

            // Compute how many empty bytes to write for the binary header.
            var helper = BinaryHeaderHelperSelector.GetHelper(binaryStruct.DataModel);
            int offset = (int)helper.GetOffset(binaryStruct.DataModel, (uint)pos);

            // Insert a binary structure header before the binary structure is written to the stream.
            BinaryConversionUtility.ForResource.WriteStream(this.stream, Enumerable.Repeat((byte)0, offset).ToArray());

            // Create a context for the binary structure.
            var context = new WriteBinaryDataContext(pos, binaryStruct);
            this.contextMap.Add(binaryStruct.DataModel, context);
        }

        /// <summary>
        /// Called when finished writing binary structure data to the stream.
        /// </summary>
        /// <param name="binaryStruct">The binary structure.</param>
        public void EndWriteBinaryStructure(BinaryStructInstance binaryStruct)
        {
            if (this.contextMap == null)
            {
                return;
            }

            WriteBinaryDataContext context;
            if (this.contextMap.TryGetValue(binaryStruct.DataModel, out context) == false ||
                context == null)
            {
                return;
            }

            // Ask the context to compute size according to the current stream position.
            context.ComputeSize(this.stream.Position);
        }

        /// <summary>
        /// Called when binary field data is written to the stream.
        /// </summary>
        /// <param name="binaryField">The binary field.</param>
        public void BeginWriteBinaryField(BinaryFieldInstance binaryField)
        {
            if (this.taggedBinaryFields == null)
            {
                return;
            }

            // We only want to keep the binary fields who has a tag defined.
            if (string.IsNullOrEmpty(binaryField.Tag) == true)
            {
                return;
            }

            // Find or create the list of binary field info with the tag.
            List<WriteBinaryFieldInfo> infoList;
            if (this.taggedBinaryFields.TryGetValue(binaryField.Tag, out infoList) == false)
            {
                infoList = new List<WriteBinaryFieldInfo>();
                this.taggedBinaryFields.Add(binaryField.Tag, infoList);
            }

            // Remember the reference to the binary field and it's position in the stream.
            var info = new WriteBinaryFieldInfo()
            {
                BinaryField = binaryField,
                Position = this.stream.Position,
                Size = 0,
            };

            infoList.Add(info);
            this.taggedBinaryFieldStack.Push(info);
        }

        /// <summary>
        /// Called when finished writing binary field data to the stream.
        /// </summary>
        /// <param name="binaryField">The binary field.</param>
        public void EndWriteBinaryField(BinaryFieldInstance binaryField)
        {
            if (this.taggedBinaryFieldStack == null)
            {
                return;
            }

            if (string.IsNullOrEmpty(binaryField.Tag) == true)
            {
                return;
            }

            WriteBinaryFieldInfo info = this.taggedBinaryFieldStack.Pop();
            if (info.BinaryField != binaryField)
            {
                Logger.Log(LogLevels.Debug, "WriteBinaryDataContext.EndWriteBinaryField : The corresponding WriteBinaryFieldInfo is not found.");
                return;
            }

            // Save the size of the binary field.
            info.Size = this.stream.Position - info.Position;
        }

        /// <summary>
        /// Finish the session, clear the resources, remove itself from the instance list
        /// and output necessary information.
        /// </summary>
        /// <param name="outputProfileInfo">True to output profiling information.</param>
        private void FinalizeSession(bool outputProfileInfo)
        {
            // Output the profiling information for this session.
            if (outputProfileInfo == true)
            {
                this.Profiler.OnFinalizeSession(this.contextMap, this.stream.Length);
            }

            if (this.stream != null && this.stream.CanRead == true)
            {
                this.stream.Close();
            }

            this.stream = null;
            this.sessionCompleteCallback = null;

            if (this.contextMap != null)
            {
                this.contextMap.Clear();
            }

            if (this.taggedBinaryFields != null)
            {
                this.taggedBinaryFields.Clear();
            }

            if (this.taggedBinaryFieldStack != null)
            {
                this.taggedBinaryFieldStack.Clear();
            }

            Instances.Remove(this);
        }

        /// <summary>
        /// Write EmitterSet file path array.
        /// </summary>
        /// <returns>True on success.</returns>
        private bool WriteEmitterSetFilePathArray()
        {
            this.stream.Seek(0, SeekOrigin.End);

            var esetContexts =
                from ctx in this.contextMap.Values
                where ctx.BinaryStruct.DataModel is EmitterSetData
                select ctx;

            // Write alignment.
            this.WriteAlignment(BinaryStructHeaderAlignment);

            uint alignmentToNext = BinaryStructHeaderAlignment;  // エミッタセットパステーブルの次につけるバイナリヘッダ(テクスチャ)のアライン

            using (var context = new BinaryResourceWriterContext(this.stream, alignmentToNext, false))
            {
                var writer = new EmitterSetFilePathArrayWriter(esetContexts.Select((ctx) => (EmitterSetData)ctx.BinaryStruct.DataModel));
                if (writer.Write(context) == false)
                {
                    return false;
                }
            }

            this.stream.Seek(0, SeekOrigin.End);

            return true;
        }

        /// <summary>
        /// Write texture binary data.
        /// </summary>
        /// <returns>True on success.</returns>
        private bool WriteTextureResources()
        {
            this.stream.Seek(0, SeekOrigin.End);

            var esetContexts =
                from ctx in this.contextMap.Values
                where ctx.BinaryStruct.DataModel is EmitterSetData
                select ctx;

            // Write alignment.
            this.WriteAlignment(BinaryStructHeaderAlignment);

            // We don't want to include the space taken for the alignment,
            // that's why we save the stream position here, instead of the
            // beginning of the method.
            var position = this.stream.Position;
            var timer = new ProfileTimer("Texture");

            uint alignmentToNext = BinaryStructHeaderAlignment;  // テクスチャの次につけるバイナリヘッダ(プリミティブ)のアライン
            int textureCount = 0;
            using (var context = new BinaryResourceWriterContext(this.stream, alignmentToNext, false))
            {
                var writer = new TextureArrayWriter(
                    esetContexts.Select((ctx) => (EmitterSetData)ctx.BinaryStruct.DataModel));

                textureCount = writer.TextureCount;
                if (writer.Write(context) == false)
                {
                    return false;
                }
            }

            timer.Stop(false);
            this.stream.Seek(0, SeekOrigin.End);

            // ブロック情報を記録
            Profiler.TextureBlockInfo.Set(textureCount, this.stream.Position - position, timer.ElapsedTime.TotalSeconds);

            return true;
        }

        /// <summary>
        /// GFX版のテクスチャバイナリをストリームに書き込みます。
        /// </summary>
        /// <returns>処理が成功したときTrueを返します。</returns>
        private bool WriteGfxTextureResources()
        {
            this.stream.Seek(0, SeekOrigin.End);

            var esetContexts =
                from ctx in this.contextMap.Values
                where ctx.BinaryStruct.DataModel is EmitterSetData
                select ctx;

            // Write alignment.
            this.WriteAlignment(BinaryStructHeaderAlignment);

            // We don't want to include the space taken for the alignment,
            // that's why we save the stream position here, instead of the
            // beginning of the method.
            var position = this.stream.Position;
            var timer = new ProfileTimer("Texture");

            uint alignmentToNext = BinaryStructHeaderAlignment;  // テクスチャの次につけるバイナリヘッダ(プリミティブ)のアライン
            int textureCount = 0;
            using (var context = new BinaryResourceWriterContext(this.stream, alignmentToNext, false))
            {
                var writer = new GfxTextureArrayWriter(
                    esetContexts.Select((ctx) => (EmitterSetData)ctx.BinaryStruct.DataModel),
                    this.TextureBinaryOutputFilePath,
                    this.ForceToWriteAllTexturesToNameTable);

                textureCount = writer.TextureCount;

                if (writer.Write(context) == false)
                {
                    return false;
                }

                // アライメントを更新
                if (this.binaryFileHeader.WriteAlignmentSize != null)
                {
                    this.binaryFileHeader.WriteAlignmentSize(context.AdjustedAlignment);
                }
            }

            timer.Stop(false);
            this.stream.Seek(0, SeekOrigin.End);

            // ブロック情報を記録
            Profiler.TextureBlockInfo.Set(textureCount, this.stream.Position - position, timer.ElapsedTime.TotalSeconds);

            return true;
        }

        /// <summary>
        /// Write primitive binary data.
        /// </summary>
        /// <returns>True on success.</returns>
        private bool WritePrimitiveResources()
        {
            this.stream.Seek(0, SeekOrigin.End);

            var esetContexts =
                from ctx in this.contextMap.Values
                where ctx.BinaryStruct.DataModel is EmitterSetData
                select ctx;

            // Write alignment.
            this.WriteAlignment(BinaryStructHeaderAlignment);

            // We don't want to include the space taken for the alignment,
            // that's why we save the stream position here, instead of the
            // beginning of the method.
            var position = this.stream.Position;
            var timer = new ProfileTimer("Primitive");

            uint alignmentToNext = BinaryStructHeaderAlignment;  // プリミティブの次につけるバイナリヘッダ(シェーダーソースorバイナリ)のアライン

            int primitiveCount = 0;
            using (var context = new BinaryResourceWriterContext(this.stream, alignmentToNext, false))
            {
                var writer = new PrimitiveArrayWriter(
                    esetContexts.Select((ctx) => (EmitterSetData)ctx.BinaryStruct.DataModel));

                primitiveCount = writer.PrimitiveCount;
                if (writer.Write(context) == false)
                {
                    return false;
                }
            }

            timer.Stop(false);
            this.stream.Seek(0, SeekOrigin.End);

            // ブロック情報を記録
            Profiler.PrimitiveBlockInfo.Set(primitiveCount, this.stream.Position - position, timer.ElapsedTime.TotalSeconds);

            return true;
        }

        /// <summary>
        /// Write g3d primitive binary data.
        /// </summary>
        /// <returns>True on success.</returns>
        private bool WriteG3dPrimitiveResource()
        {
            this.stream.Seek(0, SeekOrigin.End);

            var esetContexts =
                from ctx in this.contextMap.Values
                where ctx.BinaryStruct.DataModel is EmitterSetData
                select ctx;

            // Write alignment.
            this.WriteAlignment(BinaryStructHeaderAlignment);

            // We don't want to include the space taken for the alignment,
            // that's why we save the stream position here, instead of the
            // beginning of the method.
            var position = this.stream.Position;
            var timer = new ProfileTimer("G3d Primitive");

            uint alignmentToNext = BinaryStructHeaderAlignment;  // プリミティブの次につけるバイナリヘッダ(シェーダーソースorバイナリ)のアライン

            int primitiveCount = 0;
            bool lastBinaryResource = !string.IsNullOrEmpty(this.ComputeShaderBinaryOutpufFilePath) &
                                      !string.IsNullOrEmpty(this.ShaderBinaryOutputFilePath);
            using (var context = new BinaryResourceWriterContext(this.stream, alignmentToNext, lastBinaryResource))
            {
                var writer = new G3dPrimitiveArrayWriter(
                    esetContexts.Select((ctx) => (EmitterSetData)ctx.BinaryStruct.DataModel),
                    this.G3dPrimitiveBinaryOutputFilePath);

                primitiveCount = writer.PrimitiveCount;
                if (writer.Write(context) == false)
                {
                    return false;
                }
            }

            timer.Stop(false);
            this.stream.Seek(0, SeekOrigin.End);

            // ブロック情報を記録
            Profiler.G3dPrimitiveBlockInfo.Set(primitiveCount, this.stream.Position - position, timer.ElapsedTime.TotalSeconds);

            return true;
        }

        /// <summary>
        /// ビューアノードの次に背景テクスチャバイナリを書き込みます。
        /// </summary>
        /// <returns>True on success.</returns>
        private bool WriteViewerTextureResources()
        {
            var viewerContext = this.contextMap.Values.FirstOrDefault(
                item => item.BinaryStruct.DataModel is ViewerData);
            if (viewerContext == null)
            {
                return false;
            }

            this.stream.Seek(0, SeekOrigin.End);

            uint alignmentToNext = BinaryStructHeaderAlignment;  // 背景テクスチャの次につけるバイナリヘッダ(モデル)のアライン
            using (var context = new BinaryResourceWriterContext(this.stream, alignmentToNext, false))
            {
                var viewData = (ViewerData)viewerContext.BinaryStruct.DataModel;
                var writer = new ViewerTextureResourceWriter(viewData);

                if (writer.Write(context) == false)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// モデルノードの後ろにモデルバイナリを書き込みます。
        /// </summary>
        /// <returns>True on success.</returns>
        private bool WriteModelResources()
        {
            var modelContext = this.contextMap.Values.FirstOrDefault(
                item => item.BinaryStruct.DataModel is ModelData);
            if (modelContext == null)
            {
                return false;
            }

            this.stream.Seek(0, SeekOrigin.End);

            uint alignmentToNext = 0;  // モデルの次につけるバイナリ(なし)のアライン
            using (var context = new BinaryResourceWriterContext(this.stream, alignmentToNext, false))
            {
                var modelData = (ModelData)modelContext.BinaryStruct.DataModel;
                var writer = new ModelPrimitiveResourceWriter(modelData);

                if (writer.Write(context) == false)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// Write shader code data.
        /// </summary>
        /// <returns>True on success.</returns>
        private bool WriteShaderCodes()
        {
            // ここの処理を通さないと並列コンバート+カスタムシェーダ使用時にコンバート失敗になる可能性がある
            var esetContexts =
                from ctx in this.contextMap.Values
                where ctx.BinaryStruct.DataModel is EmitterSetData
                select ctx;

            var emitterSets = esetContexts.Select((ctx) => (EmitterSetData)ctx.BinaryStruct.DataModel);
            var combinerShaderPaths = this.WriteEftCombinerShaderIndex(emitterSets);
            // ここまで

            // シェーダーソース省略フラグが立っていたとき、シェーダーコード配列書き込みフラグが立っていなかったときはここでリターン
            if (OptionStore.RuntimeOptions.NoShaderSource || !SpecManager.CurrentSpec.ShaderConversionOption.WriteCodeArray)
            {
                return true;
            }

            this.stream.Seek(0, SeekOrigin.End);

            // Write alignment.
            this.WriteAlignment(BinaryStructHeaderAlignment);

            // We don't want to include the space taken for the alignment,
            // that's why we save the stream position here, instead of the
            // beginning of the method.
            var position = this.stream.Position;
            var timer = new ProfileTimer("ShaderSrc");

            uint alignmentToNext = BinaryStructHeaderAlignment;  // シェーダソースコードの次につけるバイナリヘッダ(シェーダバイナリ)のアライン
            using (var resWriterContext = new BinaryResourceWriterContext(this.stream, alignmentToNext, false))
            {
                var writer = new ShaderArrayWriter(emitterSets, combinerShaderPaths);

                if (writer.Write(resWriterContext) == false)
                {
                    return false;
                }
            }

            timer.Stop(false);
            this.stream.Seek(0, SeekOrigin.End);

            // ブロック情報を記録
            this.Profiler.ShaderSrcBlockInfo.Set(-1, this.stream.Position - position, timer.ElapsedTime.TotalSeconds);

            return true;
        }

        /// <summary>
        /// Write shader binary data.
        /// </summary>
        /// <param name="enableAsync">True to compile shader asynchronously.</param>
        /// <param name="completeCallback">The callback to execute when finished.</param>
        /// <returns>True on success.</returns>
        private bool WriteShaderResources(
            bool enableAsync,
            Action<bool> completeCallback)
        {
            var emitterContexts =
                from ctx in this.contextMap.Values
                where ctx.BinaryStruct.DataModel is EmitterData
                select ctx;

            this.stream.Seek(0, SeekOrigin.End);

            // Write alignment.
            this.WriteAlignment(BinaryStructHeaderAlignment);

            var shaderBinHelper = new ShaderBinaryHelper();

            return shaderBinHelper.WriteShaderResourcesImpl(
                this.binaryName,
                this.stream,
                this.ShaderBinaryOutputFilePath,
                this.ComputeShaderBinaryOutpufFilePath,
                enableAsync,
                completeCallback,
                emitterContexts,
                this.HandleShaderCompileComplete);
        }

        /// <summary>
        /// Called when shader compile is complete to process the compile result
        /// and write the data to our stream.
        /// </summary>
        /// <param name="emitterOffsets">The offsets of the emitters in the stream.</param>
        /// <param name="completeCallback">The callback to execute when finished.</param>
        /// <param name="timer">The profiler timer to compute the elapsed time.</param>
        /// <param name="isSuccessful">True when the shader compiled successfully.</param>
        /// <param name="shaderCompileFailed">シェーダのコンパイルに一度でも失敗したかどうか</param>
        /// <param name="shaderBinary">The compiled shader binary.</param>
        /// <param name="shaderCount">The number of shaders being compiled.</param>
        /// <param name="writeShaderIndicesAction">シェーダ情報の書き込み動作のコールバックです。</param>
        private void HandleShaderCompileComplete(
            List<long> emitterOffsets,
            Action<bool> completeCallback,
            ProfileTimer timer,
            ProfileTimer computeShaderTimer,
            bool isSuccessful,
            bool shaderCompileFailed,
            byte[] shaderBinary,
            long shaderBinarySize,
            int shaderCount,
            long computeShaderBinarySize,
            int computeShaderCount,
            Action<Stream> writeShaderIndicesAction)
        {
            if (isSuccessful == false || shaderBinary == null)
            {
                if (this.sessionCompleteCallback != null)
                {
                    this.sessionCompleteCallback(false, shaderCompileFailed);
                }

                this.FinalizeSession(false);
                return;
            }

            // 必要があればシェーダインデックスをエミッタデータに書き込む
            if (writeShaderIndicesAction != null)
            {
                long oldLength = this.stream.Length;
                writeShaderIndicesAction(this.stream);

                // バイナリがズレるのでストリームの書き足しは禁止
                System.Diagnostics.Debug.Assert(this.stream.Length == oldLength);
            }

            // Move to the end of the stream.
            this.stream.Seek(0, SeekOrigin.End);

            // Write the shader binary data.
            this.stream.Write(shaderBinary, 0, shaderBinary.Length);

            // Add the profiling information for shader binary conversion.
            timer.Stop(false);
            this.stream.Seek(0, SeekOrigin.End);

            // ブロック情報を記録
            // 通常シェーダ
            this.Profiler.ShaderBinBlockInfo.Set(shaderCount, shaderBinarySize, timer.ElapsedTime.TotalSeconds);

            // コンピュートシェーダ
            if (computeShaderTimer != null)
            {
                this.Profiler.ComputeShaderBinBlockInfo.Set(computeShaderCount, computeShaderBinarySize, computeShaderTimer.ElapsedTime.TotalSeconds);
            }

            // Execute the callback function.
            if (completeCallback != null)
            {
                completeCallback(shaderCompileFailed);
            }
        }

        /// <summary>
        /// Write effect combiner generated shader index to the emitter binary data,
        /// and return the effect combiner shader project file paths that is used.
        /// </summary>
        /// <param name="emitterSets">The emitter sets.</param>
        /// <returns>The effect combiner shader project file paths.</returns>
        private IEnumerable<string> WriteEftCombinerShaderIndex(
            IEnumerable<EmitterSetData> emitterSets)
        {
            IEnumerable<string> combinerShaderPaths = Enumerable.Empty<string>();

            Dictionary<string, int> pathIndexMap = new Dictionary<string, int>();

            if (OptionStore.ProjectConfig.IsEftCombinerEditorEnabled == true)
            {
                // コンバイナファイルの有無と、コンバイナファイルを利用しても空のコードにならないことをチェックする。
                // コンバイナファイルの有無だけでインデックスを割り振ると、
                // 空のコードになっている時にオンラインコンパイルでエラーになるので、中身の有無までチェックする。
                Func<string, bool> isValidCode = (combinerPath) =>
                {
                    var combiner = EffectCombinerCommunicationManager.CommunicationBridge;
                    if (combiner != null)
                    {
                        // コンバイナファイルの有無をチェックする
                        if (string.IsNullOrEmpty(combinerPath) == true)
                        {
                            return false;
                        }

                        // コンバイナファイルを利用しても、空のコードにならないことをチェックする
                        return !string.IsNullOrEmpty(combiner.GetShaderSourceCode(combinerPath));
                    }

                    return false;
                };

                // First find all the effect combiner shader paths in all the emitters.
                combinerShaderPaths =
                    from eset in emitterSets
                    from emitter in eset.AllChildEmitters
                    where isValidCode(emitter.EmitterCombinerData.EmitterCombinerEditorData.CombinerEditorProjectPath)
                    select emitter.EmitterCombinerData.EmitterCombinerEditorData.CombinerEditorProjectPath;

                // Make sure there are no duplicate paths.
                combinerShaderPaths = combinerShaderPaths.Distinct();

                // Save these distinct shader paths in a map with their indices.
                int index = 0;
                foreach (string path in combinerShaderPaths)
                {
                    pathIndexMap.Add(path, index);
                    ++index;
                }
            }

            // Find the information of the effect combiner shader index binary fields.
            List<WriteBinaryFieldInfo> eftCombinerShaderIndexFields;
            if (this.taggedBinaryFields.TryGetValue(
                "eftCombinerShaderIndex",
                out eftCombinerShaderIndexFields) == false)
            {
                return Enumerable.Empty<string>();
            }

            // Fill in the shader index to the fields.
            foreach (WriteBinaryFieldInfo info in eftCombinerShaderIndexFields)
            {
                // Find the nearest parent binary structure instance of
                // the binary field that the source data model is an emitter data.
                var parent = info.BinaryField.FindNearestParentStructByDataModel<EmitterData>();
                if (parent == null)
                {
                    continue;
                }

                var emitter = parent.DataModel as EmitterData;

                string myPath =
                    emitter.EmitterCombinerData.EmitterCombinerEditorData.CombinerEditorProjectPath;

                // Find the index of the effect combiner project path.
                int index;
                if (pathIndexMap.TryGetValue(myPath, out index) == false)
                {
                    index = -1;
                }

                // Write the index to the position the binary field occupies.
                this.stream.Seek(info.Position, SeekOrigin.Begin);
                BinaryConversionUtility.ForResource.WriteStream<int>(this.stream, index);
            }

            // Reset the stream pointer to the end of the stream.
            this.stream.Seek(0, SeekOrigin.End);

            return combinerShaderPaths;
        }

        /// <summary>
        /// Find the context of the given data model from the map and compute the
        /// offset of the context data with the specified base position.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <param name="basePosition">The base position to compute the offset from.</param>
        /// <returns>The offset from the base position.</returns>
        private uint ComputeOffset(DataModelBase dataModel, long basePosition)
        {
            if (dataModel != null)
            {
                WriteBinaryDataContext targetContext;
                if (this.contextMap.TryGetValue(dataModel, out targetContext) == true)
                {
                    return (uint)(targetContext.WritePosition - basePosition);
                }
            }

            return 0xFFFFFFFF;
        }

        /// <summary>
        /// Write the binary file header to the stream.
        /// </summary>
        private void WriteBinaryFileHeader()
        {
            // Write binary file header to the stream.
            this.binaryFileHeader = new BinaryFileHeader()
            {
                BinaryName = this.binaryName,
            };

            this.binaryFileHeader.Write(this.stream);

            // Save the position of the beginning of the binary data and
            // write an empty binary structure header as a place holder.
            this.positionAfterBinaryFileHeader = this.stream.Position;
            BinaryStructHeader.Empty.Write(this.stream);
        }

        /// <summary>
        /// Write the binary header of the binary structure.
        /// </summary>
        private void WriteBinaryHeaders()
        {
            BinaryStructHeader header;

            var rootDataModels = new List<Tuple<WriteBinaryDataContext, DataModelBase>>();

            foreach (WriteBinaryDataContext context in this.contextMap.Values)
            {
                DataModelBase dataModel = context.BinaryStruct.DataModel;

                // Get a binary header helper for the data model.
                var helper = BinaryHeaderHelperSelector.GetHelper(dataModel);
                if (helper == null)
                {
                    continue;
                }

                header = new BinaryStructHeader();

                // Set binary header tag.
                header.Tag = helper.GetTag(dataModel);

                // Set binary structure size.
                header.BinarySize = (uint)context.WriteSize;

                // Set up pointers.
                header.Child = this.ComputeOffset(helper.GetChild(dataModel), context.WritePosition);
                header.Sub = this.ComputeOffset(helper.GetSub(dataModel), context.WritePosition);
                header.Offset = helper.GetOffset(dataModel, (uint)context.WritePosition);
                header.ChildNum = BinaryResourceWriterContext.GetChildNum(helper, dataModel);

                DataModelBase nextDataModel = helper.GetNext(dataModel);
                if (helper.ShouldSetNextToEnd == true)
                {
                    uint end = (uint)(context.WritePosition + context.WriteSize);

                    // Compute alignment to the next block.
                    uint alignment = 0;
                    if (helper.AlignmentToNext > 0)
                    {
                        alignment = end % helper.AlignmentToNext;
                        if (alignment != 0)
                        {
                            alignment = helper.AlignmentToNext - alignment;
                        }
                    }

                    header.Next = (uint)context.WriteSize + alignment;
                }
                else
                {
                    header.Next = this.ComputeOffset(nextDataModel, context.WritePosition);
                }

                // Add the root data models and their next data model to the map.
                // We need this map to find the first root data model in the list.
                if (this.sessionAssetType == AssetTypes.EmitterSet &&
                    context.BinaryStruct.Parent == null)
                {
                    rootDataModels.Add(Tuple.Create(context, nextDataModel));
                }

                // Set the stream position to the beginning of the binary structure.
                // (to where the binary header is)
                this.stream.Seek(context.WritePosition, SeekOrigin.Begin);

                // Write binary header to the stream.
                header.Write(this.stream);
            }

            // Only emitter set needs emitter set array header.
            if (this.sessionAssetType == AssetTypes.EmitterSet)
            {
                this.WriteEmitterSetArrayHeader(rootDataModels);
            }
        }

        /// <summary>
        /// Write emitter set array header.
        /// </summary>
        /// <param name="rootDataModels">
        /// The list of data models and their context who has no parents.
        /// We need this list to find the first emitter set.
        /// </param>
        private void WriteEmitterSetArrayHeader(
            List<Tuple<WriteBinaryDataContext, DataModelBase>> rootDataModels)
        {
            // Loop through the data models that has been set as the "next",
            // remove them from the root data model list, the one left should be
            // the first root data model.
            var nextDataModels = rootDataModels.Select(pair => pair.Item2).ToArray();
            var esetCount = (ushort)rootDataModels.Count;
            foreach (DataModelBase next in nextDataModels)
            {
                if (next != null)
                {
                    rootDataModels.RemoveAll(pair => pair.Item1.BinaryStruct.DataModel == next);
                }
            }

            if (rootDataModels.Count != 1)
            {
                Logger.Log(LogLevels.Debug, "WriteBinaryDataSession.WriteBinaryHeaders : Failed to locate the first root data model.");
            }

            // Write emitter array header.
            if (rootDataModels.Count > 0)
            {
                DataModelBase firstRootDataModel = rootDataModels[0].Item1.BinaryStruct.DataModel;
                if (firstRootDataModel is EmitterSetData)
                {
                    // Write the emitter set array header.
                    WriteBinaryDataContext context = rootDataModels[0].Item1;

                    uint binSize = (uint)(this.stream.Length - this.positionAfterBinaryFileHeader);
                    uint offset = (uint)(context.WritePosition - this.positionAfterBinaryFileHeader);

                    // compute alignment.
                    uint alignmentSize = (uint)this.stream.Length % (uint)BinaryStructHeaderAlignment;
                    if (alignmentSize > 0)
                    {
                        alignmentSize = (uint)BinaryStructHeaderAlignment - alignmentSize;
                    }

                    var header = new BinaryStructHeader()
                    {
                        Tag = "ESTA",
                        BinarySize = binSize,
                        Child = offset,
                        Next = binSize + alignmentSize,
                        Offset = offset,
                        ChildNum = esetCount,
                    };

                    this.stream.Seek(this.positionAfterBinaryFileHeader, SeekOrigin.Begin);
                    header.Write(this.stream);
                }
            }
        }

        /// <summary>
        /// Write alignment to the binary stream.
        /// </summary>
        /// <param name="alignment">The alignment.</param>
        private void WriteAlignment(int alignment)
        {
            if (alignment <= 0)
            {
                return;
            }

            int alignmentSize = (int)this.stream.Length % alignment;
            if (alignmentSize > 0)
            {
                alignmentSize = alignment - alignmentSize;
                var alignmentBytes = Enumerable.Repeat((byte)0, alignmentSize).ToArray();
                this.stream.Write(alignmentBytes, 0, alignmentSize);
            }
        }

        /// <summary>
        /// Holds information about the data written into the stream by a binary field.
        /// </summary>
        private class WriteBinaryFieldInfo
        {
            /// <summary>
            /// Get or set the binary field instance.
            /// </summary>
            public BinaryFieldInstance BinaryField { get; set; }

            /// <summary>
            /// Get or set the position in the stream that the binary field writes to.
            /// </summary>
            public long Position { get; set; }

            /// <summary>
            /// Get or set the data size the binary field writes to the stream.
            /// </summary>
            public long Size { get; set; }
        }
    }

    /// <summary>
    /// Helper class for handling write binary data for the binary elements.
    /// </summary>
    public class WriteBinaryDataBlock : IDisposable
    {
        /// <summary>The WriteBinaryDataSession that handles the binary data.</summary>
        private WriteBinaryDataSession session;

        /// <summary>The binary element that is writing binary data.</summary>
        private IBinaryElementInstance element;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="session">The WriteBinaryDataSession that handles the binary data.</param>
        /// <param name="element">The binary element that is writing binary data.</param>
        public WriteBinaryDataBlock(
            WriteBinaryDataSession session,
            IBinaryElementInstance element)
        {
            this.session = session;
            this.element = element;

            if (this.session != null)
            {
                if (this.element is BinaryFieldInstance)
                {
                    this.session.BeginWriteBinaryField((BinaryFieldInstance)this.element);
                }
                else if (this.element is BinaryStructInstance)
                {
                    this.session.BeginWriteBinaryStructure((BinaryStructInstance)this.element);
                }
            }
        }

        /// <summary>
        /// Dispose this object.
        /// </summary>
        public void Dispose()
        {
            if (this.session != null)
            {
                if (this.element is BinaryFieldInstance)
                {
                    this.session.EndWriteBinaryField((BinaryFieldInstance)this.element);
                }
                else if (this.element is BinaryStructInstance)
                {
                    this.session.EndWriteBinaryStructure((BinaryStructInstance)this.element);
                }
            }
        }
    }
}
