﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.DataModelLogic.BinaryData;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelLogic.Utilities
{
    /// <summary>
    /// Class that collects data that is necessary for writing binary data.
    /// </summary>
    public class WriteBinaryDataContext
    {
        /// <summary>The saved position of the stream.</summary>
        private long streamPosition;

        /// <summary>The size of data that's been written to the stream.</summary>
        private long writeSize;

        /// <summary>The binary structure that is being written to the stream.</summary>
        private BinaryStructInstance binaryStruct;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="pos">The stream position the binary structure is being written to.</param>
        /// <param name="binaryStruct">The binary structure that is being written.</param>
        internal WriteBinaryDataContext(long pos, BinaryStructInstance binaryStruct)
        {
            this.streamPosition = pos;
            this.writeSize = 0;
            this.binaryStruct = binaryStruct;
        }

        /// <summary>
        /// Get the position in the stream that the context starts writing binary data.
        /// </summary>
        public long WritePosition
        {
            get { return this.streamPosition; }
        }

        /// <summary>
        /// Get the size of the data that's been written to the stream by this context.
        /// </summary>
        public long WriteSize
        {
            get { return this.writeSize; }
        }

        /// <summary>
        /// Get the binary structure instance the context holds.
        /// </summary>
        public BinaryStructInstance BinaryStruct
        {
            get { return this.binaryStruct; }
        }

        /// <summary>
        /// Compute size with the given stream position and the position saved on construct.
        /// </summary>
        /// <param name="currPosition">The current stream position.</param>
        internal void ComputeSize(long currPosition)
        {
            this.writeSize = currPosition - this.streamPosition;
            if (this.writeSize < 0)
            {
                Logger.Log(
                    LogLevels.Debug,
                    "WriteBinaryContext.ComputeSize : The computed write size is {0}, something might have gone wrong while writing the binary structure.",
                    this.writeSize);
            }
        }
    }
}
