﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;

using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.ViewerMessages;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.DataModelLogic.BinaryData;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.Utilities
{
    /// <summary>
    /// A helper class for sending messages to the viewer.
    /// </summary>
    public static class ViewerMessageHelper
    {
        /// <summary>
        /// SendBinaryをロックするためのオブジェクト
        /// </summary>
        private static readonly object LockObj = new object();

        /// <summary>
        /// バイナリコンバート停止区間のネスト数です。
        /// </summary>
        public static int DisableBinaryConvertNestCount { get; private set; }

        /// <summary>
        /// バイナリコンバート停止区間の開始位置を指定します。
        /// </summary>
        public static void BeginDisableBinaryConvertSection()
        {
            ++DisableBinaryConvertNestCount;
        }

        /// <summary>
        /// バイナリコンバート停止区間の終了位置を指定します。
        /// </summary>
        public static void EndDisableBinaryConvertSection()
        {
            System.Diagnostics.Debug.Assert(DisableBinaryConvertNestCount > 0, "バイナリコンバート停止区間の指定が不正");

            --DisableBinaryConvertNestCount;
        }

        /// <summary>
        /// メッセージドロップ区間の開始位置を指定します。
        /// </summary>
        public static void BeginMessageDropSection()
        {
            SendMessageQueue.BeginMessageDropSection();
        }

        /// <summary>
        /// メッセージドロップ区間の終了位置を指定します。
        /// </summary>
        public static void EndMessageDropSection()
        {
            SendMessageQueue.EndMessageDropSection();
        }

        /// <summary>
        /// Find the proper parent data model and send it to the viewer.
        /// </summary>
        /// <param name="data">The data to start finding.</param>
        /// <returns>True on success.</returns>
        public static bool FindPropertyParentDataAndSend(DataModelBase data)
        {
            if (DisableBinaryConvertNestCount > 0)
            {
                return true;
            }

            var eset = data.FindSelfOrParentOfType<EmitterSetData>();
            if (eset != null)
            {
                return SendEmitterSet(eset);
            }

            var viewer = data.FindSelfOrParentOfType<ViewerData>();
            if (viewer != null)
            {
                return SendViewer(viewer);
            }

            var model = data.FindSelfOrParentOfType<ModelData>();
            if (model != null)
            {
                return SendModel(model);
            }

            return false;
        }

        /// <summary>
        /// Send all the given emitter sets to the viewer. (both binary data and instances)
        /// </summary>
        /// <param name="emitterSets">The emitter sets to send.</param>
        /// <returns>True on success.</returns>
        public static bool SendEmitterSets(IEnumerable<EmitterSetData> emitterSets)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false ||
                DisableBinaryConvertNestCount > 0)
            {
                return true;
            }

            bool allSucceeded = true;
            foreach (EmitterSetData eset in emitterSets)
            {
                if (SendEmitterSet(eset, true) == false)
                {
                    allSucceeded = false;
                }
            }

            foreach (EmitterSetData eset in emitterSets)
            {
                if (SendEmitterSetVisibility(eset) == false)
                {
                    allSucceeded = false;
                }
            }

            return allSucceeded;
        }

        /// <summary>
        /// Send the given emitter set to the viewer. (both binary data and instances)
        /// </summary>
        /// <param name="eset">The emitter set.</param>
        /// <param name="withoutVisibility">Visibilityの通信を後で行う場合はtrue,そうでない場合は省略するかfalse</param>
        /// <returns>True on success.</returns>
        public static bool SendEmitterSet(EmitterSetData eset, bool withoutVisibility = false)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false ||
                DisableBinaryConvertNestCount > 0)
            {
                return true;
            }

            // First delete the emitter set.
            // RequestDeleteEmitterSet(eset);

            // Send the emitter set.
            if (SendBinary(eset) == false)
            {
                Logger.Log(LogLevels.Error, "ViewerMessageHelper.SendEmitterSet : Failed sending the emitter set '{0}' to viewer.", eset.Name);
                return false;
            }

            // Loop through all the preview data models in the emitter set.
            foreach (PreviewData preview in eset.PreviewList)
            {
                // Ask the viewer to create instances of the sent data models.
                if (RequestCreateInstance(preview, eset.Guid) == false)
                {
                    Logger.Log(LogLevels.Error, "ViewerMessageHelper.SendEmitterSet : Failed requesting viewer to create the instance of preview data '{0}.{1}'.", eset.Name, preview.Name);
                    return false;
                }

                // Send the preview data models of the emitter set.
                if (SendBinary(preview) == false)
                {
                    Logger.Log(LogLevels.Error, "ViewerMessageHelper.SendEmitterSet : Failed sending preview data '{0}.{1}' to viewer.", eset.Name, preview.Name);
                    return false;
                }
            }

            if (withoutVisibility)
            {
                return true;
            }

            return SendEmitterSetVisibility(eset);
        }

        /// <summary>
        /// エミッタセットとそのチャイルド要素における非表示要素を通信します。
        /// </summary>
        /// <param name="eset">エミッタセット</param>
        /// <returns>通信に成功したらtrue.</returns>
        public static bool SendEmitterSetVisibility(EmitterSetData eset)
        {
            if (MessageBase.ShouldSendMessage == false)
            {
                return true;
            }

            // Send visibility message if the emitter set data is hidden.
            if (eset.Displayed == false)
            {
                // Send visibility message to the viewer.
                ViewerController.Instance.SendVisibility(eset);
            }

            // Send visibility message if the emitter data is hidden.
            foreach (EmitterData emitter in eset.AllActiveChildEmitters)
            {
                if (emitter.Displayed == false)
                {
                    // Send visibility message to the viewer.
                    ViewerController.Instance.SendVisibility(emitter);
                }
            }

            // Loop through all the preview data models in the emitter set.
            foreach (PreviewData preview in eset.PreviewList)
            {
                // Send visibility message if the preview data is hidden.
                if (preview.Displayed == false)
                {
                    // Send visibility message to the viewer.
                    ViewerController.Instance.SendVisibility(preview);
                }
            }

            return true;
        }

        /// <summary>
        /// Request the viewer to delete the emitter sets. (both binary data and instances)
        /// </summary>
        /// <param name="emitterSets">The emitter sets.</param>
        /// <returns>True on success.</returns>
        public static bool RequestDeleteEmitterSets(IEnumerable<EmitterSetData> emitterSets)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false)
            {
                return true;
            }

            bool allSucceeded = true;
            foreach (EmitterSetData eset in emitterSets)
            {
                if (RequestDeleteEmitterSet(eset) == false)
                {
                    allSucceeded = false;
                }
            }

            return allSucceeded;
        }

        /// <summary>
        /// Request the viewer to delete the emitter set. (both binary data and instances)
        /// </summary>
        /// <param name="eset">The emitter set.</param>
        /// <returns>True on success.</returns>
        public static bool RequestDeleteEmitterSet(EmitterSetData eset)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false)
            {
                return true;
            }

            // Loop through all the preview data models in the emitter set.
            foreach (PreviewData preview in eset.PreviewList)
            {
                // Request the viewer to delete the instance.
                if (RequestDeleteInstance(preview, eset.Guid) == false)
                {
                    Logger.Log(LogLevels.Error, "ViewerMessageHelper.RequestDeleteEmitterSet : Failed requesting viewer to delete the instance of preview data '{0}.{1}'.", eset.Name, preview.Name);
                    return false;
                }

                // Request the viewer to delete the preview binary data.
                if (RequestDeleteBinary(preview) == false)
                {
                    Logger.Log(LogLevels.Error, "ViewerMessageHelper.RequestDeleteEmitterSet : Failed requesting viewer to delete the binary data of the preview data '{0}.{1}'.", eset.Name, preview.Name);
                    return false;
                }
            }

            //// Request the viewer to delete the emitter set binary data.
            if (RequestDeleteBinary(eset) == false)
            {
                Logger.Log(LogLevels.Error, "ViewerMessageHelper.RequestDeleteEmitterSet : Failed requesting viewer to delete the binary data of the emitter set '{0}'.", eset.Name);
                return false;
            }

            return true;
        }

        /// <summary>
        /// Ask the viewer to delete the emitter set binary data.
        /// </summary>
        /// <param name="eset">The emitter set to delete.</param>
        /// <returns>True on success.</returns>
        public static bool RequestDeleteBinary(EmitterSetData eset)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false)
            {
                return true;
            }

            var message = new DeleteBinaryMessage(
                AssetTypes.EmitterSet,
                eset.Guid);

            return message.Send();
        }

        /// <summary>
        /// Ask the viewer to delete the preview binary data.
        /// </summary>
        /// <param name="preview">The preview data to delete.</param>
        /// <returns>True on success.</returns>
        public static bool RequestDeleteBinary(PreviewData preview)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false)
            {
                return true;
            }

            var message = new DeleteBinaryMessage(
                AssetTypes.Preview,
                preview.Guid);

            return message.Send();
        }

        /// <summary>
        /// Request the viewer to create an instance with the given data.
        /// </summary>
        /// <param name="preview">The preview data.</param>
        /// <param name="resourceGuid">The Guid of resource data.</param>
        /// <returns>True on success.</returns>
        public static bool RequestCreateInstance(PreviewData preview, Guid resourceGuid)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false)
            {
                return true;
            }

            var message = new CreateInstanceMessage(
                AssetTypes.Preview,
                resourceGuid,
                preview.Guid);

            return message.Send();
        }

        /// <summary>
        /// Request the viewer to delete the instance that was created with the given data.
        /// </summary>
        /// <param name="preview">The preview data.</param>
        /// <param name="resourceGuid">The Guid of resource data.</param>
        /// <returns>True on success.</returns>
        public static bool RequestDeleteInstance(PreviewData preview, Guid resourceGuid)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false)
            {
                return true;
            }

            var message = new DeleteInstanceMessage(
                AssetTypes.Preview,
                resourceGuid,
                preview.Guid);

            return message.Send();
        }

        /// <summary>
        /// Request the viewer to create an instance with the given data.
        /// </summary>
        /// <param name="eset">The emitter set data.</param>
        /// <returns>True on success.</returns>
        public static bool RequestCreateInstance(EmitterSetData eset)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false)
            {
                return true;
            }

            // This method is supposed to be removed once the viewer implemented the correct
            // flow, which instead of emitter set, it's preview data should be sent to create
            // the instance.
            var message = new CreateInstanceMessage(
                AssetTypes.EmitterSet,
                eset.Guid,
                Guid.Empty);

            return message.Send();
        }

        /// <summary>
        /// Request the viewer to delete the instance that was created with the given data.
        /// </summary>
        /// <param name="eset">The emitter set data.</param>
        /// <returns>True on success.</returns>
        public static bool RequestDeleteInstance(EmitterSetData eset)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false)
            {
                return true;
            }

            // This method is supposed to be removed once the viewer implemented the correct
            // flow, which instead of the instance of the emitter set, the instance of it's
            // preview data should be deleted.
            var message = new DeleteInstanceMessage(
                AssetTypes.EmitterSet,
                eset.Guid,
                Guid.Empty);

            return message.Send();
        }

        /// <summary>
        /// Send all the given models to the viewer. (both binary data and instances)
        /// </summary>
        /// <param name="models">The model to send.</param>
        /// <returns>True on success.</returns>
        public static bool SendModels(IEnumerable<ModelData> models)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false ||
                DisableBinaryConvertNestCount > 0)
            {
                return true;
            }

            bool allSucceeded = true;
            foreach (ModelData model in models)
            {
                // モデル未登録のバイナリは送らない
                if (model.ModelBinaryId == -1)
                {
                    continue;
                }

                if (SendModel(model) == false)
                {
                    allSucceeded = false;
                }
            }

            return allSucceeded;
        }

        /// <summary>
        /// Send the given model to the viewer. (both binary data and instances)
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns>True on success.</returns>
        public static bool SendModel(ModelData model)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false ||
                DisableBinaryConvertNestCount > 0)
            {
                return true;
            }

            // First delete the model.
            RequestDeleteModel(model);

            // Send the model.
            if (SendBinary(model) == false)
            {
                Logger.Log(LogLevels.Error, "ViewerMessageHelper.SendModel : Failed sending the model '{0}' to viewer.", model.Name);
                return false;
            }

            return true;
        }

        /// <summary>
        /// Request the viewer to delete the models. (both binary data and instances)
        /// </summary>
        /// <param name="models">The models.</param>
        /// <returns>True on success.</returns>
        public static bool RequestDeleteModels(IEnumerable<ModelData> models)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false)
            {
                return true;
            }

            bool allSucceeded = true;
            foreach (ModelData model in models)
            {
                if (RequestDeleteModel(model) == false)
                {
                    allSucceeded = false;
                }
            }

            return allSucceeded;
        }

        /// <summary>
        /// Request the viewer to delete the model. (both binary data and instances)
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns>True on success.</returns>
        public static bool RequestDeleteModel(ModelData model)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false)
            {
                return true;
            }

            //// Request the viewer to delete the model binary data.
            if (RequestDeleteBinary(model) == false)
            {
                Logger.Log(LogLevels.Error, "ViewerMessageHelper.RequestDeleteModel : Failed requesting viewer to delete the binary data of the model '{0}'.", model.Name);
                return false;
            }

            return true;
        }

        /// <summary>
        /// Ask the viewer to delete the preview binary data.
        /// </summary>
        /// <param name="model">The preview data to delete.</param>
        /// <returns>True on success.</returns>
        public static bool RequestDeleteBinary(ModelData model)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false)
            {
                return true;
            }

            var message = new DeleteBinaryMessage(
                AssetTypes.ViewerModel,
                model.Guid);

            return message.Send();
        }

        /// <summary>
        /// Request the viewer to create an instance with the given data.
        /// </summary>
        /// <param name="model">The model data.</param>
        /// <returns>True on success.</returns>
        public static bool RequestCreateInstance(ModelData model)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false)
            {
                return true;
            }

            var message = new CreateInstanceMessage(
                AssetTypes.ViewerModel,
                model.Guid,
                Guid.Empty);

            return message.Send();
        }

        /// <summary>
        /// Send the given viewer to the viewer. (both binary data and instances)
        /// </summary>
        /// <param name="viewer">The viewer.</param>
        /// <returns>True on success.</returns>
        public static bool SendViewer(ViewerData viewer)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false ||
                DisableBinaryConvertNestCount > 0)
            {
                return true;
            }

            // Send the viewer data
            if (SendBinary(viewer) == false)
            {
                Logger.Log(LogLevels.Error, "ViewerMessageHelper.SendViewer : Failed sending the viewer to viewer.");
                return false;
            }

            return true;
        }

        /// <summary>
        /// Request the viewer to delete the instance that was created with the given data.
        /// </summary>
        /// <param name="model">The model data.</param>
        /// <returns>True on success.</returns>
        public static bool RequestDeleteInstance(ModelData model)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false)
            {
                return true;
            }

            // This method is supposed to be removed once the viewer implemented the correct
            // flow, which instead of the instance of the emitter set, the instance of it's
            // preview data should be deleted.
            var message = new DeleteInstanceMessage(
                AssetTypes.ViewerModel,
                model.Guid,
                Guid.Empty);

            return message.Send();
        }

        /// <summary>
        /// Get the asset type of the data model for DataModifyMessage.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The asset type.</returns>
        public static bool IsValidDataModelForDataModifyMessage(DataModelBase dataModel)
        {
            return GetAssetTypeForDataModifyMessage(dataModel) != AssetTypes.Unknown;
        }

        /// <summary>
        /// Send only the modified data (instead of the whole binary data) to the viewer.
        /// </summary>
        /// <param name="rootBinaryStruct">The root binary structure of the field to send.</param>
        /// <param name="field">The modified binary field.</param>
        /// <returns>True on success.</returns>
        public static bool SendModifiedData(
            BinaryStructInstance rootBinaryStruct,
            BinaryFieldInstance field)
        {
            if (MessageBase.ShouldSendMessage == false ||
                ViewerController.Instance.IsConnected == false ||
                DisableBinaryConvertNestCount > 0)
            {
                return true;
            }

            AssetTypes assetType = AssetTypes.Unknown;

            // Find the valid data model for sending the DataModifyMessage.
            int offset = field.LocalOffset;
            DataModelBase dataModel = null;
            IBinaryElementInstance parent = field.Parent;
            while (parent != null)
            {
                // Does this binary structure contains a valid data model?
                var binaryStruct = parent as BinaryStructInstance;
                if (binaryStruct == null)
                {
                    if (parent is BinaryFieldInstance)
                    {
                        offset += parent.LocalOffset;
                    }

                    // Keep searching through the parents.
                    parent = parent.Parent;
                    continue;
                }

                assetType = GetAssetTypeForDataModifyMessage(binaryStruct.DataModel);
                if (assetType != AssetTypes.Unknown)
                {
                    // We have found the parent binary structure that
                    // start calculating the offset from.
                    // This binary structure's header is also included,
                    // so we need to manually remove the header size.
                    if (binaryStruct.Definition.HasBinaryHeader == true)
                    {
                        offset -= (int)BinaryStructHeader.Size;
                    }

                    dataModel = binaryStruct.DataModel;
                    break;
                }

                offset += binaryStruct.LocalOffset;

                // Keep searching through the parents.
                parent = parent.Parent;
            }

            if (assetType == AssetTypes.Unknown)
            {
                // The data model is not yet supported.
                return false;
            }

            string assetName = GetAssetNameForDataModifyMessage(dataModel);

            using (var stream = new MemoryStream())
            {
                if (WriteBinaryDataSession.WriteBinaryField(field, stream) == false)
                {
                    return false;
                }

                // Create the message and queue it for sending to the viewer.
                var message = new DataModifyMessage(
                    assetType,
                    rootBinaryStruct.DataModel.Guid,
                    assetName,
                    stream.GetBuffer(),
                    field.Size,
                    offset,
                    (short)(field.Definition.SendModificationType == SendModificationTypes.ModifiedWithReset ? 1 : 0));

                if (message.Send() == false)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// Send emitter set binary data to the viewer.
        /// </summary>
        /// <param name="eset">The emitter set to send.</param>
        /// <returns>True on success.</returns>
        private static bool SendBinary(EmitterSetData eset)
        {
            lock (LockObj)
            {
                if (MessageBase.ShouldSendMessage == false ||
                    ViewerController.Instance.IsConnected == false)
                {
                    return true;
                }

                return SendBinaryInternal(eset, eset.Name, AssetTypes.EmitterSet, true);
            }
        }

        /// <summary>
        /// Send preview binary data to the viewer.
        /// </summary>
        /// <param name="preview">The preview data to send.</param>
        /// <returns>True on success.</returns>
        private static bool SendBinary(PreviewData preview)
        {
            lock (LockObj)
            {
                if (MessageBase.ShouldSendMessage == false ||
                    ViewerController.Instance.IsConnected == false)
                {
                    return true;
                }

                return SendBinaryInternal(preview, preview.Name, AssetTypes.Preview, true);
            }
        }

        /// <summary>
        /// Send viewer binary data to the viewer.
        /// </summary>
        /// <param name="viewer">The viewer to send.</param>
        /// <returns>True on success.</returns>
        private static bool SendBinary(ViewerData viewer)
        {
            lock (LockObj)
            {
                if (MessageBase.ShouldSendMessage == false ||
                    ViewerController.Instance.IsConnected == false)
                {
                    return true;
                }

                return SendBinaryInternal(viewer, "viewer", AssetTypes.ViewerData, true);
            }
        }

        /// <summary>
        /// Send model binary data to the viewer.
        /// </summary>
        /// <param name="model">The model to send.</param>
        /// <returns>True on success.</returns>
        private static bool SendBinary(ModelData model)
        {
            lock (LockObj)
            {
                if (MessageBase.ShouldSendMessage == false ||
                    ViewerController.Instance.IsConnected == false)
                {
                    return true;
                }

                if (SendBinaryInternal(model, model.Name, AssetTypes.ViewerModel, true) == false)
                {
                    return false;
                }

                ViewerController.Instance.SendVisibility(model);

                return true;
            }
        }

        /// <summary>
        /// Get the asset type of the data model for DataModifyMessage.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The asset type.</returns>
        private static AssetTypes GetAssetTypeForDataModifyMessage(DataModelBase dataModel)
        {
            AssetTypes assetType = MessageUtility.GetDataModelAssetType(dataModel);
            if (assetType == AssetTypes.EmitterSet ||
                assetType == AssetTypes.Emitter ||
                assetType == AssetTypes.Preview ||
                assetType == AssetTypes.ViewerData ||
                assetType == AssetTypes.ViewerModel)
            {
                return assetType;
            }
            else
            {
                return AssetTypes.Unknown;
            }
        }

        /// <summary>
        /// Get the asset name of the data model for DataModifyMessage.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The asset name.</returns>
        private static string GetAssetNameForDataModifyMessage(DataModelBase dataModel)
        {
            if (dataModel is EmitterSetData)
            {
                return ((EmitterSetData)dataModel).Name;
            }
            else if (dataModel is EmitterData)
            {
                return ((EmitterData)dataModel).Name;
            }
            else if (dataModel is PreviewData)
            {
                return ((PreviewData)dataModel).Name;
            }
            else if (dataModel is ModelData)
            {
                return ((ModelData)dataModel).Name;
            }
            else if (dataModel is ViewerData)
            {
                // TODO:ビューアーデータに名前は必要？
                return "Viewer";
            }

            return string.Empty;
        }

        /// <summary>
        /// Internal method to help sending data model binary data to the viewer.
        /// </summary>
        /// <param name="dataModel">The data model to send.</param>
        /// <param name="dataModelName">The name of the data.</param>
        /// <param name="assetType">The data model asset type.</param>
        /// <param name="forceRefreshBinary">True to force refreshing the binary data.</param>
        /// <returns>True on success.</returns>
        private static bool SendBinaryInternal(
            DataModelBase dataModel,
            string dataModelName,
            AssetTypes assetType,
            bool forceRefreshBinary)
        {
            string dataModelTypeName = dataModel.GetType().Name;

            // The stream will be closed by the session when all the asynchronous tasks are
            // finished, thus do not need to enclose the stream in a using {...} block.
            var stream = new MemoryStream();

            // Create the message and send it to the viewer.
            var message = new SendBinaryMessage(
                assetType,
                dataModel.Guid,
                stream);

            // This will be executed when the session is finished.
            // Because the session is executed asynchronously, we
            // cannot assume the data is ready to be sent when the
            // session is finished.
            var asyncSessionCompleteHandler = new Action<bool, bool>((result, shaderCompileFailed) =>
            {
                if (result == true)
                {
                    message.SetReady();
                }
                else
                {
                    message.CancelMessage();
                    if (shaderCompileFailed)
                    {
                        Logger.Log(
                            LogLevels.Error,
                            "ViewerMessageHelper.SendBinaryInternal : Failed writing binary data for {0} '{1}'",
                            dataModelTypeName,
                            dataModelName);
                    }
                    else
                    {
                        Logger.Log(
                            LogLevels.Warning,
                            "ViewerMessageHelper.SendBinaryInternal : Suppressed multiple reload for {0} '{1}'",
                            dataModelTypeName,
                            dataModelName);
                    }
                }
            });

            // Create an asynchronous session to write the binary data.
            var session = new WriteBinaryDataSession(
                stream,
                assetType,
                dataModelName,
                asyncSessionCompleteHandler);

            // Convert data model to binary.
            if (BinaryDataHelper.WriteBinary(dataModel, session, forceRefreshBinary) == false)
            {
                session.CancelSession();
                Logger.Log(
                    LogLevels.Error,
                    "ViewerMessageHelper.SendBinaryInternal : Failed converting {0} '{1}' to binary data.",
                    dataModelTypeName,
                    dataModelName);
                return false;
            }

            if (session.EndSession() == false)
            {
                Logger.Log(
                    LogLevels.Error,
                    "ViewerMessageHelper.SendBinaryInternal : Failed converting {0} '{1}' to binary data.",
                    dataModelTypeName,
                    dataModelName);
                return false;
            }

            // The message will be push to the SendMessageQueue and wait for
            // the binary data finished writing, then wait to be sent in the
            // order the messages were pushed to the queue.
            return message.Send();
        }
    }

    /// <summary>
    /// バイナリコンバート停止区間をあらわすDisposableクラスです。
    /// バイナリコンバート停止区間ではバイナリコンバートが行われず、ビューアにメッセージが送られなくなります。
    /// </summary>
    public class DisableBinaryConvertSection : IDisposable
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public DisableBinaryConvertSection()
        {
            ViewerMessageHelper.BeginDisableBinaryConvertSection();
        }

        /// <summary>
        /// ファイナライザです。
        /// </summary>
        ~DisableBinaryConvertSection()
        {
            this.InnerDispose(false);
        }

        /// <summary>
        /// Dispose処理を行います。
        /// </summary>
        public virtual void Dispose()
        {
            this.InnerDispose(true);

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// Dispose処理の本体です。
        /// </summary>
        /// <param name="disposing">Disposeメソッドからの呼び出しか</param>
        private void InnerDispose(bool disposing)
        {
            System.Diagnostics.Debug.Assert(disposing == true, "必ず using ブロックで使用する");

            ViewerMessageHelper.EndDisableBinaryConvertSection();
        }
    }

    /// <summary>
    /// メッセージドロップ区間をあらわすDisposableクラスです。
    /// メッセージドロップ区間では送信メッセージはSendMessageQueueに積まれて
    /// バイナリコンバートする所まで行われますが、バイナリ化されたメッセージを
    /// MessageManagerに追加する際にフィルタリングされてビューアにメッセージが送られなくなります。
    /// </summary>
    public class MessageDropSection : IDisposable
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MessageDropSection()
        {
            ViewerMessageHelper.BeginMessageDropSection();
        }

        /// <summary>
        /// ファイナライザです。
        /// </summary>
        ~MessageDropSection()
        {
            this.InnerDispose(false);
        }

        /// <summary>
        /// Dispose処理を行います。
        /// </summary>
        public virtual void Dispose()
        {
            this.InnerDispose(true);

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// Dispose処理の本体です。
        /// </summary>
        /// <param name="disposing">Disposeメソッドからの呼び出しか</param>
        private void InnerDispose(bool disposing)
        {
            System.Diagnostics.Debug.Assert(disposing == true, "必ず using ブロックで使用する");

            ViewerMessageHelper.EndMessageDropSection();
        }
    }
}
