﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;

using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelLogic.Utilities
{
    /// <summary>
    /// A helper class for manipulating data model binary data.
    /// </summary>
    public static class BinaryDataHelper
    {
        /// <summary>
        /// The static constructor.
        /// </summary>
        static BinaryDataHelper()
        {
            AlwaysRefreshBinaryData = false;
        }

        /// <summary>
        /// Get or set the flag indicating if the helper should always refresh binary data.
        /// </summary>
        public static bool AlwaysRefreshBinaryData { get; set; }

        /// <summary>
        /// Write data model binary data to the given stream.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <param name="session">The WriteBinaryDataSession that is being processed now.</param>
        /// <param name="refreshBinaryData">True to force convert all the binary data before writing.</param>
        /// <returns>True on success.</returns>
        public static bool WriteBinary(
            DataModelBase dataModel,
            WriteBinaryDataSession session,
            bool refreshBinaryData = false)
        {
            // Find the binary data for the data model.
            var binaryData = BinaryConversionInfoManager.FindBinaryData(dataModel);
            if (binaryData == null)
            {
                Logger.Log(LogLevels.Error, "ViewerMessageHelper.SendBinary : The binary data for the data model does not exist.");
                return false;
            }

            if (AlwaysRefreshBinaryData == true ||
                refreshBinaryData == true)
            {
                binaryData.ConvertBinary();
            }

            // Write the binary data to a memory stream.
            try
            {
                if (binaryData.WriteBinaryData(session) == false)
                {
                    Logger.Log(LogLevels.Error, "ViewerMessageHelper.SendBinary : Failed writing data model binary data to stream.");
                    return false;
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "ViewerMessageHelper.SendBinary : Exception occured while writing data model binary data to stream. Message : {0}", ex.Message);
                return false;
            }

            return true;
        }
    }

    /// <summary>
    /// Class to temporary force the BinaryDataHelper to always refresh binary data.
    /// </summary>
    public class ForceRefreshBinary : IDisposable
    {
        /// <summary>The original flag value.</summary>
        private readonly bool originalFlag;

        /// <summary>
        /// Constructor.
        /// </summary>
        public ForceRefreshBinary()
        {
            this.originalFlag = BinaryDataHelper.AlwaysRefreshBinaryData;
            BinaryDataHelper.AlwaysRefreshBinaryData = true;
        }

        /// <summary>
        /// Dispose the instance and restore the original value of the flag.
        /// </summary>
        public void Dispose()
        {
            BinaryDataHelper.AlwaysRefreshBinaryData = this.originalFlag;
        }
    }
}
