﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.DataModelOperation;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Events;

namespace EffectMaker.DataModelLogic.DataModelProxies
{
    /// <summary>
    /// Data model proxy for emitter set data.
    /// </summary>
    public class EmitterSetDataProxy : DataModelProxy, IEmitterBuilder
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="dataModel">The data model for the proxy.</param>
        public EmitterSetDataProxy(DataModelBase dataModel) :
            base(dataModel)
        {
            DataModelProxy.PropertyModified += this.OnDataModelPropertyModified;
        }

        /// <summary>
        /// Disposes the instance.
        /// </summary>
        public override void Dispose()
        {
            DataModelProxy.PropertyModified -= this.OnDataModelPropertyModified;
            base.Dispose();
        }

        /// <summary>
        /// Add the given data model to the children data models.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        public override void AddChildDataModel(DataModelBase dataModel)
        {
            if (dataModel is EmitterData)
            {
                this.AddEmitter((EmitterData)dataModel);
            }
            else if (dataModel is PreviewData)
            {
                this.AddPreview((PreviewData)dataModel);
            }
            else
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Add the given data model to the children data models.
        /// </summary>
        /// <param name="index">The index of the data model.</param>
        /// <param name="dataModel">The data model.</param>
        public override void InsertChildDataModel(int index, DataModelBase dataModel)
        {
            if (dataModel is EmitterData)
            {
                this.InsertEmitter(index, (EmitterData)dataModel);
            }
            else if (dataModel is PreviewData)
            {
                this.InsertPreview(index, (PreviewData)dataModel);
            }
            else
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Remove the specified data model from the children data models.
        /// </summary>
        /// <param name="dataModel">The dat model to be removed.</param>
        public override void RemoveChildDataModel(DataModelBase dataModel)
        {
            if (dataModel is EmitterData)
            {
                this.RemoveEmitter((EmitterData)dataModel);
            }
            else if (dataModel is PreviewData)
            {
                this.RemovePreview((PreviewData)dataModel);
            }
            else
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Move the specified data model to the specified index.
        /// </summary>
        /// <param name="dataModel">The data model to be moved.</param>
        /// <param name="distance">
        /// The distance to move the child.
        /// E.g. 2 means move down 2 items, -3 means move up 3 items.
        /// </param>
        public override void MoveChildDataModel(DataModelBase dataModel, int distance)
        {
            if (dataModel is EmitterData)
            {
                this.MoveEmitter((EmitterData)dataModel, distance);
            }
            else if (dataModel is PreviewData)
            {
                this.MovePreview((PreviewData)dataModel, distance);
            }
            else
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Create a new emitter as a child of the emitter set.
        /// </summary>
        /// <param name="name">The name of the emitter.</param>
        /// <returns>The created emitter.</returns>
        public EmitterData CreateEmitter(string name)
        {
            var emitterSetData = this.DataModel as EmitterSetData;
            if (emitterSetData == null)
            {
                return null;
            }

            // Create the emitter.
            EmitterData emitterData = DataModelCreator.CreateEmitterDataModel();
            emitterData.Name = name;

            return emitterData;
        }

        /// <summary>
        /// Add the given emitter data to the emitter set.
        /// </summary>
        /// <param name="emitterData">The emitter data.</param>
        public void AddEmitter(EmitterData emitterData)
        {
            var emitterSetData = this.DataModel as EmitterSetData;
            if (emitterSetData == null)
            {
                return;
            }

            // Check if the emitter has already been added.
            if (emitterSetData.EmitterList.IndexOf(emitterData) >= 0)
            {
                return;
            }

            // Add the emitter to the emitter set.
            emitterSetData.EmitterList.Add(emitterData);
            emitterSetData.AddChild(emitterData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterList");
        }

        /// <summary>
        /// Add the given emitter data to the emitter set.
        /// </summary>
        /// <param name="index">The index of the emitter.</param>
        /// <param name="emitterData">The emitter data.</param>
        public void InsertEmitter(int index, EmitterData emitterData)
        {
            var emitterSetData = this.DataModel as EmitterSetData;
            if (emitterSetData == null)
            {
                return;
            }

            // Check if the emitter has already been added.
            if (emitterSetData.EmitterList.IndexOf(emitterData) >= 0)
            {
                return;
            }

            // Add the emitter to the emitter set.
            emitterSetData.EmitterList.Insert(index, emitterData);
            emitterSetData.InsertChild(index, emitterData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterList");
        }

        /// <summary>
        /// Remove the specified emitter from the emitter set.
        /// </summary>
        /// <param name="emitterData">The emitter to be removed.</param>
        public void RemoveEmitter(EmitterData emitterData)
        {
            var emitterSetData = this.DataModel as EmitterSetData;
            if (emitterSetData == null)
            {
                return;
            }

            // Remove the emitter from the emitter set.
            emitterSetData.EmitterList.Remove(emitterData);
            emitterSetData.RemoveChild(emitterData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterList");
        }

        /// <summary>
        /// Move the specified child to the specified index.
        /// </summary>
        /// <param name="child">The child to be moved.</param>
        /// <param name="distance">
        /// The distance to move the child.
        /// E.g. 2 means move down 2 items, -3 means move up 3 items.
        /// </param>
        public void MoveEmitter(EmitterData child, int distance)
        {
            var emitterSetData = this.DataModel as EmitterSetData;
            if (emitterSetData == null)
            {
                return;
            }

            // Find the original index.
            int originalIndex = emitterSetData.EmitterList.IndexOf(child);
            int newIndex = originalIndex + distance;

            // Remove the emitter from the emitter set.
            emitterSetData.EmitterList.Remove(child);

            // Insert the emitter back to the specified index.
            emitterSetData.EmitterList.Insert(newIndex, child);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterList");
        }

        /// <summary>
        /// Create a new preview data as a child of the emitter set.
        /// </summary>
        /// <param name="name">The name of the preview data.</param>
        /// <returns>The created preview data.</returns>
        public PreviewData CreatePreview(string name)
        {
            var emitterSetData = this.DataModel as EmitterSetData;
            if (emitterSetData == null)
            {
                return null;
            }

            // Create the preview data.
            PreviewData previewData = DataModelCreator.CreatePreviewDataModel();
            previewData.Name = name;

            return previewData;
        }

        /// <summary>
        /// Add the given preview data to the emitter set.
        /// </summary>
        /// <param name="previewData">The preview data.</param>
        public void AddPreview(PreviewData previewData)
        {
            var emitterSetData = this.DataModel as EmitterSetData;
            if (emitterSetData == null)
            {
                return;
            }

            // Check if the preview data has already been added.
            if (emitterSetData.PreviewList.IndexOf(previewData) >= 0)
            {
                return;
            }

            // Add the emitter to the emitter set.
            emitterSetData.PreviewList.Add(previewData);
            emitterSetData.AddChild(previewData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("PreviewList");
        }

        /// <summary>
        /// Add the given preview data to the emitter set.
        /// </summary>
        /// <param name="index">The index of the preview data.</param>
        /// <param name="previewData">The preview data.</param>
        public void InsertPreview(int index, PreviewData previewData)
        {
            var emitterSetData = this.DataModel as EmitterSetData;
            if (emitterSetData == null)
            {
                return;
            }

            // Check if the preview data has already been added.
            if (emitterSetData.PreviewList.IndexOf(previewData) >= 0)
            {
                return;
            }

            // Add the preview data to the emitter set.
            emitterSetData.PreviewList.Insert(index, previewData);
            emitterSetData.InsertChild(index, previewData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("PreviewList");
        }

        /// <summary>
        /// Remove the specified preview data from the emitter set.
        /// </summary>
        /// <param name="previewData">The preview data to be removed.</param>
        public void RemovePreview(PreviewData previewData)
        {
            var emitterSetData = this.DataModel as EmitterSetData;
            if (emitterSetData == null)
            {
                return;
            }

            // Remove the preview data from the emitter set.
            emitterSetData.PreviewList.Remove(previewData);
            emitterSetData.RemoveChild(previewData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("PreviewList");
        }

        /// <summary>
        /// Move the specified child to the specified index.
        /// </summary>
        /// <param name="child">The child to be moved.</param>
        /// <param name="distance">
        /// The distance to move the child.
        /// E.g. 2 means move down 2 items, -3 means move up 3 items.
        /// </param>
        public void MovePreview(PreviewData child, int distance)
        {
            var emitterSetData = this.DataModel as EmitterSetData;
            if (emitterSetData == null)
            {
                return;
            }

            // Find the original index.
            int originalIndex = emitterSetData.PreviewList.IndexOf(child);
            int newIndex = originalIndex + distance;

            // Remove the child from the emitter set.
            emitterSetData.PreviewList.Remove(child);

            // Insert the child back to the specified index.
            emitterSetData.PreviewList.Insert(newIndex, child);

            // Fire event.
            this.TriggerPropertyModifiedEvent("PreviewList");
        }

        /// <summary>
        /// Handle PropertyModified event from data model proxy.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDataModelPropertyModified(
            object sender,
            DataModelPropertyModifiedEventArgs e)
        {
            if (e.PropertyName != "EmitterList")
            {
                return;
            }

            var eset = this.DataModel as EmitterSetData;
            if (e.DataModel == eset ||
                eset.AllChildEmitters.Contains(e.DataModel) == true)
            {
                this.TriggerPropertyModifiedEvent("AllChildEmitters");
                this.TriggerPropertyModifiedEvent("AllActiveChildEmitters");
            }
        }
    }
}
