﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;

using EffectMaker.BusinessLogic.BinaryHeaders;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.DataModelLogic.Utilities;

using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelLogic.BinaryData
{
    /// <summary>
    /// Class of a binary structure instance created with the binary structure definition.
    /// </summary>
    public class BinaryStructInstance : IBinaryElementInstance
    {
        /// <summary>The binary data structure definition.</summary>
        private BinaryStructDefinition definition;

        /// <summary>The data model the binary data converts from.</summary>
        private DataModelBase dataModel = null;

        /// <summary>The list of binary data field instances.</summary>
        private List<BinaryFieldInstance> fields = new List<BinaryFieldInstance>();

        /// <summary>The list of child binary structures.</summary>
        private List<BinaryStructInstance> childBinaryStructures = new List<BinaryStructInstance>();

        /// <summary>The binary size.</summary>
        private int binarySize = 0;

        /// <summary>Flag indicating if is currently converting binary data.</summary>
        private bool isConvertingBinary = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="parent">The parent binary element.</param>
        /// <param name="definition">The binary data structure definition.</param>
        /// <param name="dataModel">The data model the binary data converts from.</param>
        public BinaryStructInstance(
            IBinaryElementInstance parent,
            BinaryStructDefinition definition,
            DataModelBase dataModel)
        {
            this.Parent = parent;
            this.definition = definition;
            this.dataModel = dataModel;

            if (this.Parent != null)
            {
                this.Parent.ReportChildElementCreated(this);
            }
            else
            {
                this.ReportChildElementCreated(this);
                BinaryConversionInfoManager.RegisterBinaryData(this);
            }

            this.Offset = 0;

            // Create the binary data field instances.
            this.fields.AddRange(this.definition.CreateFieldInstances(this));
        }

        /// <summary>
        /// Get the binary structure definition of the instance.
        /// </summary>
        public BinaryStructDefinition Definition
        {
            get { return this.definition; }
        }

        /// <summary>
        /// Get the parent binary element.
        /// </summary>
        public IBinaryElementInstance Parent { get; private set; }

        /// <summary>
        /// Get or set the data model the binary data converts from.
        /// </summary>
        public DataModelBase DataModel
        {
            get { return this.dataModel; }
        }

        /// <summary>
        /// Get the tag of the binary element.
        /// </summary>
        public string Tag
        {
            get
            {
                if (this.definition == null || this.definition.Tag == null)
                {
                    return string.Empty;
                }

                return this.definition.Tag;
            }
        }

        /// <summary>
        /// Get the local offset of the binary data from the beginning of its parent.
        /// (including binary structure headers if any)
        /// </summary>
        public int LocalOffset { get; set; }

        /// <summary>
        /// Get the offset of the binary data in the root binary structure.
        /// </summary>
        public int Offset { get; set; }

        /// <summary>
        /// Get the size of the binary data.
        /// </summary>
        public int Size
        {
            get { return this.binarySize; }
        }

        /// <summary>
        /// Get or set the size of the binary data including the binary headers if any.
        /// </summary>
        public virtual int SizeIncludeHeader { get; protected set; }

        /// <summary>
        /// Dispose.
        /// </summary>
        public virtual void Dispose()
        {
            // Dispose the fields.
            foreach (BinaryFieldInstance field in this.fields)
            {
                field.Dispose();
            }

            if (this.Parent != null)
            {
                this.Parent.ReportChildElementRemoved(this);
            }
            else
            {
                this.ReportChildElementRemoved(this);
                BinaryConversionInfoManager.UnregisterBinaryData(this);
            }

            if (this.childBinaryStructures.Count > 0)
            {
                Logger.Log(LogLevels.Debug, "BinaryStructInstance.Dispose : Some child binary structures are not being disposed.");
            }
        }

        /// <summary>
        /// Find the root binary element of this instance.
        /// </summary>
        /// <returns>The root binary element.</returns>
        public IBinaryElementInstance FindRootElement()
        {
            if (this.Parent == null)
            {
                return this;
            }
            else
            {
                return this.Parent.FindRootElement();
            }
        }

        /// <summary>
        /// Find all the children of this instance that their definition's tag matches
        /// the specified string.
        /// </summary>
        /// <param name="tag">The tag.</param>
        /// <returns>The binary element instances that has the tag.</returns>
        public IEnumerable<IBinaryElementInstance> FindChildrenWithTag(string tag)
        {
            foreach (BinaryFieldInstance instance in this.fields)
            {
                if (string.IsNullOrEmpty(instance.Tag) == false &&
                    instance.Tag == tag)
                {
                    yield return instance;
                }

                foreach (IBinaryElementInstance child in instance.FindChildrenWithTag(tag))
                {
                    yield return child;
                }
            }
        }

        /// <summary>
        /// Find owner binary element instance of the source data model.
        /// </summary>
        /// <param name="dataModel">The source data model.</param>
        /// <returns>The binary element instance that holds the source data model.</returns>
        public IBinaryElementInstance FindSourceDataModelOwner(DataModelBase dataModel)
        {
            // Find the binary structure in this binary structure's own child structure list.
            var binaryStruct =
                this.childBinaryStructures.Find(it => it.DataModel.Guid == dataModel.Guid);

            // If the binary structure is not found in the list, ask the parent to find it.
            if (this.Parent != null && binaryStruct == null)
            {
                return this.Parent.FindSourceDataModelOwner(dataModel);
            }
            else
            {
                return binaryStruct;
            }
        }

        /// <summary>
        /// Called by children when a child element has been created.
        /// </summary>
        /// <param name="reportingElement">The child element that is created.</param>
        public void ReportChildElementCreated(IBinaryElementInstance reportingElement)
        {
            var childBinaryStructure = reportingElement as BinaryStructInstance;
            if (childBinaryStructure == null)
            {
                return;
            }

            this.childBinaryStructures.Add(childBinaryStructure);
        }

        /// <summary>
        /// Called by children when a child element has been removed.
        /// </summary>
        /// <param name="reportingElement">The child element that is removed.</param>
        public void ReportChildElementRemoved(IBinaryElementInstance reportingElement)
        {
            var childBinaryStructure = reportingElement as BinaryStructInstance;
            if (childBinaryStructure == null)
            {
                return;
            }

            this.childBinaryStructures.Remove(childBinaryStructure);
        }

        /// <summary>
        /// Called by a child element to report that it's size has changed
        /// and the offset needs to be recalculated.
        /// </summary>
        /// <param name="reportingElement">The reporting child element.</param>
        public virtual void ReportSizeChanged(IBinaryElementInstance reportingElement)
        {
            if (this.isConvertingBinary == true)
            {
                return;
            }

            int headerSize = 0;
            if (this.Definition.HasBinaryHeader == true)
            {
                headerSize = (int)BinaryStructHeader.Size;
            }

            // Update the binary offset of the array elements.
            int offset = 0;
            int offsetIncludeHeader = headerSize;
            foreach (BinaryFieldInstance field in this.fields)
            {
                field.LocalOffset = offsetIncludeHeader;
                field.Offset = offset + this.Offset;

                // Increment.
                offsetIncludeHeader += field.SizeIncludeHeader;
                offset += field.Size;
            }

            this.binarySize = offset;
            this.SizeIncludeHeader = offsetIncludeHeader;

            // Report to the parent.
            if (this.Parent != null)
            {
                this.Parent.ReportSizeChanged(this);
            }
        }

        /// <summary>
        /// Convert binary data from the data model.
        /// </summary>
        /// <returns>True on success.</returns>
        public virtual bool ConvertBinary()
        {
            this.isConvertingBinary = true;

            try
            {
                int originalBinarySize = this.binarySize;

                int headerSize = 0;
                if (this.Definition.HasBinaryHeader == true)
                {
                    headerSize = (int)BinaryStructHeader.Size;
                }

                // Convert all the fields.
                int offset = 0;
                int offsetIncludeHeader = headerSize;
                foreach (BinaryFieldInstance field in this.fields)
                {
                    // Setup the binary data offset of the field.
                    field.LocalOffset = offsetIncludeHeader;
                    field.Offset = offset + this.Offset;

                    // Convert the field.
                    if (field.ConvertBinary() == false)
                    {
                        return false;
                    }

                    // Increment.
                    offsetIncludeHeader += field.SizeIncludeHeader;
                    offset += field.Size;
                }

                // Update the binary size.
                this.binarySize = offset;
                this.SizeIncludeHeader = offsetIncludeHeader;

                if (this.Parent != null && offset != originalBinarySize)
                {
                    // Report size change.
                    this.Parent.ReportSizeChanged(this);
                }

                return true;
            }
            finally
            {
                this.isConvertingBinary = false;
            }
        }

        /// <summary>
        /// Write the binary data for the given session.
        /// </summary>
        /// <param name="session">The WriteBinaryDataSession that is being processed now.</param>
        /// <returns>True on success.</returns>
        public bool WriteBinaryData(WriteBinaryDataSession session)
        {
            using (new WriteBinaryDataBlock(session, this))
            {
                foreach (BinaryFieldInstance field in this.fields)
                {
                    if (field.WriteBinaryData(session) == false)
                    {
                        return false;
                    }
                }
            }

            return true;
        }
    }
}
