﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Runtime.CompilerServices;
using EffectMaker.DataModel.Attributes;
using EffectMaker.DataModel.DataModels;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Class for the default field converter that passes value directly to output.
    /// </summary>
    [EffectMaker.Foundation.Attributes.Default]
    public class ValueExpressionConverter : ConverterBase
    {
        /// <summary>Constant parameter name for the selected channel index.</summary>
        private const string ExpressionParamName = "Expression";

        /// <summary>The converter parameter holder.</summary>
        private static ConverterParamHolder paramHolder = new ConverterParamHolder();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static ValueExpressionConverter()
        {
            // Register parameters.
            paramHolder.RegisterParameter(
                ExpressionParamName,
                "The expression.",
                ValidateExpressionParamter);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public ValueExpressionConverter() :
            base(paramHolder)
        {
            // 式は文字列として扱いたいので、基底クラスのリストに登録します。
            this.AddStringParameterName(ExpressionParamName);
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.ValueExpressionConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.ValueExpressionConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.ValueExpressionConverterInputValue0Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            object param;
            if (this.GetParameter(ExpressionParamName, out param) != true)
            {
                return false;
            }

            string expression = param as string;
            if (string.IsNullOrEmpty(expression))
            {
                return false;
            }

            object inputValue = this.GetInputValue(0);
            object outputValue = null;

            outputValue = EvaluateExpression(expression, inputValue.ToString());
            if (outputValue == null)
            {
                return false;
            }

            if (TypeConversionUtility.TryConvert(outputValue.GetType(), inputValue.GetType(), ref outputValue))
            {
                this.SetOutputValue(outputValue);
                return true;
            }

            return false;
        }

        /// <summary>
        /// Validate channel index parameter.
        /// </summary>
        /// <param name="value">The parameter value.</param>
        /// <returns>True if the value is valid.</returns>
        private static bool ValidateExpressionParamter(object value)
        {
            string expression = value as string;
            if (expression != null)
            {
                // ダブルクォートがついたままこっちに飛んで来るようになったっぽい？ので対処
                if (expression.StartsWith("\""))
                {
                    expression = expression.Substring(1, expression.Length - 1);
                }

                if (expression.EndsWith("\""))
                {
                    expression = expression.Substring(0, expression.Length - 1);
                }

                object result = EvaluateExpression(expression, 1.0f);
                if (result != null)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 文字列を計算式として評価し、計算結果を文字列として得ます。
        /// </summary>
        /// <typeparam name="T">式に代入する値の型</typeparam>
        /// <param name="expression">式を表す文字列</param>
        /// <param name="value">式に代入する数値</param>
        /// <returns>計算結果を表す文字列(評価に失敗した場合はnull)</returns>
        private static string EvaluateExpression<T>(string expression, T value)
        {
            try
            {
                var eval = new System.Data.DataTable();
                return eval.Compute(string.Format(expression, value), string.Empty).ToString();
            }
            catch
            {
                //
            }

            return null;
        }
    }
}
