﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;

using EffectMaker.DataModel.Attributes;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Convert value by given conditions.
    /// </summary>
    public class SwitchCaseConverter : ConverterBase
    {
        /// <summary>
        /// Constant parameter name for a dictionary containing
        /// key as case value to match and value as the output
        /// value if the case matches.
        /// </summary>
        private const string CasesParamName = "Cases";

        /// <summary>
        /// Constant parameter name for the output value
        /// when no case matches the input value.
        /// </summary>
        private const string DefaultOutputParamName = "DefaultOutput";

        /// <summary>The converter parameter holder.</summary>
        private static ConverterParamHolder paramHolder = new ConverterParamHolder();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static SwitchCaseConverter()
        {
            // Register parameters.
            paramHolder.RegisterParameter(
                CasesParamName,
                "A dictionary with its containing item keys as the case value to match input value with, and the value as the output value if the case matches.",
                null);

            paramHolder.RegisterParameter(
                DefaultOutputParamName,
                "The value to output when the input value doesn't match any of the the cases.",
                null);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public SwitchCaseConverter() :
            base(paramHolder)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.TypeConversionConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.TypeConversionConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.TypeConversionConverterInputValue0Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            // Is the input value assigned at the index?
            if (this.IsInputValueAssigned(0) == false)
            {
                Logger.Log(LogLevels.Error, "SwitchCaseConverter.Convert : The input value has not been assigned yet.");
                return false;
            }

            object param;

            // Get the an array that contains the cases.
            SwitchCase[] cases = null;
            if (this.GetParameter(CasesParamName, out param) == false)
            {
                Logger.Log(LogLevels.Error, "SwitchCaseConverter.Convert : The cases parameter is mandatory.");
                return false;
            }

            cases = param as SwitchCase[];
            if (cases == null)
            {
                Logger.Log(LogLevels.Error, "SwitchCaseConverter.Convert : The cases parameter value must be an array of SwitchCase instances.");
                return false;
            }

            // Get the output value when input doesn't match the condition from the parameter.
            object defaultOutput = null;
            if (this.GetParameter(DefaultOutputParamName, out param) == false)
            {
                Logger.Log(LogLevels.Error, "SwitchCaseConverter.Convert : The default output value is mandatory.");
                return false;
            }

            defaultOutput = param;

            // The input is assigned, get it.
            object input = this.GetInputValue(0);
            if (input == null)
            {
                Logger.Log(LogLevels.Error, "SwitchCaseConverter.Convert : The input value is null and cannot be compared.");
                return false;
            }

            bool matched = false;
            foreach (SwitchCase caseData in cases)
            {
                if (input.Equals(caseData.Condition) == true)
                {
                    this.SetOutputValue(caseData.Value);
                    matched = true;
                    break;
                }
            }

            if (matched == false)
            {
                this.SetOutputValue(defaultOutput);
            }

            return true;
        }
    }

    /// <summary>
    /// Class that stores a pair of case condition and it's corresponding value.
    /// </summary>
    public class SwitchCase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="condition">The condition value to match with.</param>
        /// <param name="value">The value when the condition meets.</param>
        public SwitchCase(object condition, object value)
        {
            this.Condition = condition;
            this.Value = value;
        }

        /// <summary>
        /// Get the condition for this case.
        /// </summary>
        public object Condition { get; private set; }

        /// <summary>
        /// Get the value when the condition is met.
        /// </summary>
        public object Value { get; private set; }
    }
}
