﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// 文字列の長さをコンバートします。
    /// </summary>
    public class StringLengthConverter : ConverterBase
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public StringLengthConverter() :
            base(null)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.StringLengthConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.StringLengthConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.StringLengthConverterInputValue0Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            if (this.IsInputValueAssigned(0) == false)
            {
                Logger.Log(LogLevels.Warning, "StringLengthConverter.Convert : There is no input value assigned.");
                return false;
            }

            var str = this.GetInputValue(0) as string;
            if (str == null)
            {
                Logger.Log(LogLevels.Warning, "StringLengthConverter.Convert : The assigned input value is not a string.");
                return false;
            }

            // 文字列をバイト配列に変換
            var strByte = StringConverter.Encoding.GetBytes(str);

            // 文字列の長さを取得
            int output = strByte.Length;

            this.SetOutputValue(output);

            return true;
        }
    }
}
