﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

using EffectMaker.BusinessLogic.Manager;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.Attributes;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Converter for look for the input reserved shader user data type and
    /// output its index (setting ID).
    /// </summary>
    public class ReservedShaderIndexConverter : ConverterBase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public ReservedShaderIndexConverter() :
            base(null)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.ReservedShaderIndexConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.ReservedShaderIndexConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.ReservedShaderIndexConverterInputValue0Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            object inputValue = this.GetInputValue(0);

            var reservedShaderNodeData = inputValue as ReservedShaderNodeData;
            if (reservedShaderNodeData == null ||
                reservedShaderNodeData.PageData == null ||
                reservedShaderNodeData.PageData.ContentsData == null)
            {
                this.SetOutputValue((uint)0);
            }
            else
            {
                int id = ReservedShaderUserDataManager.FindReservedShaderID(
                    reservedShaderNodeData.PageData.ContentsData);

                if (id < 0)
                {
                    this.SetOutputValue((uint)0);
                }
                else
                {
                    this.SetOutputValue((uint)id);
                }
            }

            return true;
        }
    }
}
