﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

using EffectMaker.Foundation.Model;
using EffectMaker.Foundation.Model.Types;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Converter for converting primitive path to their unique ID.
    /// </summary>
    public class PrimitiveVertexInfoFlagsConverter : ConverterBase
    {
        /// <summary>Constant parameter name for the selected channel index.</summary>
        private const string ValidIdParamName = "ValidId";

        /// <summary>The converter parameter holder.</summary>
        private static ConverterParamHolder paramHolder = new ConverterParamHolder();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static PrimitiveVertexInfoFlagsConverter()
        {
            // Register parameters.
            paramHolder.RegisterParameter(
                ValidIdParamName,
                "The valid id.",
                ValidateValidIdParamter);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public PrimitiveVertexInfoFlagsConverter() :
            base(paramHolder)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 2; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.PrimitiveVertexInfoFlagsConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.PrimitiveIDConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.PrimitiveIDConverterInputValue0Desc;
            }
            else if (index == 1)
            {
                return Properties.Resources.PrimitiveIDConverterInputValue1Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            object param;
            int validId = -1;
            if (this.GetParameter(ValidIdParamName, out param) != true)
            {
                return false;
            }
            else
            {
                object convertedValue = param;
                TypeConversionUtility.TryConvert(param.GetType(), typeof(int), ref convertedValue);
                validId = (int)convertedValue;
            }

            string path = this.GetInputValue(0) as string;
            int id = (this.GetInputValue(1) as int?).GetValueOrDefault();

            if (string.IsNullOrEmpty(path) == true || id != validId)
            {
                this.SetOutputValue(0);
            }
            else
            {
                ModelData data;
                var result =
                    EffectMaker.BusinessLogic.Manager.PrimitiveManager.Instance.LoadModelWithData(
                    path,
                    true,
                    out data);

                if (result == LoadModelResults.Success)
                {
                    uint flags = 0;
                    flags |= data.Position.Count != 0 ? (uint)1 : 0;
                    flags |= data.Normal.Count != 0 ? (uint)2 : 0;
                    flags |= data.Tangent.Count != 0 ? (uint)4 : 0;
                    flags |= data.Color.Count != 0 ? (uint)8 : 0;
                    flags |= data.NumberOfTexCoords >= 1 ? (uint)16 : 0;
                    flags |= data.NumberOfTexCoords >= 2 ? (uint)32 : 0;
                    this.SetOutputValue(flags);
                }
                else
                {
                    this.SetOutputValue(0);
                }
            }

            return true;
        }

        /// <summary>
        /// Validate channel index parameter.
        /// </summary>
        /// <param name="value">The parameter value.</param>
        /// <returns>True if the value is valid.</returns>
        private static bool ValidateValidIdParamter(object value)
        {
            int index;
            if (value is string)
            {
                if (int.TryParse((string)value, out index) == false)
                {
                    return false;
                }
            }
            else if (value != null)
            {
                object convertedValue = value;

                bool result = TypeConversionUtility.TryConvert(
                    value.GetType(),
                    typeof(int),
                    ref convertedValue);

                if (result == false)
                {
                    return false;
                }

                index = (int)convertedValue;
            }
            else
            {
                return false;
            }

            return true;
        }
    }
}
