﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections;
using System.Linq;

using EffectMaker.DataModel.Attributes;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Class for the get the element count of the input collection.
    /// </summary>
    public class CollectionCountConverter : ConverterBase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public CollectionCountConverter() :
            base(null)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.CollectionCountConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.CollectionCountConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.CollectionCountConverterInputValue0Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            var input = this.GetInputValue(0) as IEnumerable;
            if (input == null)
            {
                return false;
            }

            if (input is ICollection)
            {
                this.SetOutputValue(((ICollection)input).Count);
            }
            else
            {
                int count = 0;
                foreach (object item in input)
                {
                    ++count;
                }

                this.SetOutputValue(count);
            }

            return true;
        }
    }
}
