﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using EffectMaker.DataModel.Attributes;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// <para>Class for the converter that simply pass data model to output.</para>
    /// OutputValueType: IEnumerable&lt;DataModelBase&gt;
    /// データモデルがnullのときは、空のIEnumerableを返します。
    /// </summary>
    public class AssignDataModelConverter : ConverterBase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public AssignDataModelConverter() :
            base(null)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.AssignDataModelConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.AssignDataModelConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.AssignDataModelConverterInputValue0Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            IEnumerable<DataModelBase> assignDataModels = null;

            // アサインされたデータモデルを取得
            var input = this.GetInputValue(0);
            if (input == null)
            {
                assignDataModels = Enumerable.Empty<DataModelBase>();
            }
            else if (input is DataModelBase)
            {
                assignDataModels = new DataModelBase[] { (DataModelBase)input };
            }
            else if (input is IEnumerable)
            {
                assignDataModels = (IEnumerable<DataModelBase>)input;
            }
            else
            {
                return false;
            }

            this.SetOutputValue(assignDataModels);

            return true;
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <param name="binaryOutput">The binary output data.</param>
        /// <returns>True on success.</returns>
        public override bool Convert(out byte[] binaryOutput)
        {
            // The binary output for this converter is always null.
            binaryOutput = null;

            return this.Convert();
        }
    }
}
